//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_CodePointIterator_h)
#define FeatStd_Platform_CodePointIterator_h

#include <FeatStd/Platform/Base.h>

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{
    /** Implements a code point iterator for standard ASCII, UCS-2 encoded TChar types */
    template<typename T, bool isUtfEncoded = false> class CodePointIterator {
    public:
        /** Constructor of CodePointIterator.
            @param string pointer to the first code point */
        CodePointIterator(const T *string) :
            mString(string)
        {
        }

        /** Advances read position to next code point. */
        inline void Advance()
        {
            if (0 != mString) {
                ++mString;
            }
        }

        /** Gets the code point at current read position.
            @return Ucs2 value of the code point at the current read position. */
        inline Ucs2Char operator*() const
        {
            return (0 != mString) ? static_cast<Ucs2Char>(*mString) : 0;
        }

    private:
        /** Read position of the current code point */
        const T *mString;
    };

    /** Implements a code point iterator for UTF-8 encoded strings */
    template<> class CodePointIterator<Char, true> {
    public:
        CodePointIterator(const Char *string) :
            mString(string)
        {
        }

        /** Advances read position to next code point.
            It is assumed that the given string points to a valid code point and is overall a
            valid UTF-8 encoded string. Invalid UTF-8 input data will yield undefined results.
            Implementation will only cover UCS-2 (16bit) character range. Characters encoded in
            4 UTF-8 bytes (21bit character codes) are not supported and may yield undefined
            results. */
        void Advance()
        {
            if (0 != mString) {
                if ((static_cast<UInt32>(*mString) & (1U << 7U)) == 0U) {
                    ++mString;
                }
                else if ((static_cast<Ucs2Char>(*mString) & 224U) == 192U) {
                    mString += 2;
                }
                else {
                    mString += 3;
                }
            }
        }

        /** Gets the code point at current read position.
            @return Ucs2 value of the code point at the current read position. */
        Ucs2Char operator*() const
        {
            if (0 == mString) {
                return 0;
            }

            if ((static_cast<UInt32>(*mString) & (1U << 7U)) == 0U) {
                return static_cast<Ucs2Char>(mString[0]);
            }
            else if ((static_cast<Ucs2Char>(*mString) & 224U) == 192U) {
                return (((static_cast<Ucs2Char>(mString[0]) & 31U) << 6U) |
                    (static_cast<Ucs2Char>(mString[1]) & 63U));
            }
            else {
                return (((static_cast<Ucs2Char>(mString[0]) & 15U) << 12U) |
                    ((static_cast<Ucs2Char>(mString[1]) & 63U) << 6U) |
                    (static_cast<Ucs2Char>(mString[2]) & 63U));
            }
        }

    private:
        /** Read position of the current code point */
        const Char *mString;
    };
/// @}
}}
#endif
