//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Platform_AtomicOp_h)
#define FeatStd_Platform_AtomicOp_h

#include <FeatStd/Platform/Base.h>

#include FEATSTD_PLATFORM_INCLUDE(AtomicOp)

namespace FeatStd { namespace Internal {
/// @addtogroup FEATSTD_PLATFORM
/// @{

    #define FEATSTD_ATOMIC_INITIALIZER(value) ::FeatStd::Internal::AtomicOp::Atomic(value)

    /** Atomic operations are operations for variables that are shared by multiple threads.
        The atomic operations protect against errors that can occur when the scheduler
        switches contexts while a thread is updating a variable that can be accessed
        by other threads. A typical usage is reference counting. */
    class AtomicOp {
        typedef Impl::AtomicOp Base;
    public:
        /** platform dependent value type of an atomic variable. */
        typedef Base::Type ValueType; 

        /** Atomic encapsulates an atomic value
            encapsulation should prevent unintended - non atomic - modifications of 
            an atomic variable */
        struct Atomic {
            /**
             * Default constructor performing C++ default initialization on mValue.
             */
            Atomic() :
                mValue(ValueType())
            {
            }

            explicit Atomic(ValueType value) :
                mValue(value)
            {
            }

            /** The value of the atomic variable.
                *do not access this member*
                Accessing this value directly might imply an unsynchronized access
                to a shared variable. */
            ValueType mValue;

            /** read the current value of the atomic variable
                Note that reading the value of an atomic variable might cause an
                unsynchronized access to a shared resource and to undefined behavior. 
                Nevertheless most platforms (ARM, Intel x86 / x64) support atomic read access 
                to atomic variables - thus, use at your own risk */
            const ValueType& operator*() const {
                return mValue;
            }

            /** write the current value of the atomic variable
                Note that writing the value of an atomic variable causes an
                unsynchronized access to a shared resource and to undefined behavior.
                Use this function only to initialize the starting value of an atomic 
                variable, but never use it when the variable is already accessed by
                multiple context */
            ValueType& operator*() {
                return mValue;
            }
        };

        /** Atomically adds a value.
            @code
                tmp = atomic;
                atomic += value;
                return tmp;
            @endcode 
            @return the initial value of atomic */
        static inline ValueType FetchAndAdd(Atomic& atomic, ValueType value) {
            return Base::FetchAndAdd(atomic.mValue, value);
        }

        /** Atomically subtracts a value.
            @code
                tmp = atomic;
                atomic -= value;
                return tmp;
            @endcode 
            @return the initial value of atomic */
        static inline ValueType FetchAndSub(Atomic& atomic, ValueType value) {
            return Base::FetchAndSub(atomic.mValue, value);
        }

        /** Atomically adds a value.
            @code
                atomic += value;
                return atomic;
            @endcode 
            @return the new value of atomic */
        static inline ValueType AddAndFetch(Atomic& atomic, ValueType value) {
            return Base::AddAndFetch(atomic.mValue, value);
        }

        /** Atomically subtracts a value.
            @code
                atomic -= value;
                return atomic;
            @endcode 
            @return the new value of atomic */
        static inline ValueType SubAndFetch(Atomic& atomic, ValueType value) {
            return Base::SubAndFetch(atomic.mValue, value);
        }

        /** Atomically increment a value.
            Function provided for backward compatibility and is mapped to AtomicOp::AddAndFetch
            @code
                ++atomic;
                return atomic;
            @endcode 
            @return the new value of atomic */
        static inline ValueType Inc(Atomic& atomic);

        /** Atomically decrement a value.
            Function provided for backward compatibility and is mapped to AtomicOp::SubAndFetch
            @code
                --atomic;
                return atomic;
            @endcode 
            @return the new value of atomic */
        static inline ValueType Dec(Atomic& atomic);

        /** Atomically sets an integer value.
            @code
                ValueType prev_value = value;
                atomic = new_value;
                return prev_value;
            @endcode

            @return     The previous value. */
        static ValueType Set(Atomic& atomic, ValueType new_value);

        /** Atomic operation to compare data with an expected value.
            If the current value is equal to the expected one it is set to the new value.
            @code
                bool match = (value == expected_value);
                if (match) {
                    value = new_value;
                }
                return match;
            @endcode

            @return <em>true</em>, if test and set was successful */
        static bool TestAndSet(Atomic& atomic, ValueType expected_value, ValueType new_value);
    };
    
    inline AtomicOp::ValueType AtomicOp::Inc(Atomic& atomic) {
        return AddAndFetch(atomic, 1);
    }
    
    inline AtomicOp::ValueType AtomicOp::Set(Atomic& atomic, ValueType new_value) {
        return Base::Set(atomic.mValue, new_value);
    }
    
    inline AtomicOp::ValueType AtomicOp::Dec(Atomic& atomic) {
        return SubAndFetch(atomic, 1);
    }

    inline bool AtomicOp::TestAndSet(Atomic& atomic, ValueType expected_value, ValueType new_value) {
        return Base::TestAndSet(atomic.mValue, expected_value, new_value);
    }
    
/// @}
}}
#endif
