//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_MemoryManagement_TypeTraits_h)
#define FeatStd_MemoryManagement_TypeTraits_h

#include <FeatStd/Base.h>

/// @addtogroup FEATSTD_MEMORYMANAGEMENT
/// @{
template<typename T> struct FeatStdTypeTrait;

namespace FeatStd { namespace TypeTraits { namespace Internal {
template<bool isPodType = false> struct Base {
    enum { IsPodType = isPodType };
};

template<typename X> struct UnConstType { typedef X Type; };
template<typename X> struct UnConstType  <const X> { typedef X Type; };

template<typename X> struct UnRef { typedef X Type; };
template<typename X> struct UnRef<X &> { typedef X Type; };

template<typename X> struct ConstType { typedef const typename UnConstType<X>::Type Type; };

/** OrCondition */
template<bool a, bool b> struct OrCondition { enum { Value = true }; };
template<> struct OrCondition<false, false> { enum { Value = false }; };

/** Pointer type identification */
template<typename T> struct IsPointerType { enum { Value = false }; };
template<typename T> struct IsPointerType<T*> { enum { Value = true }; };

/** Used in IsBaseOf */
typedef Char (&IsBaseOf_Yes)[1];
typedef Char (&IsBaseOf_No)[2];

/** Helper class for IsBaseOf */
template <typename Base, typename Derived>
struct IsBaseOfHost
{
    operator Base*() const;
    operator Derived*();
};

/**
    * @brief Determine whether a type is a base from another type.
    *
    * In this context a type B is base from a type D if conversions
    * from pointer to D to pointers to B is implicit.
    * i.e.
    *     B* b = static_cast<D*>(0);
    * is legal.
    *
    * @param Base Base type.
    * @param Derived Derived type.
    */
template <typename Base, typename Derived>
struct IsBaseOf
{
    template <typename T>
    static IsBaseOf_Yes Check(Derived*, T);

    static IsBaseOf_No Check(Base*, Int);

    static const bool Value = sizeof(Check(IsBaseOfHost<Base,Derived>(), Int())) == sizeof(IsBaseOf_Yes);
};

/** Array meta information */
template<typename T> struct ArrayTypeTrait {
    typedef T Type;
    enum {
        IsArray = false,
        Dimension = 0
    };
};

template<typename T, SizeType N> struct ArrayTypeTrait<T[N]> {
    typedef T Type;
    enum {
        IsArray = true,
        Dimension = N
    };
};

template<typename T> struct ArrayTypeTrait<T[]> {
    typedef T Type;
    enum {
        IsArray = false,
        Dimension = 0
    };
};

/** Everything which is not a pointer and not tagged to be POD is Non-POD */
template<typename T> struct TypeTrait {
    enum { IsPodType = OrCondition<IsPointerType<T>::Value, FeatStdTypeTrait<T>::IsPodType>::Value };
};

template<> struct TypeTrait<bool> : Base<true> { };

template<> struct TypeTrait<char> : Base<true> { };
template<> struct TypeTrait<signed char> : Base<true> { };
template<> struct TypeTrait<unsigned char> : Base<true> { };

template<> struct TypeTrait<wchar_t> : Base<true> { };

template<> struct TypeTrait<short> : Base<true> { };
template<> struct TypeTrait<unsigned short> : Base<true> { };

template<> struct TypeTrait<int> : Base<true> { };
template<> struct TypeTrait<unsigned int> : Base<true> { };

template<> struct TypeTrait<long> : Base<true> { };
template<> struct TypeTrait<unsigned long> : Base<true> { };

template<> struct TypeTrait<float> : Base<true> { };
template<> struct TypeTrait<double> : Base<true> { };

template<> struct TypeTrait<void> : Base<true> { };

// cost PODs

template<> struct TypeTrait<const bool> : Base<true> { };

template<> struct TypeTrait<const char> : Base<true> { };
template<> struct TypeTrait<const signed char> : Base<true> { };
template<> struct TypeTrait<const unsigned char> : Base<true> { };

template<> struct TypeTrait<const wchar_t> : Base<true> { };

template<> struct TypeTrait<const short> : Base<true> { };
template<> struct TypeTrait<const unsigned short> : Base<true> { };

template<> struct TypeTrait<const int> : Base<true> { };
template<> struct TypeTrait<const unsigned int> : Base<true> { };

template<> struct TypeTrait<const long> : Base<true> { };
template<> struct TypeTrait<const unsigned long> : Base<true> { };

template<> struct TypeTrait<const float> : Base<true> { };
template<> struct TypeTrait<const double> : Base<true> { };

template<> struct TypeTrait<const void> : Base<true> { };
}}}

/**
 *  FeatStdTypeTrait is a class to provide type meta information to FeatStd.
 *  type meta information is a source for optimization in eg. FeatStd memory management.
 *  By tagging a type as POD (Plain Old Data), FeatStd can use alternative means to
 *  operate on objects of the type. Use following statement to define a types
 *  traits:
 *  @code
 *  struct Pod {
 *  int a;
 *  int b;
 *  };
 *  template<> struct FeatStdTypeTrait<Pod> : FeatStd::TypeTraits::Internal::Base<true> { };
 *  @endcode
 */
template<typename T> struct FeatStdTypeTrait : FeatStd::TypeTraits::Internal::Base<false> { };

/** Declare the given type to be POD (plain old data). 
    Knowing if a type is POD or not enables several optimization schemes in 
    FeatStd::MemoryManagement. If a type is POD, construction and destruction
    of type instances are suppressed. */
#define FEATSTD_DECLARE_PODTYPE(type) \
    FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA C++ 2008 Required Rule 7-3-1: removing the symbol from global name space violates backward compatibility. FeatStd prefix should be sufficient to avoid conflicts.") \
    FEATSTD_LINT_NEXT_EXPRESSION(1577, "MISRA C++ 2008 Required Rule 14-7-3: specialization only can be done with type declaration.") \
    template<> struct FeatStdTypeTrait< type > : FeatStd::TypeTraits::Internal::Base<true> { }

/** Declare the given type to be POD (plain old data). 
    Knowing if a type is POD or not enables several optimization schemes in 
    FeatStd::MemoryManagement. If a type is POD, construction and destruction
    of type instances are suppressed. */
#define FEATSTD_DECLARE_TEMPLATE_PODTYPE(typeList, type) \
    FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA C++ 2008 Required Rule 7-3-1: removing the symbol from global name space violates backward compatibility. FeatStd prefix should be sufficient to avoid conflicts.") \
    FEATSTD_LINT_NEXT_EXPRESSION(1577, "MISRA C++ 2008 Required Rule 14-7-3: specialization only can be done with type declaration.") \
    template< typeList > struct FeatStdTypeTrait< type > : FeatStd::TypeTraits::Internal::Base<true>{}

/// @}
#endif
