//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_MemoryManagement_PlatformHeap_h)
#define FeatStd_MemoryManagement_PlatformHeap_h

#include <FeatStd/Base.h>
#if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)
#   include <FeatStd/MemoryPool/MemoryPool.h>
#   include <FeatStd/MemoryPool/MemoryPoolDebug.h>
#   include <FeatStd/Util/NumericUtil.h>
#endif

namespace FeatStd { namespace MemoryManagement {
/// @addtogroup FEATSTD_MEMORYMANAGEMENT
/// @{

    /**
        *  @brief This heap policy delegates heap creation to the platform interface MemoryPlatform.h. The memory
        *  platform functions defined there are used for any heap requests.
        */
    class PlatformHeap {
        public:

#if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)
            /**
                *  Allocate nbytes bytes from the ExternalHeap object.
                *  Applications should use typed New and NewArray methods to allocate memory.
                *  @param nbytes number of bytes to allocate.
                *  @param context allocation context
                *  @param attrib attributes of requested memory
                *  @return pointer to the allocated buffer or 0 if out of memory
                */
            static void* Alloc(FEATSTD_MEMORYPOOL_DECL_CONTEXT(SizeType nbytes, context), UInt8 attrib) {
                return FeatStd::DefaultMemoryPool::Alloc(FEATSTD_MEMORYPOOL_PASS_CONTEXT((FeatStd::Internal::NumericConversion<UInt32, SizeType>(nbytes)), context), attrib);
            }

#if defined(FEATSTD_MEMORYPOOL_LEAK_DETECTION_ENABLED)
            /**
                *  Reallocates the buffer to new size.
                *  @param p pointer to the buffer to reallocate.
                *  @param nbytes the new size to reallocate the buffer to.
                *  @param context allocation context
                *  @return pointer to the reallocated buffer or 0 if out of memory
                */
            static void* Realloc(void *p, FEATSTD_MEMORYPOOL_DECL_CONTEXT(SizeType nbytes, context)) {
                return FeatStd::DefaultMemoryPool::GlobalRealloc(p, FEATSTD_MEMORYPOOL_PASS_CONTEXT((FeatStd::Internal::NumericConversion<UInt32, SizeType>(nbytes)), context));
            }
#endif

            /**
                *  AllocArray allocates an array buffer for a given number of objects
                *  Beside allocating the array, AllocArray will store itemSize and itemCount meta information
                *  for later retrieval. Thus AllocArray implies an overhead of 2x4 bytes per allocation.
                *  @param itemSize size in bytes of one array item
                *  @param itemCount number of items in the array
                *  @param context allocation context
                *  @param attrib attributes of requested memory
                *  @return pointer to allocated array buffer
                */
            static void* AllocArray(SizeType itemSize, FEATSTD_MEMORYPOOL_DECL_CONTEXT(SizeType itemCount, context), UInt8 attrib);
#endif

            /**
                *  Allocate nbytes bytes from the ExternalHeap object.
                *  Applications should use typed New and NewArray methods to allocate memory.
                *  @param nbytes number of bytes to allocate.
                *  @return pointer to the allocated buffer or 0 if out of memory
                */
            static void* Alloc(SizeType nbytes);

            /**
                *  Reallocates the buffer to new size.
                *  @param p pointer to the buffer to reallocate.
                *  @param nbytes the new size to reallocate the buffer to.
                *  @return pointer to the reallocated buffer or 0 if out of memory
                */
            static void* Realloc(void *p, SizeType nbytes);


            /**
                *  AllocArray allocates an array buffer for a given number of objects
                *  Beside allocating the array, AllocArray will store itemSize and itemCount meta information
                *  for later retrieval. Thus AllocArray implies an overhead of 2x4 bytes per allocation.
                *  @param itemSize size in bytes of one array item
                *  @param itemCount number of items in the array
                *  @return pointer to allocated array buffer
                */
            static void* AllocArray(SizeType itemSize, SizeType itemCount);

            /**
                *  Releases a buffer previously allocated with ExternalHeap::Alloc.
                *  The actual implementation is empty.
                *  @param p pointer to the buffer to be released
                */
            static void Free(void *p);

            /**
                *  FreeArray frees a previously with AllocArray allocated buffer
                *  The actual implementation in StackHeap does not free memory. FreeArray will
                *  perform integrity checks on the array if heap debugging is activated
                *  ( @see FeatStdStackHeapDebugFlags )
                *  @param p pointer to a buffer previously allocated with AllocArray
                */
            static void FreeArray(void *p);

            /**
                *  GetArrayInfo returns meta information about a buffer previously allocated with AllocArray
                *  Beside retrieving array buffer meta information, GetArrayInfo will also perform
                *  integrity checks on the array
                *  @param p pointer to buffer previously allocated with AllocArray
                *  @param itemSize retrieves the size of an item in array buffer
                *  @param itemCount retrieves the number of items in the buffer
                */
            static void GetArrayInfo(void *p, SizeType *itemSize, SizeType *itemCount);

            /**
                *  Estimates the size of the largest available block which can be allocated.
                *  This interface is used for satisfying the FeatStdHeapAllocator interface max_size().
                *  @return largest allocable block available in bytes.
                */
            static SizeType Available();

    };
/// @}
}}
#endif
