//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

//! [FEATSTD_LOG_SET_REALM_CPP]
#include "PlatformHeap.h"

#include <FeatStd/Diagnostics/Log.h>
#include <FeatStd/Util/PointerUtil.h>

#if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)
    #include <FeatStd/MemoryPool/MemoryPool.h>
#else
    #include <FeatStd/Platform/Memory.h>
    #include <FeatStd/Diagnostics/SystemMemoryStatistic.h>
#endif

namespace FeatStd { namespace MemoryManagement {
    using namespace FeatStd::Internal;

    struct ArrayMetaData {
        SizeType itemSize;
        SizeType itemCount;
    };

    // ========================================================================
    #if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)

    // ------------------------------------------------------------------------
    void* PlatformHeap::Alloc(SizeType nbytes)
    {
        return DefaultMemoryPool::Alloc(FEATSTD_MEMORYPOOL_CALL_CONTEXT((FeatStd::Internal::NumericConversion<UInt32, SizeType>(nbytes))), UInt8(MemAttrib::Transient));
    }

    // ------------------------------------------------------------------------
    void PlatformHeap::Free(void *p)
    {
        DefaultMemoryPool::GlobalFree(p);
    }

    // ------------------------------------------------------------------------
    void* PlatformHeap::Realloc(void *p, SizeType nbytes)
    {
        return DefaultMemoryPool::GlobalRealloc(p, FEATSTD_MEMORYPOOL_CALL_CONTEXT((FeatStd::Internal::NumericConversion<UInt32, SizeType>(nbytes))));
    }

    // ------------------------------------------------------------------------
    void* PlatformHeap::AllocArray(SizeType itemSize, FEATSTD_MEMORYPOOL_DECL_CONTEXT(SizeType itemCount, context), UInt8 attrib)
    {
        static const SizeType metaDataSize = sizeof(ArrayMetaData);   // Size to store array MetaData.
        SizeType dataSize = itemSize * itemCount;                     // Size if requested data.

        // Total allocated size is requested size plus additional data for meta information.
        FEATSTD_LINT_NEXT_EXPRESSION(925, "alignment restrictions apply - cast is considered to be safe.")
        ArrayMetaData *p = static_cast<ArrayMetaData*>(DefaultMemoryPool::Alloc(FEATSTD_MEMORYPOOL_PASS_CONTEXT((FeatStd::Internal::NumericConversion<UInt32, SizeType>(dataSize + metaDataSize)), context), attrib));
        if (p == 0) {
            return 0;
        }

        // write meta information at beginning of allocated data.
        p->itemSize = itemSize;
        p->itemCount = itemCount;

        return p + 1;
    }

    // ------------------------------------------------------------------------
    void* PlatformHeap::AllocArray(SizeType itemSize, SizeType itemCount)
    {
        return AllocArray(itemSize, FEATSTD_MEMORYPOOL_CALL_CONTEXT(itemCount), UInt8(MemAttrib::Transient));
    }

    // ------------------------------------------------------------------------
    void PlatformHeap::FreeArray(void *p)
    {
        if (p != 0) {
            FEATSTD_LINT_NEXT_EXPRESSION(925, "alignment restrictions apply - cast is considered to be safe.")
            DefaultMemoryPool::GlobalFree(static_cast<ArrayMetaData*>(p)-1);
        }
    }

    // ------------------------------------------------------------------------
    void PlatformHeap::GetArrayInfo(void *p, SizeType *itemSize, SizeType *itemCount)
    {
        if (p != 0) {
            FEATSTD_LINT_NEXT_EXPRESSION(925, "alignment restrictions apply - cast is considered to be safe.")
            ArrayMetaData* arrayMetaData = reinterpret_cast<ArrayMetaData*>(p)-1;
            if (itemSize != 0) {
                *itemSize = arrayMetaData->itemSize;
            }

            if (itemCount != 0) {
                *itemCount = arrayMetaData->itemCount;
            }
        }
        else {
            if (itemSize != 0) {
                *itemSize = 0;
            }

            if (itemCount != 0) {
                *itemCount = 0;
            }
        }
    }

    // ------------------------------------------------------------------------
    SizeType PlatformHeap::Available()
    {
        return UInt32(0xffffffff);
    }


    // ========================================================================
    #else   // !defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)

    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::FeatStdMemoryManagement);

    using FeatStd::Diagnostics::SystemMemoryStatistic;

    void* PlatformHeap::Alloc(SizeType nbytes)
    {
        if (nbytes == 0)
        {
            return 0;
        }

        void* ptr =  Memory::MemAlloc(nbytes + SystemMemoryStatistic::GetAllocationInfoSize());
        if (ptr != 0) {
            ptr = SystemMemoryStatistic::OnMemoryAllocated(ptr, nbytes);
        }
        else {
            FEATSTD_LOG_FATAL("out of memory!");
        }
        return ptr;
    }

    void PlatformHeap::Free(void *p)
    {
        if (p != 0) {
            p = SystemMemoryStatistic::OnMemoryFreed(p);
            Memory::MemFree(p);
        }
    }

    void* PlatformHeap::Realloc(void *p, SizeType nbytes)
    {
        if (p==0) {
            return Alloc(nbytes);
        }
    
        if (nbytes==0){
            Free(p);
            return 0;
        }
    #ifdef FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED
        static_cast<void>(SystemMemoryStatistic::OnMemoryFreed(p));
        FEATSTD_LINT_NEXT_EXPRESSION(925, "pointer to pointer cast necessary to keep memory interface transparent to upper software layers.")
        p = Memory::MemRealloc((reinterpret_cast<SystemMemoryStatistic::AllocationInfoInternal*>(p)-1), nbytes + SystemMemoryStatistic::GetAllocationInfoSize()); 
        p = SystemMemoryStatistic::OnMemoryAllocated(p, nbytes);
    #else
        p = Memory::MemRealloc(p, nbytes);
    #endif

        return p;
    }

    void* PlatformHeap::AllocArray(SizeType itemSize, SizeType itemCount)
    {
        static SizeType metaDataSize = sizeof(ArrayMetaData); // Size to store array MetaData.
        SizeType dataSize = itemSize * itemCount; // Size if requested data.

        // Total allocated size is requested size plus additional data for meta information.
        ArrayMetaData *p = PointerToPointer<ArrayMetaData*>(Memory::MemAlloc(dataSize + metaDataSize + SystemMemoryStatistic::GetAllocationInfoSize()));

        if (p != 0) {
            // write meta information at beginning of allocated data.
            p->itemSize = itemSize;
            p->itemCount = itemCount;

            // return begin of "data area".
            return SystemMemoryStatistic::OnMemoryAllocated(p + 1, dataSize);
        }
        else {
            FEATSTD_LOG_FATAL("out of memory!");
            return 0;
        }
    }

    void PlatformHeap::FreeArray(void *p)
    {
        if (p != 0) {
            ArrayMetaData* completeData = PointerToPointer<ArrayMetaData*>(SystemMemoryStatistic::OnMemoryFreed(p)) - 1;
            Memory::MemFree(completeData);
        }
    }

    void PlatformHeap::GetArrayInfo(void *p, SizeType *itemSize, SizeType *itemCount)
    {
        if (p != 0) {
            ArrayMetaData* arrayMetaData = PointerToPointer<ArrayMetaData*>(SystemMemoryStatistic::GetAllocationInfoLocation(p)) - 1;
            if (itemSize != 0)
            {
                *itemSize = arrayMetaData->itemSize;
            }
            else {
                FEATSTD_LOG_FATAL("Get array info failed, null-pointer passed for out-param itemsSize.\n");
            }

            if (itemCount != 0)
            {
                *itemCount = arrayMetaData->itemCount;
            }
            else {
                FEATSTD_LOG_FATAL("Get array info failed, null-pointer passed for out-param itemCount.\n");
            }
        }
        else {
            FEATSTD_LOG_FATAL("Get array info failed, null-pointer passed as buffer.\n");
        }
    }

    // ------------------------------------------------------------------------
    SizeType PlatformHeap::Available()
    {
        return Memory::MemAvailable();
    }

#endif

}}
