//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_MemoryManagement_PlacementNew_h)
#define FeatStd_MemoryManagement_PlacementNew_h

#include <FeatStd/Base.h>
#include <FeatStd/Util/Traits.h>
#include <FeatStd/MemoryManagement/TypeTraits.h>
#include <FeatStd/Util/Alignment.h>

/// @addtogroup FEATSTD_MEMORYMANAGEMENT
/// @{
namespace FeatStd { namespace MemoryManagement { namespace Internal {
    //added because of IAR compiler
    struct PlacementNewType;
}}}

void* operator new(FeatStd::SizeType, FeatStd::MemoryManagement::Internal::PlacementNewType *p);

namespace FeatStd { namespace MemoryManagement { 
    namespace Internal {
        /**
            *  dummy type to declare placement new and delete functions (avoid conflict with standard definitions in \<new\>)
            */
        struct PlacementNewType { };

        /**
            *  Supplies Destruct methods for Non-POD objects that will invoke the objects Dtor
            */
        template<typename T, bool podType = false> struct Destructor {
            /**
                *  Destructs given object by invoking its Dtor
                *  @param object the object the Dtor should be invoked on
                */
            static inline void Destruct(T *object) {
                (void) object;  // MSVC warns unused parameter object if T is a (not for FeatSTd known) POD type
                object->~T();
            }

            /**
                *  Supplies Destruct method that will invoke the objects Dtor
                *  Objects get destructed in reverse order of construction (n - 1) to 0.
                *  @param object the array of objects the Dtor should be invoked on
                *  @param itemSize size of items
                *  @param nItems number of items in the object array - must be > 0
                */
            static void Destruct(T *object, SizeType itemSize, SizeType nItems) {
                typedef typename FeatStd::Internal::PureType<T>::Type CleanType;
                using FeatStd::Internal::aligned_reinterpret_cast;
                if (nItems > 0) {
                    UInt8 *p = aligned_reinterpret_cast<UInt8*>(const_cast<CleanType*>(object)) + (nItems * itemSize);
                    do {
                        p -= itemSize;
                        aligned_cast<T*>(p)->~T();
                    } while(p != static_cast<void*>(const_cast<CleanType*>(object)));
                }
            }
        };

        /**
            *  Supplies Destruct method for POD types (no Dtor invoke)
            */
        template<typename T> struct Destructor<T, true> {
            static inline void Destruct(T *) { }
            static inline void Destruct(T *, SizeType, SizeType) { }
        };

        template<typename T, bool isPod = false> struct ConstructHelper {
            static inline T* Construct(void *p) {
                return new(static_cast<PlacementNewType*>(p)) T;
            }

            template<typename A0> static inline T* Construct(void *p, const A0 &a0) {
                return new(static_cast<PlacementNewType*>(p)) T(a0);
            }

            template<typename A0> static inline T* Construct(void *p, A0 &a0) {
                return new(static_cast<PlacementNewType*>(p)) T(a0);
            }
        };

        template<typename T> struct ConstructHelper<T, true> {
            static inline T* Construct(void *p) {
                return static_cast<T*>(p);
            }

            template<typename A0> static inline T* Construct(void *p, const A0 &a0) {
                *static_cast<T*>(p) = a0;
                return static_cast<T*>(p);
            }

            template<typename A0> static inline T* Construct(void *p, A0 &a0) {
                *static_cast<T*>(p) = a0;
                return static_cast<T*>(p);
            }
        };

    } // namespace Internal

    /**
        *  Invokes FeatStd placement new on object
        *  @param object memory location for object
        */
    template<typename T> void Construct(T *object) {
        FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4345, "warns about POD type object construction.")
        new(reinterpret_cast<Internal::PlacementNewType*>(object)) T;
    }

    /**
        *  Invokes FeatStd placement new on object with the given initial value (invokes copy ctor)
        *  @param object memory location for object
        *  @param initVal value of the object
        */
    template<typename T> void Construct(T *object, const T &initVal) {
        FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4345, "warns about POD type object construction.")
        new(reinterpret_cast<Internal::PlacementNewType*>(object)) T(initVal);
    }
    template<typename T> void Construct(T *object, T &initVal) {
        FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4345, "warns about POD type object construction.")
            new(reinterpret_cast<Internal::PlacementNewType*>(object)) T(initVal);
    }

    /**
        *  Destructs (calls Dtor) obj.
        *  @param obj The object that will be destroyed.
        */
    template<typename T> inline void Destruct(T *obj) {
        Internal::Destructor<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::Destruct(obj);
    }

    /**
        *  Destructs (calls Dtor) obj.
        *  @param obj      The object that will be destroyed.
        *  @param itemSize The item size.
        *  @param nItems   The number of items.
        */
    template<typename T> inline void Destruct(T *obj, SizeType itemSize, SizeType nItems) {
        Internal::Destructor<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::Destruct(obj, itemSize, nItems);
    }



    template<typename T> 
    inline T* ConstructObject(void *p) {
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
        return Internal::ConstructHelper<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::Construct(p);
    }

    template<typename T, typename A0> 
    inline T* ConstructObject(void *p, const A0 &a0) {
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
        return Internal::ConstructHelper<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::Construct(p, a0);
    }

    template<typename T, typename A0>
    inline T* ConstructObject(void *p, A0 &a0) {
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
        return Internal::ConstructHelper<T, FeatStd::TypeTraits::Internal::TypeTrait<T>::IsPodType>::Construct(p, a0);
    }

    template<typename T, typename A0, typename A1> 
    inline T* ConstructObject(void *p, const A0 &a0, const A1 &a1) {
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
        return new(static_cast<FeatStd::MemoryManagement::Internal::PlacementNewType*>(p)) T(a0, a1);
    }
    template<typename T, typename A0, typename A1>
    inline T* ConstructObject(void *p, A0 &a0, A1 &a1) {
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
        return new(static_cast<FeatStd::MemoryManagement::Internal::PlacementNewType*>(p)) T(a0, a1);
    }

    template<typename T, typename A0, typename A1, typename A2> 
    inline T* ConstructObject(void *p, const A0 &a0, const A1 &a1, const A2 &a2) {
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
        return new(static_cast<FeatStd::MemoryManagement::Internal::PlacementNewType*>(p)) T(a0, a1, a2);
    }
    template<typename T, typename A0, typename A1, typename A2>
    inline T* ConstructObject(void *p, A0 &a0, A1 &a1, A2 &a2) {
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
        return new(static_cast<FeatStd::MemoryManagement::Internal::PlacementNewType*>(p)) T(a0, a1, a2);
    }

    template<typename T, typename A0, typename A1, typename A2, typename A3> 
    inline T* ConstructObject(void *p, const A0 &a0, const A1 &a1, const A2 &a2, const A3 &a3) {
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
        return new(static_cast<FeatStd::MemoryManagement::Internal::PlacementNewType*>(p)) T(a0, a1, a2, a3);
    }
    template<typename T, typename A0, typename A1, typename A2, typename A3>
    inline T* ConstructObject(void *p, A0 &a0, A1 &a1, A2 &a2, A3 &a3) {
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
        return new(static_cast<FeatStd::MemoryManagement::Internal::PlacementNewType*>(p)) T(a0, a1, a2, a3);
    }

    template<typename T, typename A0, typename A1, typename A2, typename A3, typename A4> 
    inline T* ConstructObject(void *p, const A0 &a0, const A1 &a1, const A2 &a2, const A3 &a3, const A4 &a4) {
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
        return new(static_cast<FeatStd::MemoryManagement::Internal::PlacementNewType*>(p)) T(a0, a1, a2, a3, a4);
    }
    template<typename T, typename A0, typename A1, typename A2, typename A3, typename A4>
    inline T* ConstructObject(void *p, A0 &a0, A1 &a1, A2 &a2, A3 &a3, A4 &a4) {
        FEATSTD_COMPILETIME_ASSERT_NOT(FeatStd::Internal::ArrayTrait<T>::IsArray);
        return new(static_cast<FeatStd::MemoryManagement::Internal::PlacementNewType*>(p)) T(a0, a1, a2, a3, a4);
    }

    template<typename T> inline void DestructObject(T *p) {
        Destruct(p);
    }

    template<typename T> inline void DestructObject(void *p) {
        Destruct(static_cast<T*>(p));
    }


}}

/**
 *  FeatStd specific placement new operator
 */
inline void* operator new(FeatStd::SizeType, FeatStd::MemoryManagement::Internal::PlacementNewType *p){
    return p;
}

/**
 *  Complementary operator delete for FeatStd specific placement new operator
 */
inline void operator delete(void *, FeatStd::MemoryManagement::Internal::PlacementNewType *) {
}

/**
 *  FeatStd specific placement new array operator
 */
inline void* operator new[](FeatStd::SizeType, FeatStd::MemoryManagement::Internal::PlacementNewType *p) {
    return p;
}

/**
 *  Complementary operator delete for FeatStd specific placement new array operator
 */
inline void operator delete[](void *, FeatStd::MemoryManagement::Internal::PlacementNewType *) {
}

/// @}
#endif  // PlacementNew_h
