//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_MemoryManagement_MemoryUtil_h)
#define FeatStd_MemoryManagement_MemoryUtil_h

#include <FeatStd/Base.h>

/// @addtogroup FEATSTD_MEMORYMANAGEMENT
/// @{
/**
 *  Evaluates to the number of bytes required to store nItem objects of type T in an array
 *  @param T type of the array objects
 *  @param nItems number of bytes to be stored in the array
 */
#define _ArrayByteSize(T, nItems) (sizeof(T[2]) / 2 * (nItems))

/**
 *  Evaluates to the number of UInt32 required to store nItem objects of type T in an array
 *  The following code dimensions array buffer to hold 4 Foo object
 *  @code
 *  struct Foo {
 *  ...
 *  };
 *  UInt32 buffer[ArrayUInt32Size(Foo, 4];
 *  Foo *foo = reinterpret_cast<Foo*>(buffer);
 *  @endcode
 *  @param T type of the array objects
 *  @param nItems number of bytes to be stored in the array
 */
#define _ArrayUInt32Size(T, nItems) (((sizeof(T[2]) / 2 * (nItems)) + sizeof(UInt32) - 1) / sizeof(UInt32))

/**
 *  Evaluates to the number of T items that can be stored in a buffer with size bufferByteSize
 *  @param T type of the array objects
 *  @param bufferByteSize size of the buffer in bytes
 */
#define _ArrayItemCount(T, bufferByteSize) ((bufferByteSize) / (sizeof(T[2]) / 2))

namespace FeatStd { namespace MemoryManagement {
/**
    *  Returns size of an array of T objects in bytes
    *  @param nItems number of elements in the array
    *  @return the size in bytes of the array of T objects with elemCount array items
    */
template<typename T> inline SizeType ArrayByteSize(SizeType nItems) {
    return _ArrayByteSize(T, nItems);
}

/**
    *  Returns the nummber of T items that can stored in a buffer with size bufferByteSize
    *  @param byteSize size of the buffer in bytes
    *  @return number of items that can be stored in an array of the given size
    */
template<typename T> inline UInt32 ArrayItemCount(UInt32 byteSize) {
    return _ArrayItemCount(T, byteSize);
}
}}
/// @}
#endif
