//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_MemoryManagement_Heap_h)
#define FeatStd_MemoryManagement_Heap_h

#include <new> // placement new operator

#include <FeatStd/Util/NumericUtil.h>
#include <FeatStd/Util/PointerUtil.h>
#include <FeatStd/MemoryManagement/PlatformHeap.h>
#include <FeatStd/MemoryManagement/Operators.h>
#include <FeatStd/Platform/Types.h>

#if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)
#   include <FeatStd/MemoryPool/MemoryPool.h>
#endif

#if defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)
/// @addtogroup FEATSTD_MEMORYMANAGEMENT
/// @{

// ========================================================================

/**
 *  FEATSTD_NEW allocates an arbitrary object using the Memory allocator PlatformHeap. For Deleting objects on
 *  that heap, use FEATSTD_DELETE. FEATSTD_NEW can be used for object and for POD types.
 */
#define FEATSTD_NEW(type) FEATSTD_MEMORYPOOL_TRANSIENT_NEW(FeatStd::DefaultMemoryPool, type)

/**
 *  FEATSTD_DELETE deallocates objects created with FEATSTD_NEW and calls the appropriate destructor if objectPtr
 *  points to an object type. For POD, no destructor will be called.
 */
#define FEATSTD_DELETE(objectPtr) FEATSTD_MEMORYPOOL_DELETE(objectPtr)

/** FEATSTD_SAFE_DELETE acts like FEATSTD_DELETE, and additionally sets the given object pointer to 0. */
FEATSTD_LINT_MACRO_WHILEFALSE(FEATSTD_SAFE_DELETE)
#define FEATSTD_SAFE_DELETE(ptr) \
    do { \
        FEATSTD_DELETE(ptr); \
        ptr = 0; \
    FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) \
    } while (false)

/**
 *  FEATSTD_DELETE_T acts like FEATSTD_DELETE with exception that the type of the object to be destroyed can be explicitly
 *  defined in case FEATSTD_DELETE cannot determine the type correctly.
 */
#define FEATSTD_DELETE_T(objectPtr, type) \
    FEATSTD_MEMORYPOOL_DELETE(FeatStd::Internal::PointerToPointer<type*>(objectPtr))

/**
 *  FEATSTD_NEW_ARRAY creates an array of an arbitrary object or POD type. Count specifies the number of instances to be
 *  created. Arrays created using this operation need to be deleted with FEATSTD_DELETE_ARRAY.
 */
#define FEATSTD_NEW_ARRAY(type, count) \
        FEATSTD_MEMORYPOOL_ARRAY_NEW(FeatStd::DefaultMemoryPool, \
                                     FeatStd::MemoryManagement::MemAttrib::Transient, \
                                     type, \
                                     (count))

/**
 *  FEATSTD_NEW_PRESET_ARRAY acts line FEATSTD_NEW_ARRAY but uses the copy constructor of the array elements to
 *  initialize them all with preset.
 */
#define FEATSTD_NEW_PRESET_ARRAY(type, count, preset) \
        FEATSTD_MEMORYPOOL_ARRAY_NEW_WITH_PRESET(FeatStd::DefaultMemoryPool, \
                                     FeatStd::MemoryManagement::MemAttrib::Transient, \
                                     type, \
                                     (count), \
                                     (preset))

/**
 *  FEATSTD_DELETE_ARRAY deallocates objects allocated with FEATSTD_NEW_ARRAY. For each element instance the destructor
 *  is called.
 */
#define FEATSTD_DELETE_ARRAY(objectPtr) FEATSTD_MEMORYPOOL_ARRAY_DELETE(objectPtr)

/**
 *  FEATSTD_DELETE_ARRAY_T acts like FEATSTD_DELETE_ARRAY with exception that the type of the object to be destroyed
 *  can be explicitly defined in case FEATSTD_DELETE_ARRAY cannot determine the type correctly.
 */
#define FEATSTD_DELETE_ARRAY_T(objectPtr, type) \
    FEATSTD_MEMORYPOOL_ARRAY_DELETE(FeatStd::Internal::PointerToPointer<type*>(objectPtr))

/** FEATSTD_SAFE_DELETE_ARRAY acts like FEATSTD_DELETE_ARRAY, and additionally sets the given array pointer to 0. */
FEATSTD_LINT_MACRO_WHILEFALSE(FEATSTD_SAFE_DELETE_ARRAY)
#define FEATSTD_SAFE_DELETE_ARRAY(ptr) \
    do { \
        FEATSTD_DELETE_ARRAY(ptr); \
        ptr = 0; \
    FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) \
    } while (false)

/** Allocate given number of bytes from heap */
#define FEATSTD_ALLOC(nBytes) FeatStd::DefaultMemoryPool::Alloc(FEATSTD_MEMORYPOOL_CALL_CONTEXT(nBytes), FeatStd::UInt8(FeatStd::MemoryManagement::MemAttrib::Transient))

/** Re-allocate the given buffer with the specified size */
#define FEATSTD_REALLOC(buffer, nBytes) FeatStd::DefaultMemoryPool::Realloc(buffer, FEATSTD_MEMORYPOOL_CALL_CONTEXT(nBytes))

/** free previously heap allocated buffer */
#define FEATSTD_FREE(buffer) FeatStd::DefaultMemoryPool::GlobalFree(buffer)


/// @}

// ========================================================================
#else   // !defined(FEATSTD_ENABLE_MEMORYPOOL_ALLOCATIONS)


//  Only include and use SystemMemoryStatistic if enabled in CMake. (slightly improves build time)
#if defined(FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED) 
    #include <FeatStd/Diagnostics/SystemMemoryStatistic.h>

    #if defined(FEATSTD_THREADSAFETY_ENABLED)

        #define SYS_MEM_STATS(T) FeatStd::Internal::CriticalSectionLocker lLock(&FeatStd::Diagnostics::Internal::GetAllocationInfoCs()); \
                                Diagnostics::SystemMemoryStatistic::HintNextAllocation<T>(file, line);
    #else
        #define SYS_MEM_STATS(T) Diagnostics::SystemMemoryStatistic::HintNextAllocation<T>(file, line);
    #endif
#else
    #define SYS_MEM_STATS(T)  FEATSTD_UNUSED(file);  FEATSTD_UNUSED(line);
#endif

namespace FeatStd { namespace MemoryManagement { namespace Internal {
/// @addtogroup FEATSTD_MEMORYMANAGEMENT
/// @{

    inline void* PlainAlloc(FeatStd::SizeType nBytes, const Char* file, Int32 line) {
        SYS_MEM_STATS(void*)
        return FeatStd::MemoryManagement::PlatformHeap::Alloc(nBytes);
    }

    inline void* Realloc(void *buffer, FeatStd::SizeType nBytes, const Char* file, Int32 line) {
        SYS_MEM_STATS(void*)
        return FeatStd::MemoryManagement::PlatformHeap::Realloc(buffer, nBytes);
    }

    template<typename T> static void* Alloc(const Char* file, Int32 line){
        SYS_MEM_STATS(T)
        return FeatStd::MemoryManagement::PlatformHeap::Alloc(sizeof(T));
    }

    template<typename T> static T* CreateArray(FeatStd::SizeType count, const Char* file, Int32 line){
        SYS_MEM_STATS(T)
        return FeatStd::MemoryManagement::CreateArray<T>(count);
    }

    template<typename T> static T* CreateArray(FeatStd::SizeType count, const T& preset, const Char* file, Int32 line){
        SYS_MEM_STATS(T)
        return FeatStd::MemoryManagement::CreateArray<T>(count, preset);
    }
    
/// @}
}}}

#undef SYS_MEM_STATS

/// @addtogroup FEATSTD_MEMORYMANAGEMENT
/// @{
#if defined(FEATSTD_SYSTEM_MEMORY_STATISTIC_FILE_AND_LINE_TRACKING_ENABLED)
    #define FILE_LINE FEATSTD__FILE__, __LINE__
#else
    #define FILE_LINE 0, 0
#endif

/**
 *  FEATSTD_NEW allocates an arbitrary object using the Memory allocator PlatformHeap. For Deleting objects on
 *  that heap, use FEATSTD_DELETE. FEATSTD_NEW can be used for object and for POD types.
 */
#define FEATSTD_NEW(type) \
    FEATSTD_LINT_DYNAMIC_MEMORY() \
    new(FeatStd::MemoryManagement::Internal::Alloc<type>(FILE_LINE)) type
//TODO #define FEATSTD_NEW_ARG(type, ...)

/**
 *  FEATSTD_DELETE deallocates objects created with FEATSTD_NEW and calls the appropriate destructor if objectPtr
 *  points to an object type. For POD, no destructor will be called.
 */
#define FEATSTD_DELETE(objectPtr) \
    FEATSTD_LINT_DYNAMIC_MEMORY() \
    FeatStd::MemoryManagement::Destroy(objectPtr)

/** FEATSTD_SAFE_DELETE acts like FEATSTD_DELETE, and additionally sets the given object pointer to 0. */
FEATSTD_LINT_MACRO_WHILEFALSE(FEATSTD_SAFE_DELETE)
#define FEATSTD_SAFE_DELETE(ptr) \
    do { \
        FEATSTD_DELETE(ptr); \
        ptr = 0; \
    FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) \
    } while (false)

/**
 *  FEATSTD_DELETE_T acts like FEATSTD_DELETE with exception that the type of the object to be destroyed can be explicitly
 *  defined in case FEATSTD_DELETE cannot determine the type correctly.
 */
#define FEATSTD_DELETE_T(objectPtr, type) \
    FEATSTD_LINT_DYNAMIC_MEMORY() \
    FeatStd::MemoryManagement::Destroy<type >(FeatStd::Internal::PointerToPointer<type*>(objectPtr))

/**
 *  FEATSTD_NEW_ARRAY creates an array of an arbitrary object or POD type. Count specifies the number of instances to be
 *  created. Arrays created using this operation need to be deleted with FEATSTD_DELETE_ARRAY.
 */
#define FEATSTD_NEW_ARRAY(type, count) \
    FeatStd::MemoryManagement::Internal::CreateArray<type >(static_cast<FeatStd::SizeType>(count), FILE_LINE)

/**
 *  FEATSTD_NEW_PRESET_ARRAY acts line FEATSTD_NEW_ARRAY but uses the copy constructor of the array elements to
 *  initialize them all with preset.
 */
#define FEATSTD_NEW_PRESET_ARRAY(type, count, preset) \
    FeatStd::MemoryManagement::Internal::CreateArray<type >((static_cast<FeatStd::SizeType>(count)), (preset), FILE_LINE)

/**
 *  FEATSTD_DELETE_ARRAY deallocates objects allocated with FEATSTD_NEW_ARRAY. For each element instance the destructor
 *  is called.
 */
#define FEATSTD_DELETE_ARRAY(objectPtr) \
    FeatStd::MemoryManagement::DestroyArray(objectPtr)

/**
 *  FEATSTD_DELETE_ARRAY_T acts like FEATSTD_DELETE_ARRAY with exception that the type of the object to be destroyed
 *  can be explicitly defined in case FEATSTD_DELETE_ARRAY cannot determine the type correctly.
 */
#define FEATSTD_DELETE_ARRAY_T(objectPtr, type) \
    FeatStd::MemoryManagement::DestroyArray<type >(FeatStd::Internal::PointerToPointer<type*>(objectPtr))

/** FEATSTD_SAFE_DELETE_ARRAY acts like FEATSTD_DELETE_ARRAY, and additionally sets the given array pointer to 0. */
FEATSTD_LINT_MACRO_WHILEFALSE(FEATSTD_SAFE_DELETE_ARRAY)
#define FEATSTD_SAFE_DELETE_ARRAY(ptr) \
    do { \
        FEATSTD_DELETE_ARRAY(ptr); \
        ptr = 0; \
    FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4127, while (false) accepted) \
    } while (false)


/** Allocate given number of bytes from heap */
#define FEATSTD_ALLOC(nBytes) \
    FEATSTD_LINT_DYNAMIC_MEMORY() \
    FeatStd::MemoryManagement::Internal::PlainAlloc(nBytes, FILE_LINE)

/** Re-allocate the given buffer with the specified size */
#define FEATSTD_REALLOC(buffer, nBytes) \
    FEATSTD_LINT_DYNAMIC_MEMORY() \
    FeatStd::MemoryManagement::Internal::Realloc(buffer, nBytes, FILE_LINE)

/** free previously heap allocated buffer */
#define FEATSTD_FREE(buffer) \
    FEATSTD_LINT_DYNAMIC_MEMORY() \
    FeatStd::MemoryManagement::PlatformHeap::Free(buffer)


/// @}

#endif
#endif
