//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_MemoryManagement_Disposer_h)
    #define FeatStd_MemoryManagement_Disposer_h

#include <FeatStd/MemoryManagement/Heap.h>

/// @addtogroup FEATSTD_MEMORYMANAGEMENT
/// @{
namespace FeatStd {
    namespace MemoryManagement {

        /** @brief  Type definition for dispose function.
         *          Implementations of this method are intended to provide a transferable mechanism for object disposal.
         *          Dispose function implements the mechanism for disposal of objects, examples are "delete",
         *          "free", empty implementation in case of static objects, callbacks, etc.
         */
        template <typename T>
        class DisposerBase {
            public:
                typedef void(*DisposerFunction)(T object);
        };

        /**
         *   Generic disposer function which calls FEATSTD_DELETE for object disposal.
         */
        template <typename T>
        class Disposer : public DisposerBase <T> {
            public:
                static void Dispose(T object)
                {
                    FEATSTD_DELETE(object);
                }
        };

        /**
         *   Generic disposer function which calls FEATSTD_DELETE_ARRAY for object disposal.
         */
        template <typename T>
        class ArrayDisposer : public DisposerBase<T> {
            public:
                static void Dispose(T object)
                {
                    FEATSTD_DELETE_ARRAY(object);
                }
        };

        /**
         *  @brief  A native FeatStd Array Disposer which calls FEATSTD_DELETE_ARRAY in Dispose,
         *  but first casts.
         */
        template <typename TNeeded, typename TActual>
        class AdaptedArrayDisposer : public DisposerBase<TNeeded> {
            public:
                static void Dispose(TNeeded arrayAsTNeeded)
                {
                    FEATSTD_DELETE_ARRAY(reinterpret_cast<TActual>(arrayAsTNeeded));
                }
        };

        /**
         *   Generic disposer function which calls FEATSTD_FREE for object disposal.
         */
        template <typename T>
        class FreeMemoryDisposer : public DisposerBase <T> {
            public:
                static void Dispose(T object)
                {
                    typedef typename FeatStd::Internal::PureType<T>::Type CleanType;
                    FEATSTD_FREE(const_cast<CleanType*>(object));
                }
        };

        /**
         *  @brief Disposer for all object types that contain a static Create() and a (non-static) Dispose() function for
         *  life-time management. This type is suitable as value for TDisposer type parameter of ScopedPointer template.
         *  @param  T   Type of object which is referenced.
         */
        template <typename T>
        class DisposableObjectDisposer : public DisposerBase<T> {
            public:
                static void Dispose(T object) {
                    object->Dispose();
                }
        };

    }   // namespace MemoryManagement
}   // namespace FeatStd

/// @}
#endif  // FeatStd_MemoryManagement_Disposer_h
