//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_LifeTimeManager_h)
#define FeatStd_LifeTimeManager_h

#include <FeatStd/Util/LifeTimeManaged.h>
#include <FeatStd/Util/ConvenienceMacros.h>
#include <FeatStd/Platform/CriticalSection.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>

FEATSTD_LINT_FILE(967, LifeTimeManager.h, "false positive - include guards are present")

namespace FeatStd { namespace Internal {

    class FeatStdLocker;

    typedef void (*SubSystemDestructFn)();

    /** LifeTimeManager manages initialization and de-initialization of FeatStd library
        Certain FeatStd subsystems like StaticObjectManager or MemoryPool require proper
        initialization and de-initialization. LifeTimeManager takes care that these 
        subsystems will be initialized before main function is entered. Complementary
        de-initialization will be done when main function is exit. */ 
    class LifeTimeManager : private LifeTimeManaged<LifeTimeManager> {
        public:
            /** Register a dustruction function for a FeatStd subsystem
                Subsystems can register to be destroyed on with EndOfLifeTime of
                the LifeTimeManager. This event is guaranteed to happen outside
                of main function in a single threaded environment 
                @param DestructSubSys pointer to the destructor function of the subsystem */
            static void RegisterSubsystem(SubSystemDestructFn DestructSubSys);

            using LifeTimeManaged<LifeTimeManager>::StartLifeTime;

        private:
            static const UInt32 cSubSystemMax = 4;
            static bool mInit;
            static UInt8 mSubSystemCount;
            static SubSystemDestructFn mDestruct[cSubSystemMax];

            /** FeatStdLock is a multi purpose lock for FeatStd library and FEAT libraries using FeatStd
                @return a CriticalSection to bus used to synchronize multiple thread context */
            static CriticalSection& GetFeatStdLock();

            // required by LifeTimeManaged
            static void OnBeginOfLifeTime();

            // required by LifeTimeManaged
            static void OnEndOfLifeTime();

            // grant LifeTimeManager access to OnBeginOfLifeTime / OnBeginEndLifeTime
            friend class LifeTimeManaged<LifeTimeManager>;
            friend class FeatStdLocker;

            FEATSTD_MAKE_CLASS_STATIC(LifeTimeManager);
    };

    /** FeatStdLocker obtains a lock that can be used by multiple FeatStd facilities
        FeatStdLock shall not be used by other libraries then FeatStd. As this is a 
        multi purpose lock, unwanted synchronization of actually independent functions
        might apply. 
        Thus FeatStdLocker shall only be used for sporadic synchronization of functions 
        which do not justify the creation of a function specific lock. */
    class FeatStdLocker : public CriticalSectionLocker {
        public:
            FeatStdLocker() : CriticalSectionLocker(&LifeTimeManager::GetFeatStdLock()) {
            }
    };

}}

#endif
