//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Util/StaticObject.h"

#ifdef FEATSTD_SYSTEM_MEMORY_STATISTIC_ENABLED
#   include "Diagnostics/SystemMemoryStatistic.h"
#endif
#if defined(FEATSTD_LOG_ENABLED)
#   include <FeatStd/Diagnostics/Logger.h>
#endif
#if defined(FEATSTD_MEMORYPOOL_ENABLED)
#   include <FeatStd/MemoryPool/MemoryPoolManager.h>
#endif


namespace FeatStd { namespace Internal {

    bool LifeTimeManager::mInit = false;
    UInt8 LifeTimeManager::mSubSystemCount = 0;
    SubSystemDestructFn LifeTimeManager::mDestruct[LifeTimeManager::cSubSystemMax];

    // ------------------------------------------------------------------------
    CriticalSection& LifeTimeManager::GetFeatStdLock()
    {
        static CriticalSection cs;
        return cs;
    }


    // ------------------------------------------------------------------------
    void LifeTimeManager::RegisterSubsystem(SubSystemDestructFn DestructSubSys)
    {
        if (!mInit) {
            StartLifeTime();
        }

        // lock 
        FeatStdLocker lock;
        FEATSTD_DEBUG_ASSERT(mSubSystemCount < cSubSystemMax && DestructSubSys != 0);

        // check if the destruct function has been registered before
        bool ok = true;
        for (UInt32 i = 0; ok && (i < mSubSystemCount); ++i) {
            ok = mDestruct[i] != DestructSubSys;
        }

        // if not registered yet and all other conditions met, add to destruction array
        if (ok && (mSubSystemCount < cSubSystemMax) && (DestructSubSys != 0)) {
            mDestruct[mSubSystemCount] = DestructSubSys;
            ++mSubSystemCount;
        }
    }

    // ------------------------------------------------------------------------
    void LifeTimeManager::OnBeginOfLifeTime()
    {
        FEATSTD_DEBUG_ASSERT(!mInit);
        if (mInit) {
            return;
        }

        mInit = true;

#if defined(FEATSTD_MEMORYPOOL_ENABLED)
        FeatStd::MemoryManagement::MemoryPoolManager::Init();
#endif

#if defined(FEATSTD_LOG_ENABLED)
        // setup logger subsystem as early as possible
        mInit = FeatStd::Diagnostics::Internal::InitLogger();
#endif
        // insert here subsystems / components that require initialization prior main function

        FEATSTD_DEBUG_ASSERT(mInit);
    }
    // ------------------------------------------------------------------------
    void LifeTimeManager::OnEndOfLifeTime()
    {
        FEATSTD_DEBUG_ASSERT(mInit);
        if (!mInit) {
            return;
        }

        // no locking required as executed outside of main function
        while (mSubSystemCount > 0) {
            --mSubSystemCount;
            // invoke destruct on subsystem
            mDestruct[mSubSystemCount]();
        }

        FeatStd::StaticObjectManager::DestructStaticObjects();

#if defined(FEATSTD_MEMORYPOOL_ENABLED)
        // destruct static objects declared with FEATSTD_(UN)SYNCED_STATIC_OBJECT
        FeatStd::MemoryManagement::MemoryPoolManager::DestructPools();
#endif

#if defined(FEATSTD_LOG_ENABLED)
        // finally shut down the logger subsystem
        FeatStd::Diagnostics::Internal::ShutdownLogger();
#endif

        mInit = false;
    }

}}
