//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FEATSTD_EVENTSOURCE_H)
#define FEATSTD_EVENTSOURCE_H

#include <FeatStd/Container/SingleLinkedList.h>
#include <FeatStd/Platform/CriticalSection.h>
#include <FeatStd/Platform/Types.h>

namespace FeatStd {

    class Event;
    class EventListener;
/// @addtogroup FEATSTD_EVENT
/// @{

    /**
    * @brief Event Source for dispatching events.
    *
    * Events are dispatched via an EventSource to registered EventListeners.
    */
    class EventSource {
    public:
        /**
         *  Constructor
         */
        EventSource();

        /**
         *  Destructor
         */
        ~EventSource() {};

        /**
         *  Adds an EventListener to EventSource. Event listeners receive OnEvent calls.
         *  @param  eventListener  The event listener to be added.
         *  @return                True, if the listener could be added successfully. False, otherwise.
         */
        bool AddEventListener(EventListener* eventListener);

        /**
         *  Removes an EventListener from EventSource.
         *  @param  eventListener  The event listener to be removed.
         *  @param  waitForListenerRelease
         *  @return                True, if the listener could be removed successfully. False, otherwise.
         */
        bool RemoveEventListener(EventListener* eventListener, bool waitForListenerRelease = true);

        /**
         *  Dispatch the given event to all event listeners of the event source.
         *  @param  event  The event to be dispatched.
         */
        void DispatchEvent(const Event& event);

    protected:
    #ifdef FEATSTD_THREADSAFETY_ENABLED
        Internal::CriticalSection& GetCriticalSection() const;
    #endif

    private:
        typedef Internal::SingleLinkedList<EventListener*> EventListeners;

        void NotificationLoopBegin() { ++m_notificationDepth; }
        void NotificationLoopEnd();
        bool IsListenerRemovalAllowed() const { return (0 == m_notificationDepth); }

        EventListeners m_eventListeners;
        EventListeners m_pendingEventListeners;
        UInt m_notificationDepth;
    };
/// @}
}

#endif
