//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Diagnostics_Logger_h)
#define FeatStd_Diagnostics_Logger_h

#include <FeatStd/Base.h>
#ifndef FEATSTD_LOG_ENABLED
    #error Feature FEATSTD_LOG_ENABLED is not enabled. Do not include any FeatStd logging releated header files!
#endif

#include "LogLevel.h"
#include "Appender.h"
#include <FeatStd/Container/LinkedList.h>
#include <FeatStd/Platform/Ticks.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
    #include <FeatStd/Platform/CriticalSection.h>
#endif

FEATSTD_UNIT_TEST_CLASS_DECLARATION(LoggerTest)

namespace FeatStd { namespace Diagnostics {
/// @addtogroup FEATSTD_DIAGNOSTICS
/// @{
class LogControl;
struct LocationInfo;

/**
 * Central class for almost all log activities.
 * Most logging operations, except general configuration (/link LogControl /endlink),
 * are done through this class. <br>
 * One Logger instance is responsible for one realm (/link LogRealm /endlink)
 */
class Logger
{
    /// Typesafe list to hold Appenders.
    typedef FeatStd::Internal::LinkedList<Appender, &Appender::mAppenderListNode> AppenderList;
    friend class Appender;

public:
    /**
     *  Retrieves the default log level
     *  @return The default log level.
     */
    static LogLevel::Enum GetDefaultLogLevel();

    /**
     *  Sets the default log level
     *  @param logLevel The default log level.
     */
    static void SetDefaultLogLevel(LogLevel::Enum logLevel);

    /**
     *  Retrieves the default appender.
     *  @return The default appender.
     */
    static Appender* GetDefaultAppender();

    /**
     *  Constructor for a specified logRealmName
     *  @param logRealmName The log realm name that is set.
     */
    Logger(const Char* logRealmName);

    /**
     *  Destructor
     */
    ~Logger();

    /**
     *  Retrieves the realm name
     *  @return The current realm name.
     */
    const Char* GetRealmName() const {
        FEATSTD_DEBUG_ASSERT(mLogRealmName != 0);
        return mLogRealmName;
    }

    /**
     *  Retrieves the current LogLevel
     *  @return the current LogLeve.
     */
    LogLevel::Enum GetLevel() const { return mLogLevel; }

    FEATSTD_LINT_IGNORE_RETURN_VALUE(FeatStd::Diagnostics::Logger::SetLevel)
    /**
     *  Sets the current LogLevel
     *  @param logLevel The LogLevel that is set.
     *  @return The old LogLevel
     */
    LogLevel::Enum SetLevel(LogLevel::Enum logLevel);

    /**
     *  Retrieves the appender count.
     *  @return The current appender count.
     */
    UInt32 GetAppenderCount() const;

    /**
     *  Retrieves whether the logger contains the appender.
     *  @param appender The appender that is checked.
     *  @return True if the appender is contained, false otherwise.
     */
    static bool ContainsAppender(const Appender& appender);

    typedef FeatStd::Internal::LinkedListNode<Logger> LoggerListNode;

    /// Required public by LinkedList.
    LoggerListNode mLoggerListNode;

protected:
    /**
     *  Replaces current Appenders by appender.
     *  @remark Success can be checked using GetAppenderCount().
     *  @param appender The appender that is set.
     */
    static void SetAppender(Appender* appender);

    /**
     *  Appends appender to mAppenderList.
     *  @remark Success can be checked using GetAppenderCount().
     *  @param appender The appender that is appended.
     */
    static void AppendAppender(Appender* appender);

    /**
     *  Removes appender from mAppenderList.
     *  @remark Success can be checked using GetAppenderCount().
     *  @param appender The appender that is removed.
     */
    static void RemoveAppender(Appender* appender);

    //TODO switch to const as soon as SingleLinkedList::Contains becomes const
    static Logger::AppenderList& GetAppenderList();

    static bool ContainsLoggerWithRealmName(const Char* realmName);

public:
    /**
     *  Retrieves whether log requests is enabled for the given LogLevel
     *  @remark Inlined for performance reasons (see FEATSTD_LOG macros).
     *  @param logLevel The LogLevel that is checked.
     *  @return True if log request should be logged.
     */
    bool IsEnabledFor(LogLevel::Enum logLevel) const {
        return (logLevel >= mLogLevel);
    }

    /**
     *  Log output using given arguments (see FEATSTD macros).
     *  @remark Data preparation and sending the output to the Appenders is
     *          protected by a critical section! Hence there is no need to protect
     *          downstream processing (e.g. Appenders) by critical sections.
     *  @param logLevel The LogLevel that is set.
     *  @param info The info that is set.
     */
    FEATSTD_LINT_NEXT_EXPRESSION(1916, "exceptional allowance for the ellipsis in error handling and logging. Logging is not part of production code.")
    void ForcedLog(LogLevel::Enum logLevel, const LocationInfo&, const Char* info, ...) const;

private:
#ifdef FEATSTD_THREADSAFETY_ENABLED
    static FeatStd::Internal::CriticalSection& GetCriticalSection();
#endif

    static FeatStd::Internal::Ticks mLoggingStartTime;

    static LogLevel::Enum mDefaultLogLevel;

    static AppenderList& s_forceInitAppenderList;

    const Char*const mLogRealmName;
    LogLevel::Enum mLogLevel;

    FEATSTD_MAKE_CLASS_UNCOPYABLE(Logger);

    friend class LogControl;
    friend class ForceStaticInitHelper;

    FEATSTD_UNIT_TEST_CLASS_FRIEND(LoggerTest);
};

namespace Internal {
    /// Typesafe list to hold Loggers.
    typedef FeatStd::Internal::LinkedList<Logger, &Logger::mLoggerListNode> LoggerList;
    LoggerList& GetLoggerList();

    extern bool InitLogger();
    extern void ShutdownLogger();
}
/// @}
}}
#endif
