//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(FeatStd_Diagnostics_LogControl_h)
#define FeatStd_Diagnostics_LogControl_h

#if !defined(FEATSTD_IN_DIAGNOSTICS_LOG)
#error "please do not include this file directly, include FeatStd/Diagnostics/Log.h instead"
#endif

#include <FeatStd/Base.h>
#ifndef FEATSTD_LOG_ENABLED
    #error Feature FEATSTD_LOG_ENABLED is not enabled. Do not include any FeatStd logging releated header files!
#endif

#include "LogLevel.h"

#include "LoggerTyped.h"

namespace FeatStd { namespace Diagnostics {
/// @addtogroup FEATSTD_DIAGNOSTICS
/// @{
class Logger;
class Appender;

/**
 *  Main class for logging functionality.
 *  Holds one Logger instance for every LogRealm. <br>
 *  All Logger instances are configured to LogLevel::WARNING and
 *  are logging to the ConsoleAppender by default.
 */
class LogControl
{
public:
    typedef Internal::LoggerList LoggerList;

    /**
     *  Retrieves the default log level
     *  @return The default log level.
     */
    static LogLevel::Enum GetDefaultLogLevel() {
        return Logger::GetDefaultLogLevel();
    }

    /**
     *  Retrieves the default appender.
     *  @return The default appender.
     */
    static Appender* GetDefaultAppender() {
        return Logger::GetDefaultAppender();
    }

    /**
     *  Set configuration to default (according to CTOR).
     */
    // TODO Rename (ResetToSystemDefault) or extract functionality to LogTestHelper
    static void ResetToDefault();

    /**
     *  Returns list of Logger instances.
     *  @return The current list of logger instances.
     */
    static LoggerList& GetLoggerList();

    /**
     *  Returns Logger instance for the requested LogRealm.
     *  @return The Logger instance for the requested LogRealm.
     */
    template <class LogRealm>
    static Logger& GetLogger() {
        FEATSTD_DEBUG_ASSERT(LogRealm::Name() != 0);

        return LoggerTyped<LogRealm>::GetInstance();
    }

    /**
     *  Set LogLevel for all realms.
     *  Additionally the default LogLevel (class Logger) is set to level.
     *  @param level The LogLevel that is set.
     */
    static void SetLogLevel(LogLevel::Enum level);

    /**
     *  Set LogLevel for a single realm.
     *  @param level The LogLevel that is set.
     *  @return The old LogLevel.
     */
    FEATSTD_LINT_IGNORE_RETURN_VALUE(FeatStd::Diagnostics::LogControl::SetLogLevel)
    template <class LogRealm>
    static LogLevel::Enum SetLogLevel(LogLevel::Enum level) {
        FEATSTD_DEBUG_ASSERT(level >= LogLevel::All);
        FEATSTD_DEBUG_ASSERT(level <= LogLevel::Off);

        return GetLogger<LogRealm>().SetLevel(level);
    }

    /**
     *  Set appender for all Loggers.
     *  @param appender The appender that is set for all loggers.
     */
    static void SetAppender(Appender* appender);

    /**
     *  Append appender to all Loggers.
     *  @param appender The appender that is appended.
     */
    static void AppendAppender(Appender* appender);

    /**
     *  Remove appender from all Loggers.
     *  @param appender The appender that is removed.
     */
    static void RemoveAppender(Appender* appender);

private:
    FEATSTD_MAKE_CLASS_STATIC(LogControl);
};
/// @}
}}
#endif
