//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "LocationInfo.h"
#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Diagnostics/CodeChecks.h>

#include <FeatStd/Platform/String.h>

namespace FeatStd { namespace Diagnostics {
LocationInfo::LocationInfo(const Char* fileName, Int lineNumber, const Char* methodName)
  : mFullFileName(fileName), mFileName(ExtractFileName()), mLineNumber(lineNumber),
    mFullMethodName(methodName), mMethodName(ExtractMethodName())
{
    FEATSTD_DEBUG_ASSERT(mFullFileName != 0);
    FEATSTD_DEBUG_ASSERT(mFileName != 0);
    FEATSTD_DEBUG_ASSERT(mLineNumber > 0);
    FEATSTD_DEBUG_ASSERT(mFullMethodName != 0);
    FEATSTD_DEBUG_ASSERT(FeatStd::Internal::String::Length(mFullMethodName) > 0);
    FEATSTD_DEBUG_ASSERT(mMethodName != 0);
    FEATSTD_DEBUG_ASSERT(FeatStd::Internal::String::Length(mMethodName) > 0);
}

LocationInfo::~LocationInfo()
{
}

//! [FEATSTD_DEBUG_precond]
const Char* LocationInfo::ExtractFileName() const
{
    FEATSTD_GUARD(mFullFileName != 0) {

    FEATSTD_DEBUG_ASSERT(FeatStd::Internal::String::Length(mFullFileName) >= 4);
        //! [FEATSTD_DEBUG_precond]

        //! [FEATSTD_DEBUG_invariant]
    const SizeType lLength(FeatStd::Internal::String::Length(mFullFileName));
    const Char* lNameStart = mFullFileName + lLength;
    FEATSTD_DEBUG_ASSERT(lNameStart[0] == 0x00);

    FEATSTD_DEBUG_ASSERT(lNameStart > mFullFileName);
    while (lNameStart != mFullFileName) {
            //! [FEATSTD_DEBUG_invariant]
        if ((lNameStart[-1] == '\\') /* Windows */ ||
            (lNameStart[-1] == '/')  /* Unix */) {
            break;
        }

        lNameStart--;
    }

        //! [FEATSTD_DEBUG_postcond]
    //FEATSTD_DEBUG_ASSERT(lNameStart > mFullFileName);
    return lNameStart;
    }
    return 0;
}
//! [FEATSTD_DEBUG_postcond]

const Char* LocationInfo::ExtractMethodName() const
{
    FEATSTD_GUARD(mFullMethodName != 0) {

    FEATSTD_DEBUG_ASSERT(FeatStd::Internal::String::Length(mFullMethodName) >= 1);

    const Char *lLastColonPos = mFullMethodName + FeatStd::Internal::String::Length(mFullMethodName);
    while((lLastColonPos != mFullMethodName) && (*lLastColonPos != ':')) {
        --lLastColonPos;
    }
    if (*lLastColonPos != ':') {
        return mFullMethodName;
    }

    FEATSTD_DEBUG_ASSERT(lLastColonPos > mFullMethodName);
    FEATSTD_DEBUG_ASSERT(lLastColonPos < (mFullMethodName + FeatStd::Internal::String::Length(mFullMethodName)));
    FEATSTD_DEBUG_ASSERT(*(lLastColonPos - 1) == ':'); // Namespace/classname separator
    FEATSTD_DEBUG_ASSERT(*(lLastColonPos) == ':');
    FEATSTD_DEBUG_ASSERT(*(lLastColonPos + 1) != ':');
    return (lLastColonPos + 1);
    }
    return 0;
}
}}
