//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef FEATSTD_HASHTABLE_H
#define FEATSTD_HASHTABLE_H

#include <FeatStd/MemoryManagement/Heap.h>
#include <FeatStd/Platform/Math.h>
#include <FeatStd/Platform/Memory.h>
#include <FeatStd/Util/Traits.h>

namespace FeatStd
{

namespace Internal {

/**
 *  The default hash function to calculate an index for the hash table from the
 *  given key and size of the table. Note: the returned index must be in the range [0..tableSize)
 */
template<typename Key>
struct DefaultHashFunction {
    static SizeType CalculateHash(const Key& key, SizeType tableSize)
    {
        // The default hash function works well for any Key that is an integral type or a class
        // that delivers an integral via its SizeType() operator.
        // However floating point types can degenerate depending on their range being used. The
        // implicit conversion from a float/double Key to SizeType truncates its values. E.g.
        // keys in the range of (0..1) truncate to 0, which degenerates the hash table into a list.
        // The assert is here to let you know to implement your own hash function for floating
        // point types to take care of this issue.
        bool isFloatingPoint = Internal::IsFloatingPoint<Key>::Value;
        FEATSTD_DEBUG_ASSERT(!isFloatingPoint);
        FEATSTD_UNUSED(isFloatingPoint);

        // Calculate the hash/index.
        SizeType index = static_cast<SizeType>(static_cast<UInt>(key)); // invokes key's "operator UInt() const"
        index %= tableSize;
        static const Double s_hashMultiplier = 0.5 * (Math::SquareRoot(5.0) - 1.0);
        Double fraction = Math::FMod(s_hashMultiplier * index, 1.0);
        return static_cast<SizeType>(Math::Floor(static_cast<Double>(tableSize) * fraction));
    }
};

/// @addtogroup FEATSTD_CONTAINER
/// @{

/**
 * @brief  Implements a hash table.
 *
 *         The class Key is either native data or is class data that has the
 *         following member functions:
 *         Key::Key()
 *         Key& Key::operator= (const Key&)
 *         bool Key::operator== (const Key&) const
 *         bool Key::operator!= (const Key&) const
 *         Key::operator UInt() const
 *         The implicit conversion to UInt is used to select a hash table
 *         index for the Key. The return value does not need be within the range of
 *         hash table indices. DefaultHashFunction will use modular arithmetic to make
 *         this happen.
 *
 *         The class Value is either native data or is class data that has the
 *         following member functions:
 *         Value::Value()
 *         Value& Value::operator= (const Value&)
 *
 * @param Key           Key type of the hash table.
 * @param Value         Value type of the hash table.
 * @param HashFunction  The class that implements HashFunction::CalculateHash(). See
 *                      DefaultHashFunction for details.
 *
 */
template<typename Key, typename Value, typename HashFunction = DefaultHashFunction<Key> >
class HashTable
{
public:
    /**
     *  Constructor
     *  @param tableSize  The size of the hash table.
     */
    HashTable(SizeType tableSize = 256);

    /**
     *  Destructor
     */
    ~HashTable();

    /**
     *  Get the size of the hash table.
     *  @return  Get the size of the hash table.
     */
    inline SizeType GetTableSize() const { return m_tableSize; }

    /**
     *  Get the number of key-value pairs stored in the hash table.
     *  @return  The number of key-value pairs stored in the hash table.
     */
    inline SizeType GetKeyValuePairCount() const { return m_count; }

    /**
     *  Insert a key-value pair into the hash table.
     *  @param key    The key of the value to insert into the hash table.
     *  @param value  The value to insert into the hash table for the key.
     *  @return  True, if the operation was successful.
     *           False, if the key is already in the hash table, or if no hash table
     *           entry could be allocated for the key-value pair.
     */
    bool Insert(const Key& key, const Value& value);

    /**
     *  Retrieve the value for the given key, or 0 if the key does not exist.
     *  @param key  The key to retrieve the value for.
     *  @return  The value for the given key, or 0 if the key does not exist.
     */
    Value* Find(const Key& key) const;

    /**
     *  Remove the key-value pair from the hash table.
     *  @param key  The key of the key-value pair to remove.
     *  @return  True, if the key was found and removed. False, otherwise.
     */
    bool Remove(const Key& key);

    /**
     *  Remove all key-value pairs from the hash table.
     */
    void Clear();

    struct HashItem
    {
        Key key;
        Value value;
        HashItem* next;
    };

    /**
     * @brief Implements an iterator for linear traversal of the hash table
     * @param Key    Key type of the hash table.
     * @param Value  Value type of the hash table.
     * @code
     HashTable<Key, Value>::Iterator it(hashTable);
     // Iterate over the hash table and get the values.
     for (Value* value = it.GetFirst(); (0 != value); value = it.GetNext()) {
         cout << "Value: " << *value << endl;
     }
     // Iterate over the hash table and get the values and their keys
     Key key;
     for (Value* value = it.GetFirst(&key); (0 != value); value = it.GetNext(&key)) {
         cout << "Key: " << key << ", Value: " << *value << endl;
     }
     * @endcode
     */
    class Iterator
    {
    public:
        /**
         *  Construct an iterator of the given hash table.
         *  @param hashTable  The hash table to iterate.
         */
        Iterator(const HashTable& hashTable);

        /**
         *  Get the first value (and optionally its key) stored in the hash table.
         *  Note: It also initializes/resets the Iterator for subsequent GetNext() calls.
         *  @param key  A pointer to a variable to receive the key of the value, or 0.
         *  @return  The first value of the hash table, or 0 if the hash table is empty.
         */
        Value* GetFirst(Key* key = 0) const;

        /**
         *  Get the next value (and optionally its key) stored in the hash table.
         *  Note: GetFirst() needs to be called before any GetNext() calls to
         *        initialize/reset the iterator.
         *  @param key  A pointer to a variable to receive the key of the value, or 0.
         *  @return  The next value of the hash table, or 0 if the iteration over all
         *           values is finished.
         */
        Value* GetNext(Key* key = 0) const;

    private:
        Iterator();
        FEATSTD_MAKE_CLASS_UNCOPYABLE(Iterator);

        const HashTable& m_hashTable;
        mutable SizeType m_index;
        mutable HashItem* m_item;
    };

private:
    FEATSTD_MAKE_CLASS_UNCOPYABLE(HashTable);

    SizeType m_tableSize;
    SizeType m_count;
    HashItem** m_table;
};

template<typename Key, typename Value, typename HashFunction>
HashTable<Key, Value, HashFunction>::HashTable(SizeType tableSize)
    :
    m_tableSize(tableSize),
    m_count(0),
    m_table(0)
{
    m_table = FEATSTD_NEW_ARRAY(HashItem*, tableSize);
    FEATSTD_DEBUG_ASSERT(m_table);
    if (0 != m_table) {
        Memory::Set(m_table, 0, m_tableSize * sizeof(HashItem*));
    }
}

template<typename Key, typename Value, typename HashFunction>
HashTable<Key, Value, HashFunction>::~HashTable()
{
    Clear();
    FEATSTD_DELETE_ARRAY(m_table);
    m_table = 0;
}

template<typename Key, typename Value, typename HashFunction>
bool HashTable<Key, Value, HashFunction>::Insert(const Key& key, const Value& value)
{
    // Find the hash table entry for the given key.
    SizeType index = HashFunction::CalculateHash(key, m_tableSize);
    FEATSTD_DEBUG_ASSERT(index < m_tableSize);
    HashItem* item = m_table[index];

    // Search for the item in the list associated with the key.
    while (0 != item) {
        if (key == item->key) {
            // The item is already in the hash table.
            return false;
        }

        item = item->next;
    }

    // Add the item to the beginning of the list.
    item = FEATSTD_NEW(HashItem);
    if (0 == item) {
        return false;
    }

    item->key = key;
    item->value = value;
    item->next = m_table[index];
    m_table[index] = item;
    m_count++;

    return true;
}

template<typename Key, typename Value, typename HashFunction>
Value* HashTable<Key, Value, HashFunction>::Find(const Key& key) const
{
    // Find the hash table entry for given key.
    SizeType index = HashFunction::CalculateHash(key, m_tableSize);
    FEATSTD_DEBUG_ASSERT(index < m_tableSize);
    HashItem* item = m_table[index];

    // Search for the item in the list associated with the key.
    while (0 != item) {
        if (key == item->key) {
            // The item is in the hash table.
            return &item->value;
        }

        item = item->next;
    }

    return 0;
}

template<typename Key, typename Value, typename HashFunction>
bool HashTable<Key, Value, HashFunction>::Remove(const Key& key)
{
    // Find the hash table entry for the given key.
    SizeType index = HashFunction::CalculateHash(key, m_tableSize);
    FEATSTD_DEBUG_ASSERT(index < m_tableSize);
    HashItem* item = m_table[index];

    if (0 == item) {
        return false;
    }

    if (key == item->key) {
        // The item is at the front of the list, remove it.
        HashItem* save = item;
        m_table[index] = item->next;
        FEATSTD_DELETE(save);
        --m_count;
        return true;
    }

    // Search for the item in the list.
    HashItem* previousItem = item;
    HashItem* currentItem = item->next;
    while ((0 != currentItem) && (key != currentItem->key)) {
        previousItem = currentItem;
        currentItem = currentItem->next;
    }

    if (0 != currentItem) {
        // Found the item in the list, remove it.
        previousItem->next = currentItem->next;
        FEATSTD_DELETE(currentItem);
        --m_count;
        return true;
    }

    return false;
}

template<typename Key, typename Value, typename HashFunction>
void HashTable<Key, Value, HashFunction>::Clear()
{
    if (m_count > 0) {
        for (SizeType i = 0; i < m_tableSize; ++i) {
            while (0 != m_table[i]) {
                HashItem* save = m_table[i];
                m_table[i] = m_table[i]->next;
                FEATSTD_DELETE(save);
                if (--m_count == 0) {
                    return;
                }
            }
        }
    }
}

template<typename Key, typename Value, typename HashFunction>
HashTable<Key, Value, HashFunction>::Iterator::Iterator(const HashTable& hashTable)
    :
    m_hashTable(hashTable),
    m_index(0),
    m_item(0)
{
}

template<typename Key, typename Value, typename HashFunction>
Value* HashTable<Key, Value, HashFunction>::Iterator::GetFirst(Key* key) const
{
    if (m_hashTable.m_count > 0) {
        // Find the first entry in the hash table.
        for (m_index = 0; m_index < m_hashTable.m_tableSize; ++m_index) {
            if (0 != m_hashTable.m_table[m_index]) {
                m_item = m_hashTable.m_table[m_index];
                if (0 != key) {
                    *key = m_item->key;
                }

                return &m_item->value;
            }
        }
    }

    return 0;
}

template<typename Key, typename Value, typename HashFunction>
Value* HashTable<Key, Value, HashFunction>::Iterator::GetNext(Key* key) const
{
    if (m_hashTable.m_count > 0) {
        FEATSTD_DEBUG_ASSERT(0 != m_item); // Assert that GetFirst() was called before.

        // Get the next item in the list.
        m_item = m_item->next;
        if (0 != m_item) {
            if (0 != key) {
                *key = m_item->key;
            }

            return &m_item->value;
        }

        // Find the next entry in the hash table.
        for (++m_index; m_index < m_hashTable.m_tableSize; ++m_index) {
            if (0 != m_hashTable.m_table[m_index]) {
                m_item = m_hashTable.m_table[m_index];
                if (0 != key) {
                    *key = m_item->key;
                }

                return &m_item->value;
            }
        }
    }

    return 0;
}

/// @}

}

}

#endif
