//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AsyncRequestDispatcher.h"

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace FeatStd {

FEATSTD_LOG_SET_REALM(FeatStd::Diagnostics::LogRealm::FeatStdAsync);

#ifdef FEATSTD_THREADSAFETY_ENABLED
    static Internal::CriticalSection& AsyncRequestDispatcherCriticalSection()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(Internal::CriticalSection, s_asyncRequestDispatcherCriticalSection);
        return s_asyncRequestDispatcherCriticalSection;
    }
    static Internal::CriticalSection& s_forceInitAsyncRequestDispatchereCriticalSection = AsyncRequestDispatcherCriticalSection();

    Internal::CriticalSection& AsyncRequestDispatcher::GetCriticalSection() const
    {
        return AsyncRequestDispatcherCriticalSection();
    }
#endif

// ------------------------------------------------------------------------
AsyncRequestDispatcher::AsyncRequestDispatcher(bool isMultithreaded, FeatStd::UInt32 additionalPriorities) :
    m_running(false),
    m_isMultithreaded(isMultithreaded)
{
    static_cast<void>(m_requestQueues.Reserve(static_cast<SizeType>(additionalPriorities)+1));
    for (UInt32 i = 0; i <= additionalPriorities; ++i) {
        static_cast<void>(m_requestQueues.Add(RequestQueue()));
    }

    m_isRequestSemaphoreCreated = m_requestSemaphore.Create();
    if (!m_isRequestSemaphoreCreated) {
        FEATSTD_LOG_ERROR("Failed to create semaphore.\n");
    }

    m_isExecutedFinishedSemaphoreCreated = m_executedFinishedSemaphore.Create();
    if (!m_isExecutedFinishedSemaphoreCreated) {
        FEATSTD_LOG_ERROR("Failed to create semaphore.\n");
    }

    m_isIdleSemaphoreCreated = m_idleSemaphore.Create();
    if (!m_isIdleSemaphoreCreated) {
        FEATSTD_LOG_ERROR("Failed to create semaphore.\n");
    }
}

// ------------------------------------------------------------------------
AsyncRequestDispatcher::~AsyncRequestDispatcher()
{
    if (m_isRequestSemaphoreCreated) {
        static_cast<void>(m_requestSemaphore.Destroy());
    }

    if (m_isExecutedFinishedSemaphoreCreated) {
        static_cast<void>(m_executedFinishedSemaphore.Destroy());
    }

    if (m_isIdleSemaphoreCreated) {
        static_cast<void>(m_idleSemaphore.Destroy());
    }
}

// ------------------------------------------------------------------------
void AsyncRequestDispatcher::Schedule(const AsyncRequestBase::SharedPointer& request, FeatStd::UInt32 priority)
{
    if (!request.PointsToNull()) {
        PushRequest(request, priority);
        static_cast<void>(SignalRequest());
    }
}

// ------------------------------------------------------------------------
void AsyncRequestDispatcher::Stop()
{
    // TODO: abort all requests if single threaded
    if (SetRunning(false)) {
        static_cast<void>(SignalRequest());
    }
}

// ------------------------------------------------------------------------
bool AsyncRequestDispatcher::Flush()
{
    // TODO: execute all requests until all are completed if single threaded
    if (m_isMultithreaded) {
        if (m_isIdleSemaphoreCreated) {
            return m_idleSemaphore.Obtain();
        }
        else {
            return false;
        }
    }
    return true;
}

// ------------------------------------------------------------------------
bool AsyncRequestDispatcher::IsRunning() const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    return m_running;
}

// ------------------------------------------------------------------------
bool AsyncRequestDispatcher::WaitForNextExecuteFinished()
{
    // TODO: execute next request if single
    if (m_isMultithreaded) {
        if (m_isExecutedFinishedSemaphoreCreated) {
            return m_executedFinishedSemaphore.Obtain();
        }
        else {
            return false;
        }
    }
    return true;
}

// ------------------------------------------------------------------------
void AsyncRequestDispatcher::Start()
{
    static_cast<void>(SetRunning(true));
    if (m_isMultithreaded) {
        while (IsRunning()) {
            DispatchNext(true);
        }
        AsyncRequestBase::SharedPointer request;
        while (PopRequest(request)) {
            (void)request->Abort();
        }
    }
}

bool AsyncRequestDispatcher::IsDispatcherEmpty() const
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    bool result = true;
    for (FeatStd::Int i = static_cast<Int>(m_requestQueues.Size()) - 1; i >= 0; --i) {
        if ((!m_requestQueues[i].m_nextRequest.PointsToNull()) ||
            (!m_requestQueues[i].m_lastRequest.PointsToNull())) {
            result = false;
        }
    }
    return result;
}

// ------------------------------------------------------------------------
void AsyncRequestDispatcher::DispatchNext(bool waitForNextRequest)
{
    AsyncRequestBase::SharedPointer request;
    while ((!PopRequest(request)) && IsRunning() && waitForNextRequest) {
        static_cast<void>(WaitForNextRequest());
    }
    if (!request.PointsToNull()) {
        request->Execute();
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker lockRequest(&request->GetCriticalSection());
#endif
        request->m_dispatcher = 0;
        if (request->m_reschedule) {
            request->m_reschedule = false;
            if (request->m_abort) {
                request->m_state = AsyncRequestBase::Aborted;
#ifdef FEATSTD_THREADSAFETY_ENABLED
                lockRequest.Release();
#endif
                static_cast<void>(request->SignalFinished());
            }
            else {
                request->m_dispatcher = this;
#ifdef FEATSTD_THREADSAFETY_ENABLED
                lockRequest.Release();
#endif
                Schedule(request, request->m_originalPriority);
            }
        }
        else {
            request->m_state = AsyncRequestBase::Completed;
#ifdef FEATSTD_THREADSAFETY_ENABLED
            lockRequest.Release();
#endif
            static_cast<void>(request->Finished());
        }
        static_cast<void>(SignalExecuteFinished());
    }
}

// ------------------------------------------------------------------------
bool AsyncRequestDispatcher::SetRunning(bool running)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    if (m_running != running) {
        m_running = running;
        return true;
    }
    return false;
}

// ------------------------------------------------------------------------
void AsyncRequestDispatcher::PushRequest(const AsyncRequestBase::SharedPointer& request, FeatStd::UInt32 priority)
{
    if (!request.PointsToNull()) {
        request->m_originalPriority = priority;
#ifdef FEATSTD_THREADSAFETY_ENABLED
        Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
        FeatStd::UInt32 lPriority = LimitPriority(priority);
        if (FeatStd::UInt32(m_requestQueues.Size()) > lPriority) {
            RequestQueue& requestQueue = m_requestQueues[LimitPriority(lPriority)];
            request->m_dispatcher = this;
            if (requestQueue.m_lastRequest.PointsToNull()) {
                requestQueue.m_nextRequest = request;
            }
            else {
                requestQueue.m_lastRequest->m_nextRequest = request;
            }
            requestQueue.m_lastRequest = request;
        }
    }
}

// ------------------------------------------------------------------------
bool AsyncRequestDispatcher::PopRequest(AsyncRequestBase::SharedPointer& request)
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif
    for (FeatStd::Int i = static_cast<Int>(m_requestQueues.Size()) - 1; i >= 0; --i) {
        RequestQueue& requestQueue = m_requestQueues[i];
        if (!requestQueue.m_nextRequest.PointsToNull()) {
            request = requestQueue.m_nextRequest;
            requestQueue.m_nextRequest = requestQueue.m_nextRequest->m_nextRequest;
            request->m_nextRequest = AsyncRequestBase::SharedPointer();
            if (requestQueue.m_nextRequest.PointsToNull()) {
                requestQueue.m_lastRequest = AsyncRequestBase::SharedPointer();
            }
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool AsyncRequestDispatcher::SignalRequest()
{
    if (m_isRequestSemaphoreCreated) {
        return m_requestSemaphore.Release();
    }
    return false;
}

// ------------------------------------------------------------------------
bool AsyncRequestDispatcher::SignalIdle()
{
    if (m_isIdleSemaphoreCreated) {
        return m_idleSemaphore.Release();
    }
    return false;
}

// ------------------------------------------------------------------------
bool AsyncRequestDispatcher::SignalExecuteFinished()
{
    if (m_isExecutedFinishedSemaphoreCreated) {
        return m_executedFinishedSemaphore.Release();
    }
    return false;
}

// ------------------------------------------------------------------------
bool AsyncRequestDispatcher::WaitForNextRequest()
{
    if (m_isRequestSemaphoreCreated) {
        return m_requestSemaphore.Obtain();
    }
    return false;
}

// ------------------------------------------------------------------------
FeatStd::UInt32 AsyncRequestDispatcher::LimitPriority(FeatStd::UInt32 priority) const
{
    if (m_requestQueues.Size() <= 0) {
        return 0;
    }
    if (priority + 1 >= FeatStd::UInt32(m_requestQueues.Size()))
    {
        return static_cast<UInt32>(m_requestQueues.Size()) - 1;
    }
    return priority;
}

AsyncRequestDispatcherWorkerThread::AsyncRequestDispatcherWorkerThread(FeatStd::UInt32 additionalPriorities)
    :
    m_dispatcher(true, additionalPriorities),
    m_threadId(FeatStd::UInt32(-1))
{
    m_isStopSemaphoreCreated = m_stopSemaphore.Create();
    if (!m_isStopSemaphoreCreated) {
        FEATSTD_LOG_ERROR("Failed to create semaphore.\n");
    }
}

}
