//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_ViewSceneTraverser_h)
#define Courier_Visualization_ViewSceneTraverser_h

#include <Courier/Util/Util.h>
#include <Courier/Visualization/Renderer.h>

#if defined(CANDERA_3D_ENABLED)
    #include <Candera/Engine3D/Core/Node.h>
#endif
#if defined(CANDERA_2D_ENABLED)
    #include <Candera/Engine2D/Core/Node2D.h>
#endif

namespace Candera {
    class SceneContext;
    class Scene2DContext;
}

namespace Courier {

/// @addtogroup COURIER_VISUALIZATION
/// @{
/** The ViewSceneTraverser is a template used for traversion over the scene tree and will be used for uploading purposes.
*/
template< typename T, typename SC > class ViewSceneTraverser
{
    public:
        ///
        enum TraverserAction {
            /// Traversing continues with the next child (if the current node has any) or sibling.
            ProceedTraversing,
            /// Traversing will stop for all child nodes and will continue with the next sibling.
            StopTraversingForDescendants,
            /// Traversing will stop here.
            StopTraversing,
            /// Stops but allow continuing.
            BreakTraversing
        };

        ///
        ViewSceneTraverser() : mRootNode(0), mCurrentNode(0),  mNextChild(0), mSceneContext(0),
                               mAction(ProceedTraversing), mAllChildrenOfRootProcessed(false), mIsEffectiveResultSuccessful(true)
        { }

        ///
        virtual ~ViewSceneTraverser() {
            mRootNode = 0;
            mCurrentNode = 0;
            mNextChild = 0;
            mSceneContext = 0;
        }

        /// Indicates if all nodes have been processed.
        virtual bool Finished() const { return mAllChildrenOfRootProcessed || (mAction == StopTraversing); }

        /// This method indicates if an error occured.
        virtual bool IsEffectiveResultSuccessful() const { return mIsEffectiveResultSuccessful; }

        /// Returns the SceneContext of the Scene currently uploading, this allows implementing Scene specific Uploading.
        SC * GetSceneContext() { return mSceneContext; }

        /// Has to be called when starting the traversing. Finished has to be checked if traversing is already finished.
        /**
        *   @param root
        *   @param sceneContext
        *   @param scopeMask
        */
        void InitialTraverse(T& root, SC& sceneContext, const Candera::ScopeMask &scopeMask = Candera::ScopeMask())
        {
            mRootNode = &root;
            mCurrentNode = &root;
            mSceneContext = &sceneContext;
            mAllChildrenOfRootProcessed = false;
            mNextChild = mCurrentNode->GetFirstChild();
            mAction = ProceedTraversing;
            mScopeMask = scopeMask;
            Traverse();
        }

        ///
        void Reset()
        {
            mCurrentNode = mRootNode;
            mAllChildrenOfRootProcessed = false;
            mNextChild = mRootNode->GetFirstChild();
            mAction = ProceedTraversing;
        }

        ///
        void Reset(const Candera::ScopeMask &scopeMask)
        {
            Reset();
            mScopeMask = scopeMask;
        }

        /// Has to be called until method Finished returns true.
        void Traverse()
        {
            while ((!mAllChildrenOfRootProcessed) && (mAction != StopTraversing)) {
                mAction = ProceedTraversing;
                // Process the current node before its children.
                if (mNextChild == mCurrentNode->GetFirstChild()) {
                    // Const cast: see function comment.
                    mAction = ProcessNode(const_cast<T&>(*mCurrentNode));
                }

                // Process the children (if traversal has not been stopped by ProcessNode()'s return at the parent).
                if (mAction != StopTraversing) {
                    const bool allChildrenAlreadyProcessed = (mNextChild == 0);

                    if (allChildrenAlreadyProcessed || mAction == StopTraversingForDescendants) {
                        // Up to parent, with next of parent's children (== mCurrentNode's next sibling).
                        if (mCurrentNode == mRootNode) {
                            mAllChildrenOfRootProcessed = true;
                        }
                        else {
                            mNextChild = mCurrentNode->GetNextSibling();
                        }
                        mCurrentNode = mCurrentNode->GetParent();
                    }
                    else {
                        // down to child
                        mCurrentNode = mNextChild;
                        mNextChild = mCurrentNode->GetFirstChild();
                    }
                }

                if(mAction == BreakTraversing) {
                    return;
                }
            }
        }

    protected:
        ///
        virtual TraverserAction ProcessNode(T& /*node*/) { return StopTraversing; }

    protected:
        ///
        const T * mRootNode;
        ///
        const T * mCurrentNode;
        ///
        const T * mNextChild;
        ///
        SC * mSceneContext;
        ///
        TraverserAction mAction;
        ///
        bool mAllChildrenOfRootProcessed;
        ///
        bool mIsEffectiveResultSuccessful;
        ///
        Candera::ScopeMask mScopeMask;
};
/// @}

// ------------------------------------------------------------------------
#if defined(CANDERA_3D_ENABLED)

/// @addtogroup COURIER_VISUALIZATION
/// @{
/** The UploadNodeTraverser is a specialized ViewSceneTraverser which is used for traversing 3D nodes of a 3D scene while using async uploading.
    This default implementation traverses the Node tree step by step each cycle. That means method Traverse and therefore ProcessNode is called each cycle
    until all Upload methods has been called and no error has occured.
    If this is not a suitable strategy an own Traverser may be derived from UploadNodeTraverser and the method ProcessNode shall be overwritten.
    In case the method ProcessNode returns ProceedTraversing it will be called again in the same cycle until it returns BreakTraversing.
    This allows using an own application dependent (even scene dependent) implementation.
    An instance of the derived UploadNodeTraverser has to be registered on startup using the method Courier::ViewScene3D::UploadNodeTraverserSet.
    If not doing so the default Traverser will be used.
*/
class UploadNodeTraverser: public ViewSceneTraverser< Candera::Node , Candera::SceneContext >
{
    typedef ViewSceneTraverser< Candera::Node , Candera::SceneContext > Base;

    public:
        ///
        UploadNodeTraverser() : Base() {}

    protected:
        /** The default implementation is uploading one node each time Traverse is called by the AsyncLoading mechanism.
            This behaviour maybe overwritten with an own strategy.
        *   @param node
        *   @return TraverserAction
        */
        virtual TraverserAction ProcessNode(Candera::Node & node)
        {
            if (node.IsInScopeOf(mScopeMask))
            {
                if (!node.Upload()) {
                    mIsEffectiveResultSuccessful = false;
                    return StopTraversing;
                }
            }
            return BreakTraversing;
        }
};
/// @}

#endif

// ------------------------------------------------------------------------
#if defined(CANDERA_2D_ENABLED)

/// @addtogroup COURIER_VISUALIZATION
/// @{
/** The UploadNode2DTraverser is a specialized ViewSceneTraverser which is used for traversing 2D nodes of a 2D scene while using async uploading.
    This default implementation traverses the Node2D tree step by step each cycle. That means method Traverse and therefore ProcessNode is called each cycle
    until all Upload methods has been called and no error has occured.
    If this is not a suitable strategy an own Traverser may be derived from UploadNode2DTraverser and the method ProcessNode shall be overwritten.
    In case the method ProcessNode returns ProceedTraversing it will be called again in the same cycle until it returns BreakTraversing.
    This allows using an own application dependent (even scene dependent) implementation.
    An instance of the derived UploadNode2DTraverser has to be registered on startup using the method Courier::ViewScene2D::UploadNode2DTraverserSet.
    If not doing so the default Traverser will be used.
*/
class UploadNode2DTraverser: public ViewSceneTraverser< Candera::Node2D , Candera::Scene2DContext >
{
    typedef ViewSceneTraverser< Candera::Node2D , Candera::Scene2DContext > Base;

    public:
        ///
        UploadNode2DTraverser() : Base() {}

    protected:
        /** The default implementation is uploading one node each time Traverse is called by the AsyncLoading mechanism.
            This behaviour maybe overwritten with an own strategy.
        *   @param node
        *   @return TraverserAction
        */
        virtual TraverserAction ProcessNode(Candera::Node2D & node) {
            if(! node.Upload(mScopeMask)) {
                mIsEffectiveResultSuccessful = false;
                return StopTraversing;
            }
            return BreakTraversing;
        }
};
/// @}

#endif

}

#endif
