//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ViewScene3D.h"

#if defined(CANDERA_3D_ENABLED)

#include "IViewHandler.h"
#include "ViewVisitor.h"
#include "RenderHint.h"
#include "Renderer.h"
#include "VisualizationMsgs.h"
#include <CanderaWidget/Widget3D/Widget.h>

#include <Candera/Engine3D/Core/CompositeGroup.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

#if !defined(COURIER_DEFAULT_VIEW3D_CAMERA_COUNT)
    static const Int cCOURIER_DEFAULT_VIEW3D_CAMERA_COUNT = 2;
#else
    static const Int cCOURIER_DEFAULT_VIEW3D_CAMERA_COUNT = COURIER_DEFAULT_VIEW3D_CAMERA_COUNT;
#endif

using namespace Candera;
using namespace Internal;

// ------------------------------------------------------------------------

ViewScene3D::GetTouchedWidget3DFct ViewScene3D::mGetTouchedWidget3DFunction =  ViewScene3D::GetTouchedWidget3D;

// ------------------------------------------------------------------------
ViewScene3D::ViewScene3D(bool managed) :
    Base(managed),
    mScene3DContext(0),
    mBuildState(Candera::ContentLoader::Error),
    mPartialLoadState(SynchronousLoadState),
    mPartialLoadCalls(0),
    mCurrentUploadNodeTraverser(0),
    mClearingData3DVct()
{
    if(! mCameras.Reserve(cCOURIER_DEFAULT_VIEW3D_CAMERA_COUNT)) {
        FEATSTD_PANIC("Can't reserve memory for camera pointers");
    }
}

// ------------------------------------------------------------------------
ViewScene3D::~ViewScene3D()
{
    CleanUp();
    SetParentView(0);
}

// ------------------------------------------------------------------------
void ViewScene3D::FillWidgetList()
{
    if (mScene3DContext!=0) {
        AddWidgets(mScene3DContext->GetWidgetIterator());
        AddWidgetTraverser3D addWidgetTraverer(this);
        addWidgetTraverer.Traverse(*mScene3DContext->GetScene());
    }
}

// ------------------------------------------------------------------------
class Camera3DLessBySequenceNumber
{
    public:
        bool operator()(const Camera * a, const Camera * b) const {
            return a->GetSequenceNumber() < b->GetSequenceNumber();
        }
};

// ------------------------------------------------------------------------
void ViewScene3D::InitCameras()
{
    if (0 != mScene3DContext) {
        mCameras.Clear();
        mRefCounterVct.Clear();
        DetectCamera(mScene3DContext->GetScene());
        FEATSTD_LINT_NEXT_EXPRESSION(1502, "no static members, thats okay");
        static Camera3DLessBySequenceNumber objSort;
        mCameras.Sort(objSort);
        for(SizeType idx=0; idx<mCameras.Size(); idx++) {
            if (0 != mCameras[idx]) {
                static_cast<void>(mRefCounterVct.Add(CameraRefCounter(mCameras[idx])));
                if((mCameras[idx]->GetCameraRenderStrategy()!=0) && (GetViewHandler()->GetRenderer()!=0)) {
                    Gdu * gdu = GetViewHandler()->GetRenderer()->GetGdu(mCameras[idx]->GetRenderTarget());
                    if(gdu!=0) {
                        gdu->UsesCameraWithRenderStrategy();
                    }
                }
            }
        }
    }
}

// ------------------------------------------------------------------------
static bool IsQualifiedNameEqual(const Char* qualifiedName, FeatStd::SizeType qualifiedNameLength, Node* node)
{
    Node* current = node;
    FeatStd::OffsetType strCurrentIdx = qualifiedNameLength;
    while (0 != current) {
        const Char* name = current->GetName();
        FeatStd::SizeType nameLength = StringPlatform::Length(name);
        strCurrentIdx -= static_cast<FeatStd::OffsetType>(nameLength);
        const Char* compareName = &(qualifiedName[strCurrentIdx]);
        if (strCurrentIdx < 0) {
            // incorrect camera length detected;
            return false;
        }
        if (FeatStd::Internal::String::ComparePartial(compareName, name, nameLength) != 0) {
            // incorrect name segments
            return false;
        }
        Node* parent = current->GetParent();
        if (0 != parent) {
            // iterate to delimiter '@'
            --strCurrentIdx;
            if (strCurrentIdx < 0) {
                // incorrect camera length detected
                return false;
            }
            if (qualifiedName[strCurrentIdx] != '@') {
                // incorrect name
                return false;
            }
        }
            // so far so good. name is a match and delimiter is in the right place proceed with parent
            current = parent;
        }
    return true;
}

// ------------------------------------------------------------------------
Candera::Camera * ViewScene3D::FindCamera3D(const Char * cameraName)
{
    SizeType idx;
    // for optimizing reasons compare the pointers of strings
    for(idx=0; idx<mCameras.Size(); idx++) {
        Camera * camera = mCameras[idx];
        if(camera->GetName()==cameraName) {
            return camera;
        }
    }
    // if not found then make string compares
    for(idx=0; idx<mCameras.Size(); idx++) {
        Camera * camera = mCameras[idx];
        if(0==FeatStd::Internal::String::CompareStrings(camera->GetName(), cameraName)) {
            return camera;
        }
    }
    // if not found  then compare the whole full names
    for(idx=0; idx<mCameras.Size(); idx++) {
        Camera * camera = mCameras[idx];
        if (IsQualifiedNameEqual(cameraName, StringPlatform::Length(cameraName), camera)) {
            return camera;
        }
    }
    return 0;
}

Candera::Camera* ViewScene3D::FindCamera3D(const ItemId& id)
{
    Camera **pCamera = mIdToCamera.Find(id);
    if (pCamera != 0)
    {
        return (*pCamera);
    }

    // Search by name and update maps
    Camera *camera = FindCamera3D(id.CStr());
    if (camera != 0)
    {
        static_cast<void>(mIdToCamera.Insert(id, camera));
        static_cast<void>(mCameraToId.Insert(camera, id));
    }

    return camera;
}

// ------------------------------------------------------------------------
bool ViewScene3D::GetRenderTargetPtrVector(RTPtrVector & vector)
{
    return ViewScene3D::GetRenderTargetPtrVector(vector,true);
}

// ------------------------------------------------------------------------
bool ViewScene3D::GetRenderTargetPtrVector(RTPtrVector & vector, bool clear)
{
    if(clear) {
        vector.Clear();
    }
    bool rc = true;
    for(SizeType idx=0; idx<mCameras.Size(); idx++) {
        Camera * camera = mCameras[idx];
        rc = View::AddRenderTarget(vector,camera->GetRenderTarget()) && rc;
    }
    return rc;
}

// ------------------------------------------------------------------------
void ViewScene3D::Invalidate(UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if (!IsAlreadyInvalidated(ItemId(), dirtyArea)) {
        // invalidate all invalidation dependencies.
        ViewScene::InvalidateImpl(ItemId(), dirtyArea);
        for (SizeType idx = 0; idx < mCameras.Size(); ++idx) {
            Camera * camera = mCameras[idx];
            if (camera->IsRenderingEnabled()) {
                GetViewHandler()->GetRenderer()->Queue3DJob(this, camera, renderCounter, dirtyArea, false);
            }
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene3D::InvalidateOffscreenHelper(const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if (!IsAlreadyInvalidated(ItemId(), dirtyArea)) {
        // invalidate all invalidation dependencies.
        ViewScene::InvalidateImpl(ItemId(), dirtyArea);
        for (SizeType idx = 0; idx < mCameras.Size(); ++idx) {
            Camera * camera = mCameras[idx];
            if (camera->IsRenderingEnabled()) {
                GetViewHandler()->GetRenderer()->Queue3DJob(this, camera, cDefaultRender3DCounter, dirtyArea, false, true);
            }
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene3D::Invalidate(Candera::Camera * camera, UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    // invalidate all invalidation dependencies.
    ItemId *pCameraId = mCameraToId.Find(camera);
    ViewScene::InvalidateImpl(
        (pCameraId != 0) ? (*pCameraId) : ItemId(camera->GetName()),
        dirtyArea);
    if(camera->IsRenderingEnabled()) {
        GetViewHandler()->GetRenderer()->Queue3DJob(this, camera, renderCounter, dirtyArea, false);
    }
}


// ------------------------------------------------------------------------
void ViewScene3D::Invalidate(RenderTarget * renderTarget, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    // invalidate all invalidation dependencies.
    ViewScene::InvalidateImpl(ItemId(), dirtyArea);
    Renderer * renderer = GetViewHandler()->GetRenderer();
    if ((0 != renderer) && IsContentLoaded()) {
        for (SizeType idx = 0; idx<mCameras.Size(); ++idx) {
            Camera * camera = mCameras[idx];
            if (camera->IsRenderingEnabled() && (camera->GetRenderTarget() == renderTarget)) {
                renderer->Queue3DJob(this, camera, cDefaultRender3DCounter, dirtyArea, false);
            }
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene3D::DetectCamera(Node * node)
{
    FEATSTD_DEBUG_ASSERT(node!=0);
    if (0 != node) {
        if (ViewScene::GetDetectViewReleationShip()) {
            MemoryManagement::SharedPointer<Appearance> appearance = node->GetAppearance();
            if (!appearance.PointsToNull()) {
                MemoryManagement::SharedPointer<Texture> texture = appearance->GetTexture();
                if (!texture.PointsToNull()) {
                    MemoryManagement::SharedPointer<TextureImage> textureImage = texture->GetTextureImage();
                    if (!textureImage.PointsToNull()) {
                        ChangeViewSurfaceRelation(AddViewRelation, textureImage->ToImageSource3D());
                    }
                }
            }
        }
        if (node->IsTypeOf(Camera::GetTypeId())) {
            (void)mCameras.Add(static_cast<Camera*>(node));
        }
            for (Node* child = node->GetFirstChild(); child != 0; child = child->GetNextSibling()) {
                DetectCamera(child);
            }
    }
}

// ------------------------------------------------------------------------
void ViewScene3D::Finalize()
{
    (void)InitContent(false);
    CleanUp();
}

// ------------------------------------------------------------------------
void ViewScene3D::FinishInitContent()
{
    ContentLoader * contentLoader = GetViewHandler()->GetContentLoader();
    // check if we are completed or have more packets
    if ((ContentLoader::Completed == mBuildState) || (ContentLoader::MorePackets == mBuildState)) {
        // get the scene context as we need it afterwards
        Candera::Id sceneId = Candera::Internal::AssetProviderFunctions::GetIdByName(GetViewHandler()->GetAssetProvider(), SceneLib, GetId().CStr());
        mScene3DContext = contentLoader->GetSceneContextById(sceneId);
        // Initialized widgets and cameras
        InitWidgets();
        InitCameras();
        ResetActivateData(); // doesn't send notification onActivate
        ResetRenderingData(); // doesn't send notification OnRenderingEnabled
        OnInitContent(true);
        ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
    } else {
        COURIER_LOG_ERROR("Start Loading Asset for ViewScene3D '%s' failed.",GetId().CStr());
        mPartialLoadState = SynchronousLoadState;
        mScene3DContext = 0;
        ViewScene::SetLastLoadError(ViewScene::AssetLoadError);
    }

    if (0 != mScene3DContext) {
        COURIER_LOG_INFO("ViewScene3D '%s' Asset initialized, cameras(%u) widgets(%u)",GetId().CStr(), mCameras.Size(), mFrameworkWidgets.Size());
        ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
    }
}

// ------------------------------------------------------------------------
bool ViewScene3D::InitContent(bool init)
{
    static_cast<void>(ViewScene::InitContent(init));
    if (IsInitialized() && init && (0 != GetViewHandler()) && (0 != GetViewHandler()->GetAssetProvider())) {
        if(! IsContentInitialized()) {
            mPartialLoadCalls = 0;
            if (SynchronousLoadState == mPartialLoadState) {
                mPartialLoadState = InitialState;
                ContentLoader* contentLoader = GetViewHandler()->GetContentLoader();
                // first load scene without uploading to nvram
                Candera::Id sceneId = Candera::Internal::AssetProviderFunctions::GetIdByName(GetViewHandler()->GetAssetProvider(), SceneLib, GetId().CStr());
                mBuildState = contentLoader->BuildSceneContextById(sceneId, ContentLoader::NoVramUploadPolicy);
                FinishInitContent();
            } else {
                (void)BuildScene3DContextAsync();
                mPartialLoadState = WaitOnAsynchronousInitFinishedState;
            }
        }
        return (mScene3DContext!=0);
    }
    if(IsContentInitialized() && (! init)) {
        UnfocusWidgets();
        DeleteRenderJobs();

        // send notification independent of ref counter
        if (mIsActive) {
            OnActivate(false);
            mIsActive = false;
            OnPostActivate(false);
            mConsecutiveActivate = false;
        }
        ResetActivateData(); // doesn't send other notification onActivate

        // send notification independent of ref counter
        if (mRenderingEnabled) {
            OnRenderingEnabled(false);
        }
        ResetRenderingData();// doesn't send other notification OnRenderingEnabled

        mRefCounterVct.Clear();

        // force unload, because we destroying data structures
        (void)LoadContent(false, false);
        OnInitContent(false);

        GetViewHandler()->GetRenderer()->RemoveViewGduRelations(this);
        mCameras.Clear();
        ResetWidgets();

        GetViewHandler()->GetContentLoader()->ReleaseSceneContext(mScene3DContext);
        mScene3DContext = 0;

        COURIER_LOG_INFO("ViewScene3D '%s' Scenedata destroyed",GetId().CStr());
        ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
        return true;
    }
    ViewScene::SetLastLoadError(ViewScene::GeneralError);
    return false;
}

// ------------------------------------------------------------------------
bool ViewScene3D::LoadContent(bool load, bool forceUpload)
{
    if ((SynchronousLoadState != mPartialLoadState) && load) {
        // finish async load
        while (PartialLoad(forceUpload)) {
            if (!mPartialLoadAsyncRequest.PointsToNull()) {
                if (!mPartialLoadAsyncRequest->IsCompleted()) {
                    static_cast<void>(mPartialLoadAsyncRequest->WaitForFinished());
                }
            }
        }
        return mIsUploaded;
    }
    bool rc = ViewScene::LoadContent(load, forceUpload);
    rc = rc && IsContentInitialized();

    if(rc) {
        // check if we shall upload
        if(load) {
            // check if we still have something to load from asset
            if(mBuildState==ContentLoader::MorePackets) {
                mPartialLoadState = AssetLoadingState;
                // check if we haven´t load the complete asset
                ContentLoader * contentLoader = GetViewHandler()->GetContentLoader();
                // if we still have somthing to load then load it
                Candera::Id sceneId = Candera::Internal::AssetProviderFunctions::GetIdByName(GetViewHandler()->GetAssetProvider(), SceneLib, GetId().CStr());
                while (mBuildState == ContentLoader::MorePackets) {
                    mBuildState = contentLoader->BuildSceneContextById(sceneId, ContentLoader::NoVramUploadPolicy);
                }
                // Finally check if we have loaded everything from asset
                if (mBuildState==ContentLoader::Completed) {
                    COURIER_LOG_INFO("ViewScene3D '%s' Asset loaded",GetId().CStr());
                    ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                } else {
                    COURIER_LOG_ERROR("Finish Loading Asset for ViewScene3D '%s' failed.",GetId().CStr());
                    mPartialLoadState = SynchronousLoadState;
                    mScene3DContext = 0;
                    ViewScene::SetLastLoadError(ViewScene::AssetLoadError);
                    return false;
                }
            }

            // nothing uploaded yet
            if (! mIsUploaded) {
                mPartialLoadState = UploadingState;
                // create rendertargets
                EnableRendertargets(true, forceUpload);
                // upload here to nvram because rendertargets are now loaded
                Vector<Camera*>::Iterator it = mCameras.Begin();
                Renderer * renderer = GetViewHandler()->GetRenderer();
                for (; it != mCameras.End(); ++it) {
                    RenderTarget3D* renderTarget = (*it)->GetRenderTarget();
                    if (renderTarget != 0) {
                        Gdu* gdu = renderer->GetGdu(renderTarget);
                        if (gdu != 0) {
                            gdu->Activate();
                            rc = mScene3DContext->GetScene()->UploadAll() && rc;
                        }
                    }
                }
                if(rc) {
                    mPartialLoadState = FinishingState;
                    mIsUploaded = true;
                    // inform controller and widgets, that we uploaded to nvram
                    OnLoad(true);
                    COURIER_LOG_INFO("ViewScene3D '%s' uploaded",GetId().CStr());
                    ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                } else {
                    COURIER_LOG_ERROR("Upload ViewScene3D '%s' failed.",GetId().CStr());
                    ViewScene::SetLastLoadError(ViewScene::VRamUploadError);
                }
            // at least enable rendertargets if already uploaded
            } else {
                // COURIER_LOG_WARN("View '%s' already uploaded.",GetId().CStr());
                EnableRendertargets(true, forceUpload);
            }
            mPartialLoadState = SynchronousLoadState;
            return mIsUploaded;
        }

        if (mIsUploaded) {
            // clear the jobs, cause we have nothing to render anymore
            DeleteRenderJobs();
            // turn of rendering which further sends ParentViewRenderingEnabledEvent(false) if view scene was enabled for rendering.
            // change flag mConsecutiveRenderingEnabled because request was made from inside the view.
            EnableRendering(false);
            // unload the scene
            Vector<Camera*>::Iterator it = mCameras.Begin();
            Renderer * renderer = GetViewHandler()->GetRenderer();
            for (; it != mCameras.End(); ++it) {
                RenderTarget3D* renderTarget = (*it)->GetRenderTarget();
                if (renderTarget != 0) {
                    Gdu* gdu = renderer->GetGdu(renderTarget);
                    if (gdu != 0) {
                        gdu->Activate();
                        rc = mScene3DContext->GetScene()->UnloadAll() && rc;
                    }
                }
            }
            if(rc) {
                mIsUploaded = false;
                // inform controller and widgets, that we unload from nvram
                OnLoad(false);
                COURIER_LOG_INFO("ViewScene3D '%s' unloaded",GetId().CStr());
                ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
            } else {
                COURIER_LOG_ERROR("Unload ViewScene3D '%s' failed.",GetId().CStr());
                ViewScene::SetLastLoadError(ViewScene::VRamUploadError);
            }
            // free now the rendertargets
            EnableRendertargets(false, false);
            return (! mIsUploaded);
        }
        else {
            // MISRA Rule 6-4-2: all if ... else if constructs shall be terminated with an else clause
        }
    }
    return false;
}

// ------------------------------------------------------------------------

bool ViewScene3D::BuildScene3DContextAsync()
{
    ContentLoader* contentLoader = GetViewHandler()->GetContentLoader();
    if (0 == contentLoader) {
        mBuildState = ContentLoader::Error;
        return true;
    }
    if (mPartialLoadAsyncRequest.PointsToNull()) {
        Candera::Id sceneId = Candera::Internal::AssetProviderFunctions::GetIdByName(GetViewHandler()->GetAssetProvider(), SceneLib, GetId().CStr());
        mPartialLoadAsyncRequest = contentLoader->BuildSceneContextAsync(sceneId, ContentLoader::NoVramUploadPolicy);
    }
    if (mPartialLoadAsyncRequest.PointsToNull()) {
        mBuildState = ContentLoader::Error;
        return true;
    }
    if (mPartialLoadAsyncRequest->IsCompleted()) {
        mBuildState = mPartialLoadAsyncRequest->GetResult();
        mPartialLoadAsyncRequest = Candera::ContentLoader::AsyncBuildState::SharedPointer();
        return true;
    }
    return false;
}

bool ViewScene3D::PartialLoad(bool forceUpload)
{
    ++mPartialLoadCalls;
    // if a still pending asynchronous initialization of the content is finished
    if (WaitOnAsynchronousInitFinishedState == mPartialLoadState) {
        if (BuildScene3DContextAsync()) {
            // we are now in the initialized content state. we still have to finish the synchronous part of the init
            mPartialLoadState = InitialState;
            FinishInitContent();
            if (mIsUploaded) {
                EnableRendertargets(true, forceUpload);
            }
        }
        else {
            return true;
        }
    }
    if (0 == mScene3DContext) {
        return false;
    }
    if (InitialState == mPartialLoadState) {
        if (BuildScene3DContextAsync()) {
            // we are now in the asset loading state
            mPartialLoadState = AssetLoadingState;
        }
        else {
            return true;
        }
    }
    bool continuePartialLoad = false;
    // if we are in the asset loading state then access the asset.
    if (AssetLoadingState == mPartialLoadState) {
        // if we have loaded the asset, then start with uploading the stuff using the traverser.
        if (ContentLoader::Completed == mBuildState) {
            // create rendertargets
            EnableRendertargets(true, forceUpload);
            // upload here to nvram because rendertargets are now loaded
            UploadNodeTraverser& uploadTraverser = (0 != mCurrentUploadNodeTraverser) ? *mCurrentUploadNodeTraverser : mUploadTraverser;
            m_contextTraverser.InitTraverser(mScene3DContext->GetScene(), mScene3DContext, GetViewHandler()->GetRenderer(), &GetCameraPtrVector());
            m_contextTraverser.Traverse(&uploadTraverser);
            if (m_contextTraverser.IsFinished()) {
                mPartialLoadState = FinishingState;
            }
            else {
                mPartialLoadState = UploadingState;
                continuePartialLoad = true;
            }
        }
        else if (ContentLoader::MorePackets == mBuildState) {
            (void)BuildScene3DContextAsync();
            continuePartialLoad = true;
        }
        else {
            COURIER_LOG_ERROR("Async loading asset ViewScene3D '%s' failed, used steps(%u)", GetId().CStr(), mPartialLoadCalls);
            continuePartialLoad = false;
            ViewScene::SetLastLoadError(ViewScene::AssetLoadError);
            mScene3DContext = 0;
        }
        // if we are in the uploading state then check if we have finished.
    }
    else if (UploadingState == mPartialLoadState) {
        UploadNodeTraverser& uploadTraverser = (0 != mCurrentUploadNodeTraverser) ? *mCurrentUploadNodeTraverser : mUploadTraverser;
        m_contextTraverser.Traverse(&uploadTraverser);
        if (m_contextTraverser.IsFinished()) {
            mPartialLoadState = FinishingState;
        }
        else {
            continuePartialLoad = true;
        }
    }
    else {
        // MISRA Rule 6-4-2: all if ... else if constructs shall be terminated with an else clause
    }

    // if we have finished then send the success/error message
    if (FinishingState == mPartialLoadState) {
        bool rc = m_contextTraverser.IsSuccessful();
        if (rc) {
            if (!mIsUploaded)
            {
                mIsUploaded = true;
                // inform controller and widgets, that we uploaded to nvram
                OnLoad(true);
            }
            COURIER_LOG_INFO("Async ViewScene3D '%s' uploaded, used steps(%u)", GetId().CStr(), mPartialLoadCalls);
            ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
        }
        else {
            COURIER_LOG_ERROR("Async Upload ViewScene3D '%s' failed, used steps(%u)", GetId().CStr(), mPartialLoadCalls);
            ViewScene::SetLastLoadError(ViewScene::VRamUploadError);
        }
        Message * postMsg = (COURIER_MESSAGE_NEW(AsyncLoadIndMsg)(GetId(), true, rc, ViewScene::GetLastLoadError()));
        if (postMsg != 0) {
            static_cast<void>(postMsg->Post());
        }
        mPartialLoadState = SynchronousLoadState;
    }

    return continuePartialLoad;
}

// ------------------------------------------------------------------------
bool ViewScene3D::AsyncLoadContent(bool loadContent, bool forceUpload)
{
    mPartialLoadState = AsynchronousLoadState;
    bool rc = ViewScene::AsyncLoadContent(loadContent, forceUpload);
    rc = rc && IsContentInitialized();

    if (!mPartialLoadAsyncRequest.PointsToNull()) {
        if (!mPartialLoadAsyncRequest->IsCompleted()) {
            (void)GetViewHandler()->SchedulePartialLoad(this, forceUpload);
            return true;
        }
        rc = true;
    }

    if(rc) {
        if (mIsUploaded && loadContent) {
            // COURIER_LOG_WARN("View '%s' already uploaded.",GetId().CStr());
            EnableRendertargets(true, forceUpload);
            return true;
        }

        // check if we shall upload
        if ((!mIsUploaded) && loadContent) {
            // check if we are allowed to register the loading scene.
            mPartialLoadState = InitialState;
            // first step of async load
            bool continuePartialLoad = PartialLoad(forceUpload);
            // if we shall not continue the reset the current loading scene.
            if (continuePartialLoad) {
                (void)GetViewHandler()->SchedulePartialLoad(this, forceUpload);
                }
                // if it is loaded, or loading was started then return true
                rc = (mScene3DContext!=0);
                if(rc) {
                    rc = mIsUploaded || (mPartialLoadState==UploadingState) || (mPartialLoadState==AssetLoadingState);
                }
            return rc;
        }

        if (mIsUploaded && (!loadContent)) {
            return LoadContent(loadContent, false);
        }
    }
    return false;
}

// ------------------------------------------------------------------------
void ViewScene3D::EnableRendertargets(bool enable, bool forceUpload)
{
    if (IsContentInitialized()) {
        for (SizeType i = 0; i < mCameras.Size(); ++i) {
            Renderer * renderer = GetViewHandler()->GetRenderer();
            if (0 != renderer) {
                renderer->SetIs2DRequest(false);
            }
            EnableRenderTarget(enable, mCameras[i]->GetRenderTarget(), forceUpload);
        }
        if (enable && mClearOnSceneLoading) {
            Clear(false, FeatStd::Optional<Candera::Rectangle>());
        }
    }
}

// ------------------------------------------------------------------------
bool ViewScene3D::IsContentInitialized() const
{
    return mScene3DContext!=0;
}

// ------------------------------------------------------------------------
bool ViewScene3D::IsContentLoaded() const
{
    return IsContentInitialized() && mIsUploaded;
}

// ------------------------------------------------------------------------
void ViewScene3D::Activate(bool activate)
{
    if (activate != mConsecutiveActivate) {
        if (!IsContentInitialized()) {
            // no need of ref count for cameras, as the content is not initialized
            OnActivate(activate);
            mIsActive = activate;
            OnPostActivate(activate);
        }
        else {
            ActivateImpl(activate);
        }
            mConsecutiveActivate = activate;
    }
}

// ------------------------------------------------------------------------
void ViewScene3D::EnableRendering(bool enable)
{
    if (enable != mConsecutiveRenderingEnabled) {
        EnableRenderingImpl(enable);
        mConsecutiveRenderingEnabled = enable;
    }
}

// ------------------------------------------------------------------------
void ViewScene3D::DeleteRenderJobs()
{
    Renderer * renderer = GetViewHandler()->GetRenderer();
    if ((renderer!=0) && (mScene3DContext!=0)) {
        for(SizeType idx=0; idx<mCameras.Size(); idx++) {
            Camera * camera = mCameras[idx];
            renderer->DeleteRenderJob(camera);
        }

        for (FeatStd::SizeType idx = 0; idx < mClearingData3DVct.Size(); idx++) {
            if (mClearingData3DVct[idx] != 0) {
                Camera * camera = mClearingData3DVct[idx]->GetClearCamera();
                if (camera != 0) {
                    renderer->DeleteRenderJob(camera);
                }
            }
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene3D::ReleaseSceneContext()
{
}

// ------------------------------------------------------------------------
void ViewScene3D::Accept(ViewVisitor & visitor)
{
    visitor.Visit(this);
}

// ------------------------------------------------------------------------
void ViewScene3D::Clear(bool onlyCameraViewPorts, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    Renderer* renderer = GetViewHandler()->GetRenderer();
    if (0 != renderer) {
        RTPtrVector renderTargets;
        if (GetRenderTargetPtrVector(renderTargets,true) && (renderTargets.Size() > 0)) {
            for (SizeType i = 0; i < renderTargets.Size(); ++i) {
                Candera::RenderTarget *renderTarget = renderTargets[i];
                Courier::Gdu* gdu = renderer->GetGdu(renderTarget);
                if (0 != gdu) {
                    if (onlyCameraViewPorts) {
                        if (!mClearingDataInitFlag.Find(renderTarget))
                        {
                            // create the vector of clear data only once with first call of Clear(true)
                            InitClearingData3D(gdu);
                            mClearingDataInitFlag.Insert(renderTarget, true); //at least one call to UpdateClearingDataVct was made
                        }
                        else
                        {
                            // Update clear data for updated cameras (if any)
                            UpdateClearingData3D();
                        }
                        for (SizeType idx = 0; idx < mClearingData3DVct.Size(); ++idx) {
                            ScheduleCamera(mClearingData3DVct[idx]->GetClearCamera(), renderer, dirtyArea, GetViewHandler(), gdu);
                        }
                    }
                    else {
                        Camera* clearCamera = gdu->GetClearCamera3D();
                        ScheduleCamera(clearCamera, renderer, dirtyArea, GetViewHandler(), gdu);
                    }
                }
            }
        }
    }
}

// ------------------------------------------------------------------------
FrameworkWidget * ViewScene3D::GetTouchedWidget3D(ViewScene3D * view3D, const TouchInfo & info)
{
    FrameworkWidget * nearestWidget = 0;
    if((view3D!=0) && view3D->IsActive()) {
        Float currentDistance = 9999999999.0F;
        Float minimalDistance = currentDistance;

        const ViewScene::FrameworkWidgetPtrVector & widgets = view3D->GetFrameworkWidgetPtrVector();
        const ViewScene3D::CameraPtrVector & cameras = view3D->GetCameraPtrVector();
        for (SizeType i=0; i<widgets.Size(); ++i) {
            FrameworkWidget * fw = widgets[i];
            if(fw->IsTouchable()) {
                Candera::WidgetBase* wb = static_cast<Candera::WidgetBase*>(fw);
                Candera::Widget* w = Candera::Dynamic_Cast<Candera::Widget*>(wb);
                if (w!=0) {
                    Node * node = w->GetNode();
                    if(node!=0) {
                        for(SizeType idx=0; idx<cameras.Size(); idx++) {
                            Camera * camera3D = cameras[idx];
                            if(camera3D->IsRenderingEnabled()) {
                                if(node->IsPickIntersectingGeometry(*camera3D, static_cast<Int>(info.mX), static_cast<Int>(info.mY), currentDistance)) {
                                    if(currentDistance < minimalDistance) {
                                        minimalDistance = currentDistance;
                                        nearestWidget = fw;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return nearestWidget;
}

// ------------------------------------------------------------------------
FrameworkWidget * ViewScene3D::GetTouchedWidget(const TouchInfo & info)
{
    if(mGetTouchedWidget3DFunction!=0) {
        return mGetTouchedWidget3DFunction(this,info);
    }
    return 0;
}

// ------------------------------------------------------------------------
class FindAnimationTraverser3D : public Candera::TreeTraverserBase<Node>
{
public:
    FindAnimationTraverser3D(const CompositePath & compositePath, const ItemId & animationId) : mCompositePath(compositePath), mAnimationId(animationId) { }

    Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> GetAnimationPlayer() const { return mAnimationPlayer; }

protected:
    virtual TraverserAction ProcessNode(Node& node)
    {
        CompositeGroup* composite = Dynamic_Cast<CompositeGroup*>(&node);
        if (0 != composite) {
            if (CompositePath(composite->GetStringId()) == mCompositePath) {
                mAnimationPlayer = composite->GetAnimationByName(mAnimationId.CStr());
                return StopTraversing;
            }
        }
        return ProceedTraversing;
    }

private:
    CompositePath mCompositePath;
    ItemId mAnimationId;
    Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> mAnimationPlayer;
};

Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> ViewScene3D::GetAnimation(const CompositePath & compositePath, const ItemId & animationId)
{
    FindAnimationTraverser3D traverser(compositePath, animationId);
    if (0 != mScene3DContext) {
        traverser.Traverse(*mScene3DContext->GetScene());
    }
    return traverser.GetAnimationPlayer();
}

// ------------------------------------------------------------------------
void ViewScene3D::ScheduleCamera(Camera* camera, Renderer* renderer, const FeatStd::Optional<Candera::Rectangle>& dirtyArea, IViewHandler* viewHandler, const Gdu* gdu)
{
    if ((0 != camera) && (0 != renderer)){
        renderer->Queue3DJob(this, camera, cDefaultRender3DCounter, dirtyArea, true);
        if ((mInvalidateCamerasOnSceneClearing) && (0 != viewHandler) && (0 != gdu)) {
            viewHandler->Invalidate(gdu->GetRenderTarget3D(), dirtyArea);
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene3D::CleanUp()
{
    mScene3DContext = 0;

    for (SizeType i = 0; i< mClearingData3DVct.Size(); ++i) {
        FEATSTD_DELETE(mClearingData3DVct[i]); //Finalize will be called in the destructor
    }
    mClearingData3DVct.Clear();
    mCameraToClearingData.Clear();
    mRefCounterVct.Clear();
}

// ------------------------------------------------------------------------
void ViewScene3D::ActivateImpl(bool activate)
{
    for (SizeType i = 0; i < mCameras.Size(); ++i) {
        ActivateForCamera(activate, mCameras[i]);
    }
}

// ------------------------------------------------------------------------
void ViewScene3D::EnableRenderingImpl(bool enable)
{
    for (SizeType i = 0; i < mCameras.Size(); ++i) {
        EnableRenderingForCamera(enable, mCameras[i]);
    }
}

// ------------------------------------------------------------------------
void ViewScene3D::EnableRenderingScene(bool enable)
{
    if (0 != mScene3DContext) {
        Scene * scene = mScene3DContext->GetScene();
        if (0 != scene) {
            scene->SetRenderingEnabled(enable);
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene3D::UpdateCameraRenderingState(bool enable, Candera::CanderaObject* camera)
{
    Camera* camera3D = static_cast<Camera*>(camera);
    if ((0 != camera3D) && (enable != camera3D->IsRenderingEnabled())){
            camera3D->SetRenderingEnabled(enable);
    }
}

void ViewScene3D::InitClearingData3D(const Gdu* gdu)
{
    for (CameraPtrVector::ConstIterator it = mCameras.ConstBegin(); it != mCameras.ConstEnd(); ++it)
    {
        Camera *camera = (*it);
        if (gdu == GetViewHandler()->GetRenderer()->GetGdu(camera->GetRenderTarget())) {
            ClearingData3D* clearData = FEATSTD_NEW(ClearingData3D);
            if (0 != clearData) {
                clearData->Init(gdu);
                clearData->SetupClearCamera(camera);

                static_cast<void>(mClearingData3DVct.Add(clearData));
                static_cast<void>(mCameraToClearingData.Insert(camera, clearData));
            }
        }
    }
}

void ViewScene3D::UpdateClearingData3D()
{
    for (CameraPtrVector::ConstIterator it = mCameras.ConstBegin(); it != mCameras.ConstEnd(); ++it)
    {
        Camera *camera = (*it);
        if ((camera != 0) && camera->WasUpdated())
        {
            ClearingData3D::Base **clearData = mCameraToClearingData.Find(camera);
            if ((clearData != 0) && (*clearData != 0))
            {
                (*clearData)->SetupClearCamera(camera);
            }
        }
    }
}

}
#endif
