//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ViewScene2D.h"

#if defined(CANDERA_2D_ENABLED)

#include "IViewHandler.h"
#include "ViewVisitor.h"
#include "RenderHint.h"
#include "Renderer.h"
#include "VisualizationMsgs.h"
#include <CanderaWidget/Widget2D/Widget2D.h>

#include <CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrushColorBlend.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrushHslBlend.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrushMaskBlend.h>
#include <CanderaPlatform/Device/Common/Effects/BlurBitmapBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/MirrorBitmapBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/ShadowBitmapBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/ShearBitmapBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrushColor.h>

#include <Candera/Engine2D/Core/CompositeGroup2D.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

#if !defined(COURIER_DEFAULT_VIEW2D_CAMERA_COUNT)
    static const Int cCOURIER_DEFAULT_VIEW2D_CAMERA_COUNT = 2;
#else
    static const Int cCOURIER_DEFAULT_VIEW2D_CAMERA_COUNT = COURIER_DEFAULT_VIEW2D_CAMERA_COUNT;
#endif

using namespace Candera;
using namespace Internal;

// ------------------------------------------------------------------------

ViewScene2D::GetTouchedWidget2DFct ViewScene2D::mGetTouchedWidget2DFunction =  ViewScene2D::GetTouchedWidget2D;

// ------------------------------------------------------------------------
ViewScene2D::ViewScene2D(bool managed) :
    Base(managed),
    mScene2DContext(0),
    mBuildState(Candera::ContentLoader::Error),
    mPartialLoadState(SynchronousLoadState),
    mPartialLoadCalls(0),
    mCurrentUploadNodeTraverser(0),
    mClearingData2DVct()
{
    if(! mCameras.Reserve(cCOURIER_DEFAULT_VIEW2D_CAMERA_COUNT)) {
        FEATSTD_PANIC("Can't reserve memory for camera pointers");
    }
}

// ------------------------------------------------------------------------
ViewScene2D::~ViewScene2D()
{
    CleanUp();
    SetParentView(0);
}

// ------------------------------------------------------------------------
void ViewScene2D::FillWidgetList()
{
    if (mScene2DContext!=0) {
        AddWidgets(mScene2DContext->GetWidgetIterator());
        AddWidgetTraverser2D addWidgetTraverer(this);
        addWidgetTraverer.Traverse(*mScene2DContext->GetScene());
    }
}

// ------------------------------------------------------------------------
class Camera2DLessBySequenceNumber
{
    public:
        bool operator()(const Camera2D * a, const Camera2D * b) const {
            return a->GetSequenceNumber() < b->GetSequenceNumber();
        }
};

// ------------------------------------------------------------------------
void ViewScene2D::InitCameras()
{
    if (0 != mScene2DContext) {
        mCameras.Clear();
        mRefCounterVct.Clear();
        DetectCamera(mScene2DContext->GetScene());
        FEATSTD_LINT_NEXT_EXPRESSION(1502, "no static members, thats okay")
        static Camera2DLessBySequenceNumber objSort;
        mCameras.Sort(objSort);
        for(FeatStd::SizeType idx=0; idx<mCameras.Size(); idx++) {
            if (0 != mCameras[idx]) {
                static_cast<void>(mRefCounterVct.Add(CameraRefCounter(mCameras[idx])));
                if((mCameras[idx]->GetCameraRenderStrategy()!=0) && (GetViewHandler()->GetRenderer()!=0)) {
                    Gdu * gdu = GetViewHandler()->GetRenderer()->GetGdu(mCameras[idx]->GetRenderTarget());
                    if(gdu!=0) {
                        gdu->UsesCameraWithRenderStrategy();
                    }
                }
            }
        }
    }
}

// ------------------------------------------------------------------------
static bool IsQualifiedNameEqual(const Char* qualifiedName, FeatStd::SizeType qualifiedNameLength, Node2D* node)
{
    Node2D* current = node;
    FeatStd::OffsetType strCurrentIdx = qualifiedNameLength;
    while (0 != current) {
        const Char* name = current->GetName();
        FeatStd::SizeType nameLength = StringPlatform::Length(name);
        strCurrentIdx -= static_cast<FeatStd::OffsetType>(nameLength);
        const Char* compareName = &(qualifiedName[strCurrentIdx]);
        if (strCurrentIdx < 0) {
            // incorrect camera length detected;
            return false;
        }
        if (FeatStd::Internal::String::ComparePartial(compareName, name, nameLength) != 0) {
            // incorrect name segments
            return false;
        }
        Node2D* parent = current->GetParent();
        if (0 != parent) {
            // iterate to delimiter '@'
            --strCurrentIdx;
            if (strCurrentIdx < 0) {
                // incorrect camera length detected
                return false;
            }
            if (qualifiedName[strCurrentIdx] != '@') {
                // incorrect name
                return false;
            }
        }
        // so far so good. name is a match and delimiter is in the right place proceed with parent
        current = parent;
    }
    return true;
}

// ------------------------------------------------------------------------
Candera::Camera2D * ViewScene2D::FindCamera2D(const Char * cameraName)
{
    FeatStd::SizeType idx;
    // for optimizing reasons compare the pointers of strings
    for(idx=0; idx<mCameras.Size(); idx++) {
        Camera2D * camera = mCameras[idx];
        if(camera->GetName()==cameraName) {
            return camera;
        }
    }
    // if not found then make string compares
    for(idx=0; idx<mCameras.Size(); idx++) {
        Camera2D * camera = mCameras[idx];
        if(0==FeatStd::Internal::String::CompareStrings(camera->GetName(), cameraName)) {
            return camera;
        }
    }

    // if not found  then compare the whole full names
    for (idx = 0; idx < mCameras.Size(); ++idx) {
        Camera2D* camera = mCameras[idx];
        if (IsQualifiedNameEqual(cameraName, StringPlatform::Length(cameraName), camera)) {
            return camera;
        }
    }
    return 0;
}

Candera::Camera2D* ViewScene2D::FindCamera2D(const ItemId& id)
{
    Camera2D **pCamera = mIdToCamera.Find(id);
    if (pCamera != 0)
    {
        return (*pCamera);
    }

    // Search by name and update maps
    Camera2D *camera = FindCamera2D(id.CStr());
    if (camera != 0)
    {
        static_cast<void>(mIdToCamera.Insert(id, camera));
        static_cast<void>(mCameraToId.Insert(camera, id));
    }

    return camera;
}

// ------------------------------------------------------------------------
bool ViewScene2D::GetRenderTargetPtrVector(RTPtrVector & vector)
{
    return ViewScene2D::GetRenderTargetPtrVector(vector,true);
}

// ------------------------------------------------------------------------
bool ViewScene2D::GetRenderTargetPtrVector(RTPtrVector & vector, bool clear)
{
    if(clear) {
        vector.Clear();
    }
    bool rc = true;
    for(SizeType idx=0; idx<mCameras.Size(); idx++) {
        Camera2D * camera = mCameras[idx];
        rc = View::AddRenderTarget(vector,camera->GetRenderTarget()) && rc;
    }
    return rc;
}

// ------------------------------------------------------------------------
void ViewScene2D::Invalidate(UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if (!IsAlreadyInvalidated(ItemId(), dirtyArea)) {
        // invalidate all invalidation dependencies.
        ViewScene::Invalidate(dirtyArea);
        for (SizeType idx = 0; idx < mCameras.Size(); ++idx) {
            Camera2D * camera = mCameras[idx];
            if (camera->IsRenderingEnabled()) {
                GetViewHandler()->GetRenderer()->Queue2DJob(this, camera, renderCounter, dirtyArea, false);
            }
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::InvalidateOffscreenHelper(const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if (!IsAlreadyInvalidated(ItemId(), dirtyArea)) {
        // invalidate all invalidation dependencies.
        ViewScene::Invalidate(dirtyArea);
        for (SizeType idx = 0; idx < mCameras.Size(); ++idx) {
            Camera2D * camera = mCameras[idx];
            if (camera->IsRenderingEnabled()) {
                GetViewHandler()->GetRenderer()->Queue2DJob(this, camera, cDefaultRender2DCounter, dirtyArea, false, true);
            }
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::Invalidate(Candera::Camera2D * camera, UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    // invalidate all invalidation dependencies.
    ItemId *pCameraId = mCameraToId.Find(camera);
    ViewScene::InvalidateImpl(
        (pCameraId != 0) ? (*pCameraId) : ItemId(camera->GetName()),
        dirtyArea);
    if (camera->IsRenderingEnabled()) {
        GetViewHandler()->GetRenderer()->Queue2DJob(this, camera, renderCounter, dirtyArea, false);
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::Invalidate(RenderTarget * renderTarget, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    // invalidate all invalidation dependencies.
    ViewScene::InvalidateImpl(ItemId(), dirtyArea);
    Renderer * renderer = GetViewHandler()->GetRenderer();
    if ((0 != renderer) && IsContentLoaded()) {
        for (SizeType idx = 0; idx < mCameras.Size(); ++idx) {
            Camera2D * camera = mCameras[idx];
            if (camera->IsRenderingEnabled() && (camera->GetRenderTarget() == renderTarget)) {
                renderer->Queue2DJob(this, camera, cDefaultRender2DCounter, dirtyArea, false);
            }
        }
    }
}

// ------------------------------------------------------------------------
FEATSTD_LINT_CURRENT_SCOPE(1762, "method cannot be made const for certain builds")
void ViewScene2D::LookupSurfaceHandle(const Node2D * node)
{
    FEATSTD_DEBUG_ASSERT(node!=0);
    if (node!=0) {
        FEATSTD_DEBUG_ASSERT(node->IsTypeOf(RenderNode::GetTypeId()));
        const RenderNode * rn = static_cast<const RenderNode*>(node);
        UInt8 idx = 0;
        for (Effect2D* effect2D = rn->GetEffect(idx); 0 != effect2D; effect2D = rn->GetEffect(++idx)) {
            BitmapBrush* bb = Candera::Dynamic_Cast<BitmapBrush*>(effect2D->GetBrushEffect2D());
            if (0 != bb) {
                Image2D* image = bb->Image().Get();
                if (0 != image) {
                    ProxyImage2D* proxyImage = Dynamic_Cast<ProxyImage2D*>(image);
                    if (0 != proxyImage) {
                        ChangeViewSurfaceRelation(AddViewRelation, proxyImage->GetImageSource());
                    }
                }
            }
        }
    }
}

void ViewScene2D::UpdateClearingData2D()
{
    for (CameraPtrVector::ConstIterator it = mCameras.ConstBegin(); it != mCameras.ConstEnd(); ++it)
    {
        Camera2D *camera = (*it);
        if ((camera != 0) && camera->WasUpdated())
        {
            ClearingData2D::Base **clearData = mCameraToClearingData.Find(camera);
            if ((clearData != 0) && (*clearData != 0))
            {
                (*clearData)->SetupClearCamera(camera);
            }
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::DetectCamera(Node2D * node)
{
    FEATSTD_DEBUG_ASSERT(node!=0);
    if (node!=0) {
        if (ViewScene::GetDetectViewReleationShip() && node->IsTypeOf(RenderNode::GetTypeId())) {
            LookupSurfaceHandle(node);
        } else if (node->IsTypeOf(Camera2D::GetTypeId())) {
            (void)mCameras.Add(static_cast<Camera2D*>(node));
        } else {
            for (Node2D* child = node->GetFirstChild(); child != 0; child = child->GetNextSibling()) {
                DetectCamera(child);
            }
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::Finalize()
{
    (void)InitContent(false);
    CleanUp();
}

// ------------------------------------------------------------------------
void ViewScene2D::FinishInitContent()
{
    ContentLoader * contentLoader = GetViewHandler()->GetContentLoader();
    // check if we are completed or have more packets
    if ((ContentLoader::Completed == mBuildState) || (ContentLoader::MorePackets == mBuildState)) {
        // get the scene context as we need it afterwards
        Candera::Id sceneId = Candera::Internal::AssetProviderFunctions::GetIdByName(GetViewHandler()->GetAssetProvider(), Scene2DLib, GetId().CStr());
        mScene2DContext = contentLoader->GetScene2DContextById(sceneId);
        // Initialized widgets and cameras
        InitWidgets();
        InitCameras();
        ResetActivateData(); // doesn't send notification onActivate
        ResetRenderingData(); // doesn't send notification OnRenderingEnabled
        OnInitContent(true);
        ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
    } else {
        COURIER_LOG_ERROR("Start Loading Asset for ViewScene2D '%s' failed.",GetId().CStr());
        mPartialLoadState = SynchronousLoadState;
        mScene2DContext = 0;
        ViewScene::SetLastLoadError(ViewScene::AssetLoadError);
    }

    if (0 != mScene2DContext) {
        COURIER_LOG_INFO("ViewScene2D '%s' Asset initialized, cameras(%d) widgets(%d)",GetId().CStr(), mCameras.Size(), mFrameworkWidgets.Size());
        ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
    }
}

// ------------------------------------------------------------------------
bool ViewScene2D::InitContent(bool init)
{
    static_cast<void>(ViewScene::InitContent(init));
    if (IsInitialized() && init && (0 != GetViewHandler()) && (0 != GetViewHandler()->GetAssetProvider())) {
        if(! IsContentInitialized()) {
            mPartialLoadCalls = 0;
            if (SynchronousLoadState == mPartialLoadState) {
                mPartialLoadState = InitialState;
                ContentLoader* contentLoader = GetViewHandler()->GetContentLoader();
                // first load scene without uploading to nvram
                Candera::Id sceneId = Candera::Internal::AssetProviderFunctions::GetIdByName(GetViewHandler()->GetAssetProvider(), Scene2DLib, GetId().CStr());
                mBuildState = contentLoader->BuildScene2DContextById(sceneId, ContentLoader::NoVramUploadPolicy);
                FinishInitContent();
            }
            else {
                (void)BuildScene2DContextAsync();
                mPartialLoadState = WaitOnAsynchronousInitFinishedState;
            }
        }
        return (mScene2DContext!=0);
    }
    if(IsContentInitialized() && (! init)) {
        UnfocusWidgets();
        DeleteRenderJobs();

        // send notification independent of ref counter
        if (mIsActive) {
            OnActivate(false);
            mIsActive = false;
            OnPostActivate(false);
            mConsecutiveActivate = false;
        }
        ResetActivateData(); // doesn't send other notification

        // send notification independent of ref counter
        if (mRenderingEnabled) {
            OnRenderingEnabled(false);
        }
        ResetRenderingData();// doesn't send other notification

        mRefCounterVct.Clear();

        // force unload, because we destroying data structures
        (void)LoadContent(false, false);
        OnInitContent(false);

        GetViewHandler()->GetRenderer()->RemoveViewGduRelations(this);
        mCameras.Clear();
        ResetWidgets();

        GetViewHandler()->GetContentLoader()->ReleaseScene2DContext(mScene2DContext);
        mScene2DContext = 0;

        COURIER_LOG_INFO("ViewScene2D '%s' Scenedata destroyed",GetId().CStr());
        ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
        return true;
    }
    ViewScene::SetLastLoadError(ViewScene::GeneralError);
    return false;
}

// ------------------------------------------------------------------------
bool ViewScene2D::LoadContent(bool load, bool forceUpload)
{
    if ((SynchronousLoadState != mPartialLoadState) && load) {
        // finish async load
        while (PartialLoad(forceUpload)) {
            if (!mPartialLoadAsyncRequest.PointsToNull()) {
                if (!mPartialLoadAsyncRequest->IsCompleted()) {
                    static_cast<void>(mPartialLoadAsyncRequest->WaitForFinished());
                }
            }
        }
        return mIsUploaded;
    }
    bool rc = ViewScene::LoadContent(load, forceUpload);
    rc = rc && IsContentInitialized();

    if(rc) {
        // check if we shall upload
        if(load) {
            // check if we still have something to load from asset
            if(mBuildState==ContentLoader::MorePackets) {
                mPartialLoadState = AssetLoadingState;
                // check if we haven't load the complete asset
                ContentLoader * contentLoader = GetViewHandler()->GetContentLoader();
                // if we still have somthing to load then load it
                Candera::Id sceneId = Candera::Internal::AssetProviderFunctions::GetIdByName(GetViewHandler()->GetAssetProvider(), Scene2DLib, GetId().CStr());
                while (mBuildState == ContentLoader::MorePackets) {
                    mBuildState = contentLoader->BuildScene2DContextById(sceneId, ContentLoader::NoVramUploadPolicy);
                }
                // Finally check if we have loaded everything from asset
                if (mBuildState==ContentLoader::Completed) {
                    COURIER_LOG_INFO("ViewScene2D '%s' Asset loaded",GetId().CStr());
                    ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                } else {
                    COURIER_LOG_ERROR("Finish Loading Asset for ViewScene2D '%s' failed.",GetId().CStr());
                    mPartialLoadState = SynchronousLoadState;
                    mScene2DContext = 0;
                    ViewScene::SetLastLoadError(ViewScene::AssetLoadError);
                    return false;
                }
            }

            // nothing uploaded yet
            if (! mIsUploaded) {
                mPartialLoadState = UploadingState;
                // create rendertargets
                EnableRendertargets(true, forceUpload);
                // upload here to nvram because rendertargets are now loaded
                rc = mScene2DContext->GetScene()->Upload(ScopeMask(), Node2D::Deep);
                if(rc) {
                    mPartialLoadState = FinishingState;
                    mIsUploaded = true;
                    // inform controller and widgets, that we uploaded to nvram
                    OnLoad(true);
                    COURIER_LOG_INFO("ViewScene2D '%s' uploaded",GetId().CStr());
                    ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
                } else {
                    COURIER_LOG_ERROR("Upload ViewScene2D '%s' failed.",GetId().CStr());
                    ViewScene::SetLastLoadError(ViewScene::VRamUploadError);
                }
            // at least enable rendertargets if already uploaded
            } else {
                // COURIER_LOG_WARN("View '%s' already uploaded.",GetId().CStr());
                EnableRendertargets(true, forceUpload);
            }
            mPartialLoadState = SynchronousLoadState;
            return mIsUploaded;
        }

        if (mIsUploaded) {
            // clear the jobs, cause we have nothing to render anymore
            DeleteRenderJobs();
            // turn of rendering which further sends ParentViewRenderingEnabledEvent(false) if view scene was enabled for rendering.
            // change flag mConsecutiveRenderingEnabled because request was made from inside the view.
            EnableRendering(false);
            // unload the scene
            rc = mScene2DContext->GetScene()->Unload(ScopeMask(), Node2D::Deep);
            if(rc) {
                mIsUploaded = false;
                // inform controller and widgets, that we unload from nvram
                OnLoad(false);
                COURIER_LOG_INFO("ViewScene2D '%s' unloaded",GetId().CStr());
                ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
            } else {
                COURIER_LOG_ERROR("Unload ViewScene2D '%s' failed.",GetId().CStr());
                ViewScene::SetLastLoadError(ViewScene::VRamUploadError);
            }
            // free now the rendertargets
            EnableRendertargets(false, false);
            return ! mIsUploaded;
        }
        else {
            // MISRA Rule 6-4-2: all if ... else if constructs shall be terminated with an else clause
        }
    }
    return false;
}

// ------------------------------------------------------------------------

bool ViewScene2D::BuildScene2DContextAsync()
{
    ContentLoader* contentLoader = GetViewHandler()->GetContentLoader();
    if (0 == contentLoader) {
        mBuildState = ContentLoader::Error;
        return true;
    }
    if (mPartialLoadAsyncRequest.PointsToNull()) {
        Candera::Id sceneId = Candera::Internal::AssetProviderFunctions::GetIdByName(GetViewHandler()->GetAssetProvider(), Scene2DLib, GetId().CStr());
        mPartialLoadAsyncRequest = contentLoader->BuildScene2DContextAsync(sceneId, ContentLoader::NoVramUploadPolicy);
    }
    if (mPartialLoadAsyncRequest.PointsToNull()) {
        mBuildState = ContentLoader::Error;
        return true;
    }
    if (mPartialLoadAsyncRequest->IsCompleted()) {
        mBuildState = mPartialLoadAsyncRequest->GetResult();
        mPartialLoadAsyncRequest = Candera::ContentLoader::AsyncBuildState::SharedPointer();
        return true;
    }
    return false;
}

bool ViewScene2D::PartialLoad(bool forceUpload)
{
    ++mPartialLoadCalls;
    // if a still pending asynchronous initialization of the content is finished
    if (WaitOnAsynchronousInitFinishedState == mPartialLoadState) {
        if (BuildScene2DContextAsync()) {
            // we are now in the initialized content state. we still have to finish the synchronous part of the init
            mPartialLoadState = InitialState;
            FinishInitContent();
            if (mIsUploaded) {
                EnableRendertargets(true, forceUpload);
            }
        }
        else {
            return true;
        }
    }
    if (0 == mScene2DContext) {
        return false;
    }
    if (InitialState == mPartialLoadState) {
        if (BuildScene2DContextAsync()) {
            // we are now in the asset loading state
            mPartialLoadState = AssetLoadingState;
        }
        else {
            return true;
        }
    }
    bool continuePartialLoad = false;
    // if we are in the asset loading state then access the asset.
    if (AssetLoadingState == mPartialLoadState) {
        // if we have loaded the asset, then start with uploading the stuff using the traverser.
        if (ContentLoader::Completed == mBuildState) {
            // create rendertargets
            EnableRendertargets(true, forceUpload);
            // upload here to nvram because rendertargets are now loaded
            UploadNode2DTraverser& uploadTraverser = (0 != mCurrentUploadNodeTraverser) ? *mCurrentUploadNodeTraverser : mUploadTraverser;
            m_contextTraverser.InitTraverser(mScene2DContext->GetScene(), mScene2DContext, GetViewHandler()->GetRenderer(), &GetCameraPtrVector());
            m_contextTraverser.Traverse(&uploadTraverser);
            if (m_contextTraverser.IsFinished()) {
                mPartialLoadState = FinishingState;
            }
            else {
                mPartialLoadState = UploadingState;
                continuePartialLoad = true;
            }
        }
        else if (ContentLoader::MorePackets == mBuildState) {
            (void)BuildScene2DContextAsync();
            continuePartialLoad = true;
        }
        else {
            COURIER_LOG_ERROR("Async loading asset ViewScene2D '%s' failed, used steps(%u)", GetId().CStr(), mPartialLoadCalls);
            continuePartialLoad = false;
            ViewScene::SetLastLoadError(ViewScene::AssetLoadError);
            mScene2DContext = 0;
        }
        // if we are in the uploading state then check if we have finished.
    }
    else if (UploadingState == mPartialLoadState) {
        UploadNode2DTraverser& uploadTraverser = (0 != mCurrentUploadNodeTraverser) ? *mCurrentUploadNodeTraverser : mUploadTraverser;
        m_contextTraverser.Traverse(&uploadTraverser);
        if (m_contextTraverser.IsFinished()) {
            mPartialLoadState = FinishingState;
        }
        else {
            continuePartialLoad = true;
        }
    }
    else {
        // MISRA Rule 6-4-2: all if ... else if constructs shall be terminated with an else clause
    }

    // if we have finished then send the success/error message
    if (FinishingState == mPartialLoadState) {
        bool rc = m_contextTraverser.IsSuccessful();
        if (rc) {
            if (!mIsUploaded)
            {
                mIsUploaded = true;
                // inform controller and widgets, that we uploaded to nvram
                OnLoad(true);
            }
            COURIER_LOG_INFO("Async ViewScene2D '%s' uploaded, used steps(%u)", GetId().CStr(), mPartialLoadCalls);
            ViewScene::SetLastLoadError(ViewScene::LoadSuccess);
        }
        else {
            COURIER_LOG_ERROR("Async Upload ViewScene2D '%s' failed, used steps(%u)", GetId().CStr(), mPartialLoadCalls);
            ViewScene::SetLastLoadError(ViewScene::VRamUploadError);
        }
        Message * postMsg = (COURIER_MESSAGE_NEW(AsyncLoadIndMsg)(GetId(), true, rc, ViewScene::GetLastLoadError()));
        if (0 != postMsg) {
            static_cast<void>(postMsg->Post());
        }
        mPartialLoadState = SynchronousLoadState;
    }
    return continuePartialLoad;
}

// ------------------------------------------------------------------------
bool ViewScene2D::AsyncLoadContent(bool loadContent, bool forceUpload)
{
    mPartialLoadState = AsynchronousLoadState;
    bool rc = ViewScene::AsyncLoadContent(loadContent, forceUpload);
    rc = rc && IsContentInitialized();

    if (!mPartialLoadAsyncRequest.PointsToNull()) {
        if (!mPartialLoadAsyncRequest->IsCompleted()) {
            (void)GetViewHandler()->SchedulePartialLoad(this, forceUpload);
            return true;
        }
        rc = true;
    }

    if(rc) {
        if (mIsUploaded && loadContent) {
            // COURIER_LOG_WARN("View '%s' already uploaded.",GetId().CStr());
            EnableRendertargets(true, forceUpload);
            return true;
        }

        // check if we shall upload
        if ((!mIsUploaded) && loadContent) {
            // check if we are allowed to register the loading scene.
            mPartialLoadState = InitialState;
            // first step of async load
            bool continuePartialLoad = PartialLoad(forceUpload);
            // if we shall not continue the reset the current loading scene.
            if (continuePartialLoad) {
                (void)GetViewHandler()->SchedulePartialLoad(this, forceUpload);
                }
                // if it is loaded, or loading was started then return true
                rc = (mScene2DContext!=0);
                if(rc) {
                    rc = mIsUploaded || (mPartialLoadState==UploadingState) || (mPartialLoadState==AssetLoadingState);
                }
            return rc;
        }

        if (mIsUploaded && (!loadContent)) {
            return LoadContent(loadContent, false);
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool ViewScene2D::IsContentInitialized() const
{
    return mScene2DContext!=0;
}

// ------------------------------------------------------------------------
bool ViewScene2D::IsContentLoaded() const
{
    return IsContentInitialized() && mIsUploaded;
}

// ------------------------------------------------------------------------
void ViewScene2D::EnableRendertargets(bool enable, bool forceUpload)
{
    if (IsContentInitialized()) {
        for (FeatStd::SizeType i = 0; i < mCameras.Size(); ++i) {
            Renderer * renderer = GetViewHandler()->GetRenderer();
            if (0 != renderer) {
                renderer->SetIs2DRequest(true);
            }
            EnableRenderTarget(enable, mCameras[i]->GetRenderTarget(), forceUpload);
        }
        if (enable && mClearOnSceneLoading) {
            Clear(false, FeatStd::Optional<Candera::Rectangle>());
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::Activate(bool activate)
{
    if (activate != mConsecutiveActivate) {
        if (!IsContentInitialized()) {
            // no need of ref count for cameras, as the content is not initialized
            OnActivate(activate);
            mIsActive = activate;
            OnPostActivate(activate);
        }
        else {
            ActivateImpl(activate);
        }
        mConsecutiveActivate = activate;
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::EnableRendering(bool enable)
{
    if (enable != mConsecutiveRenderingEnabled) {
        EnableRenderingImpl(enable);
        mConsecutiveRenderingEnabled = enable;
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::DeleteRenderJobs()
{
    Renderer * renderer = GetViewHandler()->GetRenderer();
    if ((renderer != 0) && (mScene2DContext != 0)) {
        for (FeatStd::SizeType idx = 0; idx < mCameras.Size(); idx++) {
            Camera2D * camera = mCameras[idx];
            renderer->DeleteRenderJob(camera);
        }
        for (FeatStd::SizeType idx = 0; idx < mClearingData2DVct.Size(); idx++) {
            if (mClearingData2DVct[idx] != 0) {
                Camera2D * camera = mClearingData2DVct[idx]->GetClearCamera();
                if (camera != 0) {
                    renderer->DeleteRenderJob(camera);
                }
            }
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::ReleaseSceneContext()
{
}

// ------------------------------------------------------------------------
void ViewScene2D::Accept(ViewVisitor & visitor)
{
    visitor.Visit(this);
}

// ------------------------------------------------------------------------
void ViewScene2D::Clear(bool onlyCameraViewPorts, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    Renderer* renderer = GetViewHandler()->GetRenderer();
    if (0 != renderer) {
        RTPtrVector renderTargets;
        if (GetRenderTargetPtrVector(renderTargets,true) && (renderTargets.Size() > 0)) {
            for (SizeType i = 0; i < renderTargets.Size(); ++i) {
                Candera::RenderTarget *renderTarget = renderTargets[i];
                Courier::Gdu* gdu = renderer->GetGdu(renderTarget);
                if (0 != gdu) {
                    if (onlyCameraViewPorts)
                    {
                        if (!mClearingDataInitFlag.Find(renderTarget))
                        {
                            // create the vector of clear data only once with first call of Clear(true)
                            InitClearingData2D(gdu);
                            mClearingDataInitFlag.Insert(renderTarget, true);
                        }
                        else
                        {
                            // Update clear data for updated cameras (if any)
                            UpdateClearingData2D();
                        }
                        for (FeatStd::SizeType idx = 0; idx < mClearingData2DVct.Size(); ++idx) {
                            ScheduleCamera(mClearingData2DVct[idx]->GetClearCamera(), renderer, dirtyArea, GetViewHandler(), gdu);
                        }
                    }
                    else
                    {
                        Camera2D* clearCamera = gdu->GetClearCamera2D();
                        ScheduleCamera(clearCamera, renderer, dirtyArea, GetViewHandler(), gdu);
                    }
                }
            }
        }
    }
}

// ------------------------------------------------------------------------
FrameworkWidget * ViewScene2D::GetTouchedWidget2D(ViewScene2D * view2D, const TouchInfo & info)
{
    if((view2D!=0) && view2D->IsActive()) {
        const ViewScene::FrameworkWidgetPtrVector & widgets = view2D->GetFrameworkWidgetPtrVector();
        const ViewScene2D::CameraPtrVector & cameras = view2D->GetCameraPtrVector();
        for (SizeType i=0; i<widgets.Size(); ++i) {
            FrameworkWidget * fw = widgets[i];
            if(fw->IsTouchable()) {
                Candera::WidgetBase* wb = static_cast<Candera::WidgetBase*>(fw);
                Candera::Widget2D * w = Candera::Dynamic_Cast<Candera::Widget2D*>(wb);
                if (w!=0) {
                    Node2D * node = w->GetNode();
                    if(node!=0) {
                        for(FeatStd::SizeType idx=0; idx<cameras.Size(); idx++) {
                            Camera2D * camera2D = cameras[idx];
                            if(camera2D->IsRenderingEnabled()) {
                                const Candera::Vector2 point(static_cast<Courier::Float>(info.mX), static_cast<Courier::Float>(info.mY));
                                if(node->IsPickIntersectingBoundingRectangle(*camera2D, point)) {
                                    return fw;
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return 0;
}

// ------------------------------------------------------------------------
FrameworkWidget * ViewScene2D::GetTouchedWidget(const TouchInfo & info)
{
    if(mGetTouchedWidget2DFunction!=0) {
        return mGetTouchedWidget2DFunction(this,info);
    }
    return 0;
}

// ------------------------------------------------------------------------
class FindAnimationTraverser2D : public Candera::TreeTraverserBase<Node2D>
{
public:
    FindAnimationTraverser2D(const CompositePath & compositePath, const ItemId & animationId) : mCompositePath(compositePath), mAnimationId(animationId) { }

    Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> GetAnimationPlayer() { return mAnimationPlayer; }

protected:
    virtual TraverserAction ProcessNode(Node2D& node)
    {
        CompositeGroup2D* composite = Dynamic_Cast<CompositeGroup2D*>(&node);
        if (0 != composite) {
            if (CompositePath(composite->GetStringId()) == mCompositePath) {
                mAnimationPlayer = composite->GetAnimationByName(mAnimationId.CStr());
                return StopTraversing;
            }
        }
        return ProceedTraversing;
    }

private:
    CompositePath mCompositePath;
    ItemId mAnimationId;
    Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> mAnimationPlayer;
};

Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> ViewScene2D::GetAnimation(const CompositePath & compositePath, const ItemId & animationId)
{
    FindAnimationTraverser2D traverser(compositePath, animationId);
    if (0 != mScene2DContext) {
        traverser.Traverse(*mScene2DContext->GetScene());
    }
    return traverser.GetAnimationPlayer();
}

// ------------------------------------------------------------------------
void ViewScene2D::ScheduleCamera(Camera2D* camera, Renderer* renderer, const FeatStd::Optional<Candera::Rectangle>& dirtyArea, IViewHandler* viewHandler, const Gdu* gdu)
{
    if ((0 != camera) && (0 != renderer)){
        renderer->Queue2DJob(this, camera, cDefaultRender2DCounter, dirtyArea, true);
        if ((mInvalidateCamerasOnSceneClearing) && (0 != viewHandler) && (0 != gdu)) {
            viewHandler->Invalidate(gdu->GetRenderTarget2D(), dirtyArea);
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::CleanUp()
{
    mScene2DContext = 0;

    for (SizeType i = 0; i< mClearingData2DVct.Size(); ++i) {
        FEATSTD_DELETE(mClearingData2DVct[i]); //Finalize will be called in the destructor
    }
    mClearingData2DVct.Clear();
    mCameraToClearingData.Clear();
    mRefCounterVct.Clear();
}

// ------------------------------------------------------------------------
void ViewScene2D::ActivateImpl(bool activate)
{
    for (SizeType i = 0; i < mCameras.Size(); ++i) {
        ActivateForCamera(activate, mCameras[i]);
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::EnableRenderingImpl(bool enable)
{
    for (SizeType i = 0; i < mCameras.Size(); ++i) {
        EnableRenderingForCamera(enable, mCameras[i]);
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::EnableRenderingScene(bool enable)
{
    if (0 != mScene2DContext) {
        Scene2D * scene = mScene2DContext->GetScene();
        if (0 != scene) {
            scene->SetRenderingEnabled(enable);
        }
    }
}

// ------------------------------------------------------------------------
void ViewScene2D::UpdateCameraRenderingState(bool enable, Candera::CanderaObject* camera)
{
    Camera2D* camera2D = static_cast<Camera2D*>(camera);
    if ((0 != camera2D) && (enable != camera2D->IsRenderingEnabled())) {
        camera2D->SetRenderingEnabled(enable);
    }
}

void ViewScene2D::InitClearingData2D(const Gdu* gdu)
{
    for (CameraPtrVector::ConstIterator it = mCameras.ConstBegin(); it != mCameras.ConstEnd(); ++it)
    {
        Camera2D *camera = (*it);
        if (gdu == GetViewHandler()->GetRenderer()->GetGdu(camera->GetRenderTarget())) {
            ClearingData2D* clearData = FEATSTD_NEW(ClearingData2D);
            if (0 != clearData) {
                clearData->Init(gdu);
                clearData->SetupClearCamera(camera);

                static_cast<void>(mClearingData2DVct.Add(clearData));
                static_cast<void>(mCameraToClearingData.Insert(camera, clearData));
            }
        }
    }
}

}
#endif
