//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_ViewScene_h)
#define Courier_Visualization_ViewScene_h

#include <Candera/Candera.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget.h>
#include <Courier/Visualization/View.h>

#include <CanderaWidget/WidgetBase/WidgetBase.h>
#include <CanderaAssetLoader/AssetLoaderBase/SceneContextBase.h>
#include <Candera/EngineBase/Common/TreeTraverserBase.h>

namespace Courier {

class Gdu;

/// @addtogroup COURIER_VISUALIZATION
/// @{
/** A ViewScene is a specialized View which is the base class for ViewScene2D and ViewScene3D.
*/
class ViewScene : public View
{
    friend class CameraGroupHandler;

    typedef View Base;

    protected:
        class WidgetComparerLessByName
        {
            public:
                bool operator()(const FrameworkWidget * a, const FrameworkWidget * b) const {
                    const Candera::WidgetBase * aWidget = static_cast<const Candera::WidgetBase *>(a);
                    const Candera::WidgetBase * bWidget = static_cast<const Candera::WidgetBase *>(b);
                    return FeatStd::Internal::String::CompareStrings(aWidget->GetName(),bWidget->GetName())<0;
                }
        };

        /// Struct for holding data for activation and rendering requests.
        struct CameraRefCounter
        {
            CameraRefCounter():
                m_camera(0),
                m_activateRefCount(0),
                m_renderingRefCount(0) {}

            CameraRefCounter(Candera::CanderaObject* cameraPtr):
                m_camera(cameraPtr),
                m_activateRefCount(0),
                m_renderingRefCount(0) {}

            ~CameraRefCounter()
            {
                m_camera = 0;
            }

            void UpdateActivateRefCounter(bool activate);
            void UpdateRenderingRefCounter(bool enable);

            Candera::CanderaObject* m_camera;
            FeatStd::SizeType m_activateRefCount;
            FeatStd::SizeType m_renderingRefCount;
        };

        ///
        enum RefCounterType {
            /// Reference counter for activation requests.
            ACTIVATE_REF_COUNTER,
            /// Reference counter for rendering requests.
            RENDERING_REF_COUNTER
        };

    public:
        typedef Vector< FrameworkWidget *> FrameworkWidgetPtrVector;

        ///
        enum LoadError {
            /// Successful action
            LoadSuccess,
            /// An error while loading from asset happened.
            AssetLoadError,
            /// An upload error happened.
            VRamUploadError,
            /// An unload error happened.
            VRamUnloadError,
            /// Another error occurred.
            PendingLoadingError,
            /// Another error occurred.
            GeneralError
        };

        enum WidgetSortingStrategy {
            /// The widget order is defined by the asset. No sorting is applied.
            NoneWidgetSorting,
            /// The widgets are sorted by their name.
            NameWidgetSorting
        };

    public:
        ///
        ViewScene(bool managed);

        ///
        virtual ~ViewScene();

        ///
        virtual bool Init(IViewHandler * viewHandler, const Char * viewName, ViewController * viewController);

        ///
        virtual bool Init(IViewHandler * viewHandler, const Char * viewName);

        /** Returns the string identification of the view.
            @return <em>string</em> the string identifier, in this case the name of the scene. */
        virtual const ViewId & GetId() const;

        /** Activates the view for message receiving, regardless of their rendering state.
            Optional ViewController and widegts are informed by calling method OnParentViewActivate.
            @param activate Activates or deactivates message receiving. */
        virtual void Activate(bool activate);

        /** Activates / deactivates the view for message receiving, regardless of their rendering state and
        manages a reference count for activation requests for a specific camera of the view.
        OnActivate which further sends ParentViewActivateEvent event is called only with the first activation request for a camera of the ViewScene
        and with the last deactivation request, after all cameras of the ViewScene have their associated reference count on 0.
        @param activate activates or deactivates message receiving;
        @param camera is the camera for which a activation/deactivation request is made.*/
        virtual void ActivateForCamera(bool activate, const Candera::CanderaObject* camera);

        /// Returns true if the view is capable of message receiving.
        virtual bool IsActive() const;

        /** Enables / disables rendering for the current ViewScene and a specific camera and
        manages a reference count for rendering requests for a specific camera of the view.
        OnRenderingEnabled which further sends ParentViewRenderingEnabledEvent is called only
        with the first rendering request for a camera of the ViewScene
        and with the last rendering request, after all cameras of the ViewScene have their associated reference count on 0.
        @param enable enables / disables rendering for the camera;
        @param camera is the camera for which a rendering request is made.*/
        virtual void EnableRenderingForCamera(bool enable, const Candera::CanderaObject* camera);

        /// Returns true if the view is enabled for rendering.
        virtual bool IsRenderingEnabled() const;

        /** Returns the widget object identified by the composite path and the widget identifier.

        @return <em>FrameworkWidget</em> if the widget could be found,
        <em>0</em> otherwise. */
        virtual FrameworkWidget * GetFrameworkWidget(const CompositePath & compositePath, const ItemId & widgetId);
        using View::GetFrameworkWidget;

        ///
        virtual void Update(RenderHint * renderHint);

        /** */
        virtual bool PartialLoad(bool forceUpload);

        /** */
        virtual bool AsyncLoadContent(bool loadContent, bool forceUpload);

        ///
        bool IsUploaded() const { return mIsUploaded; }

        ///
        virtual View * FindView(const ViewId & viewId);

        ///
        virtual bool DistributeMessage(const Message & msg);

        ///
        virtual void OnTransitionStarted(const TransitionStartedEvent & msg);

        ///
        virtual void OnTransitionFinished(const TransitionFinishedEvent & msg);

        /// Enables or disables the clearing of the Rendertargets if a Scene gets loaded (default = true).
        static void SetClearOnSceneLoading(bool clear) { mClearOnSceneLoading = clear; }

        ///
        static bool GetClearOnSceneLoading() { return mClearOnSceneLoading; }

        /// Enables or disables the invalidation of enabled cameras rendering into the cleared Rendertargets of a previous cleared Scene (default = false, for backwards compatibility).
        static void SetInvalidateCamerasOnSceneClearing(bool clear) { mInvalidateCamerasOnSceneClearing = clear; }

        ///
        static bool GetInvalidateCamerasOnSceneClearing() { return mInvalidateCamerasOnSceneClearing; }

        ///
        ViewScene::FrameworkWidgetPtrVector & GetFrameworkWidgetPtrVector() { return mFrameworkWidgets; }

        ///
        const ViewScene::FrameworkWidgetPtrVector & GetFrameworkWidgetPtrVector() const { return mFrameworkWidgets; }

        ///
        static void SetLastLoadError(ViewScene::LoadError lastLoadError) { mLastLoadError = lastLoadError; }

        ///
        static ViewScene::LoadError GetLastLoadError() { return mLastLoadError; }

        /// Set the strategy for the order in which widgets are processed. By default and for backward compatibility the NameWidgetSorting strategy is used.
        static void SetWidgetSortingStrategy(WidgetSortingStrategy widgetSortingStrategy) { mWidgetSortingStrategy = widgetSortingStrategy; }

        /** */
        static WidgetSortingStrategy GetWidgetSortingStrategy() { return mWidgetSortingStrategy; }

        /// Enable/Disable the automatic detection of view relation ship.
        static void SetDetectViewReleationShip(bool detectViewReleationShip) { mDetectViewReleationShip = detectViewReleationShip; }

        ///
        static bool GetDetectViewReleationShip() { return mDetectViewReleationShip; }

        typedef bool(*MessagingTransitionDeactivationHandler)(Courier::ViewScene& viewScene, Candera::CanderaObject* camera);
        static void SetMessagingTransitionDeactivationHandler(MessagingTransitionDeactivationHandler mssagingTransitionDeactivationHandler);

    protected:
        /** */
        enum LoadState {
            /** */
            SynchronousLoadState,
            /** */
            AsynchronousLoadState,
            /** */
            WaitOnAsynchronousInitFinishedState,
            /** */
            InitialState,
            /** */
            AssetLoadingState,
            /** */
            UploadingState,
            /** */
            FinishingState
        };

        enum ViewRelationOperation {
            ///
            AddViewRelation,
            ///
            RemoveViewRelation
        };

        typedef Candera::SceneContextBase::WidgetIterator WidgetIterator;

        template <typename NodeType, typename CompositeType>
        class AddWidgetTraverser : public Candera::TreeTraverserBase<NodeType>
        {
        public:
            AddWidgetTraverser(::Courier::ViewScene* view) : mView(view) { }

        protected:
            typedef ::Candera::TreeTraverserBase<NodeType> Base;
            typedef typename Base::TraverserAction AddWidgetTraverserAction;

            virtual AddWidgetTraverserAction ProcessNode(NodeType& node)
            {
                if (0 == mView) {
                    return Base::StopTraversing;
                }
                CompositeType* composite = Candera::Dynamic_Cast<CompositeType*>(&node);
                if (0 != composite) {
                    mView->AddWidgets(composite->GetWidgetIterator());
                }
                return Base::ProceedTraversing;
            }

        private:
            ::Courier::ViewScene* mView;
        };

#ifdef CANDERA_2D_ENABLED
        friend class AddWidgetTraverser<Candera::Node2D, Candera::CompositeGroup2D>;
        typedef AddWidgetTraverser<Candera::Node2D, Candera::CompositeGroup2D> AddWidgetTraverser2D;
#endif

#ifdef CANDERA_3D_ENABLED
        friend class AddWidgetTraverser<Candera::Node, Candera::CompositeGroup>;
        typedef AddWidgetTraverser<Candera::Node, Candera::CompositeGroup> AddWidgetTraverser3D;
#endif

        /// Is called when a scene is loaded/unloaded from Asset. ViewController and Widgets are informed via OnParentViewLoad.
        virtual void OnInitContent(bool init);

        /// Is called when a scene gets uploaded or unloaded. ViewController and Widgets are informed via OnParentViewLoad.
        virtual void OnLoad(bool load);

        /// Initializes the widgets, fills the widget list and sets the parent view of the widgets.
        virtual void InitWidgets();

        /// Enables / disables the Rendertarget. The upload of the render target can forced by using the flag forceUpload.
        virtual void EnableRendertargets(bool enable, bool forceUpload) = 0;

        /// Fills the camera vector.
        virtual void InitCameras() = 0;

        virtual void OnScopeMaskChanged();
        ///Fill the vector of widgets;
        virtual void FillWidgetList() = 0;

        /// Returns if View is correctly initialized.
        bool IsInitialized();

        /// Frees widget list and resets view parent.
        void ResetWidgets();

        /// Unfocus the focused Widget of the ViewHandler, if the Widget belongs to the ViewScene.
        void UnfocusWidgets();

        /// Delegates ParentViewActivateEvent event to controller and widgets.
        void OnActivate(bool activate);

        void OnPostActivate(bool activate) const;

        /// Delegates ParentViewRenderingEnabledEvent event controller to widgets.
        void OnRenderingEnabled(bool enable);

        ///
        void EnableRenderTargets(bool enable, const RTPtrVector & rtPtrVector, bool forceUpload);

        ///
        void EnableRenderTarget(bool enable, Candera::RenderTarget* renderTarget, bool forceUpload);

        ///
        void AddWidgets(const WidgetIterator& widgetIterator);

        ///
        void ChangeViewGduRelation(ViewRelationOperation op, Gdu* gdu);

        ///
        void ChangeViewSurfaceRelation(ViewRelationOperation op, const Candera::Surface* surface);

        ///
        void ChangeViewRenderTargetRelation(ViewRelationOperation op, Candera::RenderTarget* renderTarget);

        /// Framework widgets belonging to the view.
        FrameworkWidgetPtrVector mFrameworkWidgets;

        /// Activation flag.
        bool mIsActive;

        /// Is Scene uploaded.
        bool mIsUploaded;

        /// The name of the scene and also the id of the view.
        ViewId mSceneName;

        ///
        static bool mClearOnSceneLoading;

        ///
        static bool mInvalidateCamerasOnSceneClearing;

        ///
        static ViewScene::LoadError mLastLoadError;

        ///
        static WidgetSortingStrategy mWidgetSortingStrategy;

        ///
        static bool mDetectViewReleationShip;

        /// bool flag used to insure that OnRenderingEnabled is called only when the first camera of the scene is enabled
        /// or all cameras of the view scene are disabled.
        bool mRenderingEnabled;

        /// bool flag used to insure that consecutive calls of EnableRendering with the same value are processed only the first time.
        bool mConsecutiveRenderingEnabled;

        /// bool flag used to insure that consecutive calls of Activate with the same value are processed only the first time.
        bool mConsecutiveActivate;

        typedef Courier::Vector<CameraRefCounter> CameraRefCounterVct;
        CameraRefCounterVct mRefCounterVct;

        static CameraRefCounter* FindInRefCounterVct(CameraRefCounterVct& refCounterVct, const Candera::CanderaObject* camera);
        static bool AllRefCountersZero(CameraRefCounterVct& refCounterVct, const RefCounterType& type);
        static void ResetRefCounterVct(CameraRefCounterVct& refCounterVct, const RefCounterType& type);

        virtual void UpdateViewSceneRenderingState(bool enable);
        virtual void UpdateCameraRenderingState(bool enable, Candera::CanderaObject* camera);

        /// Sets all data related to activation to default values;
        virtual void ResetActivateData();

        /// Sets all data related to rendering to default values;
        virtual void ResetRenderingData();

        FEATSTD_MAKE_CLASS_UNCOPYABLE(ViewScene);
};

/**
* ViewActivationEvent will be emitted whenever a ViewScene is activated/deactivated.
*/
class ViewActivationEvent : public FeatStd::Event
{
public:
    FEATSTD_RTTI_DECLARATION();

    /**
    * ViewActivationEvent default constructor.
    */
    ViewActivationEvent(bool active, const ViewScene& view) :
        m_active(active),
        m_view(view)
    {
    }

    /**
    * ViewActivationEvent destructor.
    */
    virtual ~ViewActivationEvent()
    {
    }

    bool IsActive() const
    {
        return m_active;
    }

    const ViewScene& GetView() const
    {
        return m_view;
    }

private:
    bool m_active;
    const ViewScene& m_view;
    FEATSTD_MAKE_CLASS_UNCOPYABLE(ViewActivationEvent);
};

/// @}
}

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {
    template<> UInt32 StringBufferAppender< ::Courier::ViewScene::LoadError >::Append(StringBuffer& stringBuffer, ::Courier::ViewScene::LoadError const & object);
}
#endif
#endif
