//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_ViewId_h)
#define Courier_Visualization_ViewId_h

#include <Courier/Util/FixedSizeString.h>
#include <Courier/Util/HashId.h>
#include <FeatStd/Platform/String.h>
#include <FeatStd/Util/StringBuffer.h>

// ----------------------------------------------------------------------------
namespace Courier {
/// @addtogroup COURIER_VISUALIZATION
/// @{

/** The ViewId class identifies a view.
    This class is used to identify a view, either by its name or by its hash value which is computed out of the view name.
    The view name must point to a valid string buffer, usually into the asset buffer or to another allocated string buffer.
    The name is always containing the full qualified path including the name of the item. It is used for sorting the views
    by their names inside a view container.
    View lookup is done by comparing hash values as this usually faster than comparing the string.
*/
class ViewId
{
    public:
        ///
        ViewId();

        ///
        explicit ViewId(const Char * viewName, bool createCopyOfViewName = false);

        ///
        ViewId(const ViewId & viewId);

        ~ViewId();

        ///
        bool operator==(const ViewId & id) const {
            return id.mHashId==mHashId;
        }
        ///
        bool operator!=(const ViewId & id) const {
            return id.mHashId!=mHashId;
        }
        ///
        bool operator>(const ViewId & id) const {
            return FeatStd::Internal::String::CompareStrings(mId,id.mId)>0;
        }
        ///
        bool operator<(const ViewId & id) const {
            return FeatStd::Internal::String::CompareStrings(mId,id.mId)<0;
        }
        ///
        const Char * CStr() const {
            return mId;
        }
        ///
        const HashId GetHashId() const {
            return mHashId;
        }

    private:
        ///
        const Char * mId;
        ///
        HashId mHashId;
        ///
        bool mViewNameCopied;
};

/// @}
}

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {
    template<> UInt32 StringBufferAppender<Courier::ViewId>::Append(StringBuffer& stringBuffer, Courier::ViewId const & object);
} // namespace FeatStd
#endif
#endif
