//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_ViewHandler_h)
#define Courier_Visualization_ViewHandler_h

#include <Courier/Visualization/IViewHandler.h>
#include <Courier/Visualization/ViewContainer.h>
#include <Courier/Visualization/ViewAccessor.h>
#include <Courier/Visualization/AnimationInvalidator.h>
#include <Courier/Visualization/CameraGroupHandler.h>
#ifdef CANDERA_TRANSITIONS_ENABLED
#include <Courier/Transitions/CanderaTransitionHandler.h>
#endif
#include <Candera/EngineBase/Animation/AnimationPlayerListener.h>
#include <Candera/System/GlobalizationBase/CultureChangeListener.h>
#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/Thread.h>
#endif

namespace Candera {
    class DefaultAssetProvider;
}

namespace Courier { namespace UnitTest {
    class ViewHandlerWrapper;
}}

namespace Courier {
    class ViewControllerFactory;
    class ViewScene;
    class ViewVisitor;
    class Renderer;
    class TransitionHandler;
    class RenderHint;
    class ViewHandlerSession;
}

#if defined(COURIER_RENDERING_MONITOR_ENABLED)

namespace Courier {
    class RenderingMonitor;
}

#endif


namespace Courier {
/// @addtogroup COURIER_VISUALIZATION
/// @{
/** ViewHandler is the implementation of the IViewHandler interface and encapsulates Candera functionality.
    It also holds a hierachical tree view structure, whereas the ViewHandler can be seen as the root. It contains
    a ViewContainer which holds an alphabetical sorted list of Views. Each of this Views is either a ViewContainer
    which again hold a list of Views, or ViewScenes (2D or 3D) which are representing a Scene. Those ViewScenes are
    usually the leafs of the tree. Anyway also a flat structure where only ViewScenes are held by the ViewHandler is
    possible and depends on the structure defined by the SceneComposer and its generated asset.

    For example: SceneComposer structure is consisting of following Folders and Scenes

    * - Scenes
    *     - Group1
    *         - SubGroup
    *              - FirstScene2D
    *         - FirstScene3D
    *         - SecondScene3D
    *     - Group2
    *         - SubGroup
    *              - Scene2D_1
    *              - Scene2D_2
    *         - FourthScene3D
    *         - ThirdScene3D
    *         - SecondScene2D

    is represented by following view instances and their ids:

    * - ViewContainer("Scenes")
    *     - ViewContainer("Scenes#Group1")
    *         - ViewContainer("Scenes#Group1#SubGroup")
    *              - ViewScene2D("Scenes#Group1#SubGroup#FirstScene2D")
    *         - ViewScene3D("Scenes#Group1#FirstScene3D")
    *         - ViewScene3D("Scenes#Group1#SecondScene3D")
    *     - ViewContainer("Scenes#Group2")
    *         - ViewContainer("Scenes#Group2#SubGroup")
    *              - ViewScene2D("Scenes#Group2#SubGroup#Scene2D_1")
    *              - ViewScene2D("Scenes#Group2#SubGroup#Scene2D_2")
    *         - ViewScene3D("Scenes#Group2#FourthScene3D")
    *         - ViewScene3D("Scenes#Group2#ThirdScene3D")
    *         - ViewScene2D("Scenes#Group2#SecondScene2D")

    The sorting of the Views inside a ViewContainer allow to control the process flow, e.g when rendering of the
    cameras is done. The naming of the folders and scenes has therefore influence to the render order and additional processing
    which relies on the tree structure.
*/

class ViewHandler : public IViewHandler
{
    friend class Renderer;
    friend class AnimationInvalidator;
    friend class ViewHandlerSession;
    friend class Courier::UnitTest::ViewHandlerWrapper;

    public:
        /**
         *  Declaration of callback function FontStoreProviderCallbackFunction. See function SetFontStoreProviderCallback for further details.
         */
        typedef Candera::TextRendering::FontStore * (*FontStoreProviderCallbackFunction)();

        ///
        ViewHandler();

        ///
        virtual ~ViewHandler();

        /// Frees resources.
        virtual void Finalize();

        /** Initializes the ViewHandler.
            @param viewFactory ViewFactory used for view creation by the ViewHandler.
            @param assetConfiguration AssetConfiguration used by the ViewHandler.
            @param renderer Renderer used by the ViewHandler.
            @param rootPath defines the root path inside the scene composer asset for view creation, e.g. MyModule#Scenes.
            @return <em>true</em> if initialization was successful,
                    <em>false</em> otherwise. */
        virtual bool Init(ViewFactory * viewFactory, Candera::AssetConfig * assetConfiguration, Renderer * renderer, const Char * rootPath);

        ///
        virtual bool Init(ViewFactory * viewFactory, Candera::AssetConfig * assetConfiguration, Renderer * renderer);

        /// Traverses Update thoughout the view tree.
        virtual void Update(RenderHint * renderHint);

        /** Renders all cameras needed to be rendered. Rendering is done over all views and their child views.
            @param renderHint hint object which control the rendering of views.
            @param renderAll true if all Views will be rendered, false if only invalidated views shall be rendered.
            @return <em>true</em> if rendering was executing,
                    <em>false</em> otherwise. */
        virtual bool Render(RenderHint * renderHint, bool renderAll);

        ///
        virtual bool Render(RenderHint * renderHint);

        /** Is called at the beginning of the Render method.
            @param renderHint hint object which indicates if 2D, 3D or both are rendered. */
        virtual void OnPreRender(RenderHint * renderHint);

        /** Is called at the end of the Render method.
            @param renderHint hint object which indicates if 2D, 3D or both are rendered. */
        virtual void OnPostRender(RenderHint * renderHint);

        /** Executes operations on animations.
            Please refer to the corresponding Courier::AnimationReqMsg.
            @return <em>true</em> if execution was successful,
                    <em>false</em> otherwise. */
        virtual bool ExecuteAnimationAction(AnimationAction::Enum action, const ViewId & viewId, const CompositePath & compositePath, const ItemId & animationId, const AnimationProperties & properties);

        /** Executes operations on views.
            Please refer to the corresponding Courier::ViewReqMsg.
            @return <em>true</em> if execution was successful,
                    <em>false</em> otherwise. */
        virtual bool ExecuteViewAction(ViewAction::Enum action, const ViewId & viewId, bool loadScene, bool initScene);

        /** Executes transitions. Following operations are currently defined:
            Please refer to the corresponding Courier::TransitionReqMsg.
            @return <em>true</em> if command execution was successful,
                    <em>false</em> otherwise. */
        virtual bool ExecuteTransitionAction(TransitionAction::Enum action, const ItemId & transitionId,
                                             const ViewId & firstViewId, const ViewId & secondViewId, const Payload & optionalPayload);
#ifdef CANDERA_TRANSITIONS_ENABLED
        /** Creates the Candera transition request fragments based on the CanderaTrRequestAction param.
            Please refer to the corresponding Courier::CanderaTransitionRequestReqMsg.
            @return <em>true</em> if command execution was successful,
                    <em>false</em> otherwise.*/
        virtual bool ExecuteCanderaTransitionReqAction(CanderaTransitionRequestAction::Enum action,
                                               const Candera::Transitions::Identifier& identifier,
                                               const Candera::Transitions::Hint& hint);

        /** Idicates beginning or ending composing a transition request based on the CanderaTrControlFlowAction param.
            Please refer to the corresponding Courier::CanderaTransitionControlFlowReqMsg.
            @return <em>true</em> if command execution was successful,
                    <em>false</em> otherwise.*/
        virtual bool ExecuteCanderaTransitionControlFlowAction(CanderaTransitionControlFlowAction::Enum action);
#endif

        /** Changes the theme, views have to be reloaded afterwards. An empty themeId resets to the default theme.
            Before this call can be done, views (scenes) have to be unloaded, after this call views (scenes) have to be loaded again.
            @param themeId the theme to be used when loading a scene.
            @return <em>true</em> if switching was successful,
                    <em>false</em> otherwise. */
        virtual bool ChangeTheme(const ItemId & themeId);

        /// Accessing the content loader.
        virtual Candera::ContentLoader * GetContentLoader();

        /// Accessing the animation time dispatcher.
        virtual Candera::Animation::AnimationTimeDispatcher * GetAnimationTimeDispatcher();

        /// Accessing the animation time dispatcher.
        virtual Candera::Animation::AnimationTimeDispatcher::SharedPointer GetAnimationTimeDispatcherSharedPointer();

        /// Returns the used AssetProvider.
        virtual Candera::AssetProvider * GetAssetProvider();

        /// Delegates invalidation to the Renderer object.
        FEATSTD_LINT_NEXT_EXPRESSION(1735, "Default argument on virtual function matches the base-class argument it overrides. (See IViewHandler.h)")
        virtual void Invalidate(Candera::RenderTarget * renderTarget, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

        /** Causes Invalidating of all Views/Cameras.
            @param forceInvalidateAllRendertargets controls if all rendertargets shall be invalidated.
            @param dirtyArea
            @note If marking all Gdu objects as invalidated this will cause swapping of all render targets. */
        FEATSTD_LINT_NEXT_EXPRESSION(1735, "Default argument on virtual function matches the base-class argument it overrides. (See IViewHandler.h)")
        virtual void InvalidateAll(bool forceInvalidateAllRendertargets, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

        /// Delegates layer activation to the Renderer object.
        virtual bool ActivateLayer(Int layerId, bool activate);

        /** Release a View object. Releasing is delegated to the View factory.
            @param view the View to be released. */
        virtual void DestroyView(View * view);

        /** Adds a view object to the ViewHandlers own view container.
            If the parent views (ViewContainers) are not existing, they will be created dynamically.
            In case this shall not be done automatically, the caller has to ensure that the parent views
            has been added before by calling this method. E.g if a view identifed by "Scenes#Test#Scene2d"
            and 'Scenes' and 'Test' ViewContainers are not existing they will be created automatically.
            @param view the view object.
            @return <em>true</em> if adding was successful,
                    <em>false</em> otherwise. */
        virtual bool AddView(View * view);

        /** Removes a view object from the ViewHandlers own view container.
            @param view the view object.
            @return <em>true</em> if removing was successful,
                    <em>false</em> otherwise. */
        virtual bool RemoveView(View * view);

        /** Returns a view object if already created.
            @param viewId the Id of the view.
            @return <em>View</em> if the view is already created and can be accessed,
                    <em>0</em> otherwise. */
        virtual View * FindView(const ViewId & viewId);

        /** Returns the parent view of an view object.
            @param viewId the Id of the view.
            @return <em>View</em> the parent view of the view represented by the viewId,
                    <em>0</em> if there is no parent view, in case the views are held by the ViewHandler itself. */
        virtual View * FindParentView(const ViewId & viewId);

        /** Returns the widget of a specified view, identified by the widget name.
            @param viewId the Id of the view.
            @param compositePath the node path to the composite in which the widget is located or ItemId() if it is located directly in a scene.
            @param widgetId the name of the widget.
            @return <em>FrameworkWidget</em> the widget identified both by the view id and the widget name.
                    <em>0</em> if no widget can be found. */
        virtual FrameworkWidget * FindWidget(const ViewId & viewId, const CompositePath & compositePath, const ItemId & widgetId);

        /** Is called when a message shall be distributed through the view tree and its views and widgets,
            Additionally the message is sent to the optional ViewHandlerSession object for processing purposes
            and delegated toe the focused message if the focus tag is set inside the message.
            @param msg Message object to be processed.
            @return <em>true</em> if the message is consumed
                    <em>false</em> if the message shall be sent to following message receiving entities. */
        virtual bool OnMessage(const Message & msg);

        /** Sets the value of a property of a widget.
            @param viewId the Id of the view.
            @param compositePath the node path to the composite in which the widget is located or ItemId() if it is located directly in a scene.
            @param widgetId the name of the widget.
            @param propertyId the name of the property.
            @param value the value to be set.
            @return <em>true</em> if setting was successful,
                    <em>false</em> otherwise. */
        virtual bool SetWidgetProperty(const ViewId & viewId, const CompositePath & compositePath, const ItemId & widgetId, const ItemId & propertyId, const Char * value);

        /** Sets the focus to a widget. In case a widget loses its focus
            an asynchronous message LostFocusIndMsg is sent.
            @param viewId the Id of the view.
            @param compositePath the node path to the composite in which the widget is located or ItemId() if it is located directly in a scene.
            @param widgetId the name of the widget.
            @return <em>true</em> if focus setting was successful,
                    <em>false</em> otherwise. */
        virtual bool SetFocus(const ViewId & viewId, const CompositePath & compositePath, const ItemId & widgetId);

        /** Sets the focus to a widget. In case a widget loses its focus
            an asynchronous message LostFocusIndMsg is sent.
            @param frameworkWidget the FrameworkWidget which will be focused.
            @return <em>true</em> if focus setting was successful,
                    <em>false</em> otherwise. */
        virtual bool SetFocus(FrameworkWidget* frameworkWidget);

        /** Method is called if a focused widget loses its focus because its view gets deactivated.
            This causes an asynchronous message LostFocusIndMsg to be sent.
            @param widget the FrameworkWidget which loses the focus.
            @return <em>true</em> if processing was successful,
                    <em>false</em> otherwise. */
        virtual bool LoseFocus(FrameworkWidget * widget);

        /** Sets the optional begin render visitor. If set the visitor visits all views
            before renderings starts.
            @param viewVisitor the visitor instance. */
        void SetBeginRenderVisitor(ViewVisitor * viewVisitor);

        /** Sets the optional finalize render visitor. If set the visitor visits all views
            after the final render call to the Renderer object.
            @param viewVisitor the visitor instance. */
        void SetFinalizeRenderVisitor(ViewVisitor * viewVisitor);

        /** Sets the optional view controller factory. If set it will be used to create ViewController objects
            which are then attachted to Views during their view creation.
            @param viewControllerFactory the factory instance. */
        void SetViewControllerFactory(ViewControllerFactory * viewControllerFactory);

        /** Sets the optional transition handler. If set it will be used to execute user specific transitions.
            @param transitionHandler the transition handler instance. */
        void SetTransitionHandler(TransitionHandler * transitionHandler);

        /** Sets the optional session object: If set it may be used e.g for touch message handling.
            @param viewHandlerSession the session instance. */
        void SetViewHandlerSession(ViewHandlerSession * viewHandlerSession);

        /** Sets the global animation speed factor. This is mainly for debugging purposes.
            @param speedFactor the speed factor which shall be used. */
        void SetAnimationSpeedFactor(FeatStd::Float speedFactor = 1.0F);

#if defined(COURIER_RENDERING_MONITOR_ENABLED)
        /** Sets the optional rendering monitoring.
            @param renderingMonitor the rendering monitor instance. */
        void SetRenderingMonitor(RenderingMonitor * renderingMonitor);
#endif

        /** Returns the world time (time point) which is set before each render call.
            @return <em>UInt32</em> the world time set by DefineWorldTime */
        virtual UInt32 GetWorldTime();

        /// Adds the specified AnimationPlayer
        virtual bool AddAnimationPlayer(Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> player);

        /// Removes the specified AnimationPlayer
        virtual void RemoveAnimationPlayer(Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> player);

        /// Returns the the specified AnimationPlayer
        Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> GetAnimationPlayer(const ItemId & animationId);

        /// Access to the used Renderer instance, e.g Transitions may use this object directly.
        virtual Renderer * GetRenderer();

        /// Access to the used CameraGroupHandler instance.
        virtual CameraGroupHandler * GetCameraGroupHandler();

        /** Returns 3D frames per second.
            @return <em>UInt16</em> the frames per second. */
        virtual UInt16 GetFramesPerSecond();

        /** Returns the last 3D frame duration in milliseconds.
            Only if 3D is rendering in its own thread, this value is valid for 3D only.
            @return <em>FeatStd::SizeType</em> the duration in milliseconds. */
        virtual FeatStd::SizeType GetLastFrameDuration();

        /** Returns 2D frames per second.
            @return <em>UInt16</em> the frames per second.*/
        virtual UInt16 GetFramesPerSecond2D();

        /** Returns the last 2D frame duration in milliseconds.
            Only if 2D is rendering in its own thread, this value is valid for 2D only.
            @return <em>FeatStd::SizeType</em> the duration in milliseconds. */
        virtual FeatStd::SizeType GetLastFrameDuration2D();

        /** Accepts the provided visitor. The OnBegin and OnEnd methods of the visitor are called and the accept calls
            is delegated to the views. */
        virtual void Accept(ViewVisitor* visitor);

        /** Changes the current culture. */
        virtual bool SetCurrentCulture(const Char * locale);

        /// Calls PartialLoad of the current loading ViewScene.
        virtual bool PartialLoad();

        /// Add a view for partial loading
        virtual void SchedulePartialLoad(ViewScene * view, bool forceUpload);

        /**
         * Set the callback function that is called to obtain the font store.
         * The callback function will be during the Init method call to initialize the font engine.
         * If no callback is set the GetDefaultFontStore function will be called instead.
         */
        static void SetFontStoreProviderCallback(FontStoreProviderCallbackFunction fontStoreProviderCallback) { sFontStoreProviderCallback = fontStoreProviderCallback; }

        /**
         *  Returns a valid FontStore with no additional settings.
         */
        static Candera::TextRendering::FontStore * GetDefaultFontStore();

    protected:
        /// Defines the time point which is used by the animation dispatcher.
        virtual void DefineWorldTime();

        /** Creates a dynamically allocated empty ViewScene2D or ViewScene3D object when using CreateViews method.
            This method might be overriden if View classes derived from ViewScene2D and/or ViewScene3D shall be created.
            @param is2D if an 2D view shall be created, 3D otherwise.
            @return <em>ViewScene</em> a 2D or 3D object. */
        virtual ViewScene * CreateViewScene(bool is2D);

        /** Creates a dynamically allocated ViewContainer when using CreateViews method.
            This method shall be overriden if an own class derived from ViewContainer shall be created.
            @return <em>ViewContainer</em> a ViewContainer object. */
        virtual ViewContainer * CreateViewContainer();

        /** Is called when a message shall be distributed only through the view tree and its views and widgets.
            @param msg Message object to be processed.
            @return <em>true</em> if the message is consumed
                    <em>false</em> if the message shall be sent to following message receiving entities. */
        virtual bool OnMessageViewTreeOnly(const Message & msg);

    private:

        virtual bool SetCurrentLoadingViewSceneInternal(ViewScene *, bool);
        virtual ViewScene * GetCurrentLoadingViewSceneInternal();

        /** SceneEntry objects are used temporarily when creating views automatically using all scenes of an asset. */
        class SceneEntry {
            public:
                ///
                SceneEntry(const Char * name, bool is2D) : mSceneName(name), mIs2D(is2D) {}
                ///
                const Char * mSceneName;
                ///
                bool mIs2D;
                friend class SceneEntryVector;
        };

        ///
        typedef Courier::Vector< SceneEntry > SceneEntryVector;

        /** Fills a SceneEntryVector object with SceneEntry objects used for automatically creating views.
            Therefore all 2D and 3D scene names are retrieved from the given asset. */
        bool RetrieveSceneEntries(SceneEntryVector & sceneEntryVector) const;

        ///
        static bool ContainsSceneEntry(const Char* sceneName, const SceneEntryVector &sceneEntryVector);

        /// Create all views from the content of the SceneEntryVector, where all parent views and child views are created.
        bool CreateViews(const SceneEntryVector & sceneEntryVector, const ViewId & viewId, bool initScene);

        /// Create a view from the content of a SceneEntry.
        bool CreateView(const SceneEntry & sceneEntry, bool initScene);

        /** Creates the parent view of a given View Id. If the parent View is already existing, the View will be returned immediately.
            If the View is not existing it will be dynamically created. If the parent view of an parent view is not existing it will be
            also created until all views of a given path are created. */
        View * CreateParentView(const ViewId & viewId);

        /// Returns if an animation is currently animated.
        bool IsAnimationEnabled();

        ///
        void DetectAnimationScenes(RenderHint * renderHint, bool force);

        /// Stops all running animations.
        bool StopAllRunningAnimations();

        /// Checks if the animation is an animation group
        bool IsAnimationGroup(const ItemId & animationId) const;

        /// Iterates over all animation of an animation group
        void AddPlayerIteration(Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationGroupPlayer> group,
                                Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> player);

        /// Callback which is called if an animation has ended.
        static void AnimationFinishedCallback(Int32 completedIterationsCount, Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> player, void* userData);

        /// Callback which is called if the culture was changed.
        static void CultureChangeCallback(const Candera::Globalization::Culture & culture);

        ///
        ViewAccessor & GetViewAccessor() { return mViewAccessor; }

        ///
        static FontStoreProviderCallbackFunction sFontStoreProviderCallback;

        ///
        View* InitializeViewImpl(View* view, const Char* viewId, bool initScene);

        /// The used Rendering object.
        Renderer * mRenderer;

        /// The used ViewFactory.
        ViewFactory * mViewFactory;

        /// The used ViewControllerFactory.
        ViewControllerFactory * mViewControllerFactory;

        /// DefaultAssetProvider used by the ViewHandler.
        Candera::DefaultAssetProvider * mAssetProvider;

        /// AnimationTimeDispatcher used for processing the animations.
        Candera::Animation::AnimationTimeDispatcher::SharedPointer mAnimationTimeDispatcher;

        class ViewHandlerAnimationPlayerListener : public Candera::Animation::AnimationPlayerListener
        {
            public:
                virtual void OnPastEnd(Candera::Animation::AnimationPlayerBase* animationPlayer, Int32 completedIterationsCount) {
                    Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> sPtr(animationPlayer);
                    ViewHandler::AnimationFinishedCallback(completedIterationsCount, sPtr,0);
                }
        };

        /// AnimationPlayerListener for recognizing the end of an animation.
        ViewHandlerAnimationPlayerListener mAnimationPlayerListener;

        class ViewHandlerCultureChangeListener : public Candera::Globalization::CultureChangeListener
        {
            protected:
                virtual void OnCultureChanged(const Candera::Globalization::Culture & culture) {
                    ViewHandler::CultureChangeCallback(culture);
                }

#ifdef FEATSTD_THREADSAFETY_ENABLED
                virtual void Obtain()
                {
                    (void)FeatStd::Internal::AtomicOp::Inc(m_obtainCounter);
                }

                virtual void Release()
                {
                    (void)FeatStd::Internal::AtomicOp::Dec(m_obtainCounter);
                }

        public:
            virtual void WaitForRelease()
            {
                while (!FeatStd::Internal::AtomicOp::TestAndSet(m_obtainCounter, 0, 0)) {
                    FeatStd::Internal::Thread::Sleep(5);
                }
            }

        private:
            FeatStd::Internal::AtomicOp::Atomic m_obtainCounter;
#endif
        };

        /// ViewHandlerCultureChangeListener for recognizing the culture changes.
        ViewHandlerCultureChangeListener mCultureChangeListener;

        /// An optional ViewVisitor object which can be set, and will be called after the rendering call.
        ViewVisitor * mFinalizeRenderVisitor;

        /// An optional ViewVisitor object which can be set, and will be called before the rendering call starts.
        ViewVisitor * mBeginRenderVisitor;

        /// An optional TransitionHandler object which can be set, and will be used when TransitionReqMsg are used.
        TransitionHandler * mTransitionHandler;

        /// An optional ViewHandlerSession object which can be set, and may be used before messages are processed by the ViewHandler itself.
        ViewHandlerSession * mViewHandlerSession;

#if defined(COURIER_RENDERING_MONITOR_ENABLED)
        /// An optional RenderingMonitor object which can be set.
        RenderingMonitor * mRenderingMonitor;
#endif

        /// Currently focused widget.
        FrameworkWidget * mFocusedFrameworkWidget;

        /// The time point set by each render call.
        UInt32 mWorldTime;

        /// Reference world time used in computing animation time.
        UInt32 mRefWT;

        /// Animation reference world time used in computing animation time.
        UInt32 mAnimationRefWT;

        /// Animation time fragment used in computing animation time.
        Float mAnimationTimeFragment;

        /// Helper method to update the animation related reference times.
        void UpdateRefWorldTime(UInt32 worldTime);

        /// SpeedFactor (default 1.0f)
        FeatStd::Float mAnimationSpeedFactor;

        /// AnimationPlayers used for processing the animations.
        Courier::Vector< Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> > mAnimationPlayers;

        ///
        ViewAccessor mViewAccessor;

        /// The base view container which hold a list of views, either ViewScene2D, ViewScene3D or ViewContainers.
        ViewContainer mViewContainer;

        /// A View pointer vector which hold a list of all dynamically allocated Views.
        ViewPtrVector mMemoryManagedViews;

        ///
        AnimationInvalidator mAnimationInvalidator;

        ///
        CameraGroupHandler mCameraGroupHandler;

        struct PartialLoadData
        {
            PartialLoadData(ViewScene* scene, bool forceUpload) :
                mScene(scene),
                mForceUpload(forceUpload)
            {
            }

            PartialLoadData(const PartialLoadData& partialLoadData) :
                mScene(partialLoadData.mScene),
                mForceUpload(partialLoadData.mForceUpload)
            {
            }

            ViewScene* mScene;
            bool mForceUpload;
        };
        ///
        Vector<PartialLoadData> mCurrentLoadingViewScenes;

        ViewScene * mCurrentLoadingViewScene;
        bool mCurrentLoadingViewSceneForceUpload;

        /// indicate that view handler is in finalization no views transitions or animations can be created/started in this phase
        bool mFinalizing;
        FeatStd::Initialized<bool> mForceAnimationInvalidator;
#ifdef CANDERA_TRANSITIONS_ENABLED
        /// The CanderaTransitionHandler object which is used when CanderaTransitionRequestMsg and CanderaTransitionControlFlowReqMsg are used.
        Internal::CanderaTransitionHandler mCanderaTrHandler;

        /// The time point set for CanderaTransitions.
        UInt32 mCanderaTransitionTime;

        virtual void UpdateCanderaTr();
#endif

        /// Structure for holding data for rendering requests.
        /// Whenever a request counter reaches zero a Courier::LoadReqMsg is posted to unload the view in the next loop iteration
        struct DelayedUnloadRequest
        {
            DelayedUnloadRequest(Courier::ViewId viewId, FeatStd::UInt8 delayCounter) :
                m_viewId(viewId),
                m_delayCounter(delayCounter)
            {
            }

            DelayedUnloadRequest(const DelayedUnloadRequest& delayedUnloadRequest) :
                m_viewId(delayedUnloadRequest.m_viewId),
                m_delayCounter(delayedUnloadRequest.m_delayCounter)
            {
            }

            Courier::ViewId m_viewId;
            FeatStd::UInt8 m_delayCounter;

        private:
            FEATSTD_MAKE_CLASS_STATIC(DelayedUnloadRequest);
        };
        FeatStd::Internal::Vector<DelayedUnloadRequest> m_delayedUnloadRequest;
};

/// @}
}

#endif
