//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ViewHandler.h"

#include "ViewVisitor.h"
#include "ViewFacade.h"
#include "ViewController.h"
#include "AssetConfiguration.h"
#include "AssetAccessor.h"
#include "VisualizationMsgs.h"
#include "ViewFactory.h"
#include "ViewControllerFactory.h"
#include "Renderer.h"

#include <FeatStd/Util/StaticObject.h>
#include <FeatStd/Platform/String.h>

#if defined(COURIER_RENDERING_MONITOR_ENABLED)
    #include "RenderingMonitor.h"
#endif

#include "RenderHint.h"
#include "TransitionHandler.h"
#include "ViewHandlerSession.h"

#include <Courier/Foundation/FoundationMsgs.h>
#include <Courier/Messaging/MessagingMsgs.h>

#include <Courier/Version.h>

#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/ContentLoader.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetFontStore.h>
#include <Candera/System/GlobalizationBase/CultureManager.h>
#include <CanderaBehavior/BehaviorBase/StateMachineBehavior.h>
#include <CanderaAssetLoader/AssetLoader.h>

#ifdef CANDERA_TRANSITIONS_ENABLED
#include <CanderaTransitions/Rule.h>
#endif

#include <Courier/Platform/Ticks.h>

#ifdef COURIER_SCRIPTING_ENABLED
#   include <Courier/Scripting/LuaScriptSystem.h>
#endif

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

#if !defined(COURIER_DEFAULT_ANIMATIONPLAYERS_COUNT)
    FEATSTD_LINT_NEXT_EXPRESSION(1923, "FEATSTD_LOGGER_MAX_BUFFER_SIZE can be controlled from outside and therefore cannot be converted to a constant")
    #define COURIER_DEFAULT_ANIMATIONPLAYERS_COUNT 8
#endif

using namespace Candera;
using Candera::MemoryManagement::SharedPointer;
const UInt32 UPDATE_TIME_PERIOD = 10000;

Candera::TextRendering::FontStore * ViewHandler::GetDefaultFontStore()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(AssetFontStore, s_fontStore);
    return &s_fontStore;
}

ViewHandler::FontStoreProviderCallbackFunction ViewHandler::sFontStoreProviderCallback = 0;

// ------------------------------------------------------------------------
ViewHandler::ViewHandler() : mRenderer(0), mViewFactory(0), mViewControllerFactory(0),
                             mAssetProvider(0), mAnimationTimeDispatcher(0),
                             mFinalizeRenderVisitor(0), mBeginRenderVisitor(0), mTransitionHandler(0), mViewHandlerSession(0),
#if defined(COURIER_RENDERING_MONITOR_ENABLED)
                             mRenderingMonitor(0),
#endif
                             mFocusedFrameworkWidget(0),
                             mWorldTime(0),
                             mRefWT(0), mAnimationRefWT(0), mAnimationTimeFragment(0.0F), mAnimationSpeedFactor(1.0F),
                             mCurrentLoadingViewScene(0), mCurrentLoadingViewSceneForceUpload(true),
                             mFinalizing(false),
                             mForceAnimationInvalidator(false)
#ifdef CANDERA_TRANSITIONS_ENABLED
                             , mCanderaTrHandler(), mCanderaTransitionTime(0)
#endif
{
    if (!mViewContainer.Init(this, "")) {
        FEATSTD_LOG_WARN("View container could not be initialized!");
    }
    if(! mAnimationPlayers.Reserve(COURIER_DEFAULT_ANIMATIONPLAYERS_COUNT)) {
        FEATSTD_PANIC("Can't reserve memory for animation player pointers");
    }
    Candera::Globalization::CultureManager::GetInstance().AddCultureChangeListener(&mCultureChangeListener);
}

// ------------------------------------------------------------------------
ViewHandler::~ViewHandler()
{
    (void)Candera::Globalization::CultureManager::GetInstance().RemoveCultureChangeListener(&mCultureChangeListener);

    mRenderer = 0;
    mViewFactory = 0;
    mViewControllerFactory = 0;
    mAssetProvider = 0;
    mFinalizeRenderVisitor = 0;
    mBeginRenderVisitor = 0;
    mTransitionHandler = 0;
    mViewHandlerSession = 0;
#if defined(COURIER_RENDERING_MONITOR_ENABLED)
    mRenderingMonitor = 0;
#endif
    mFocusedFrameworkWidget = 0;
    mCurrentLoadingViewScene = 0;
    mCurrentLoadingViewSceneForceUpload = true;
}

// ------------------------------------------------------------------------
void ViewHandler::Finalize()
{
    mFinalizing = true;
    if (0 != mTransitionHandler) {
        mTransitionHandler->Finalize();
        mTransitionHandler = 0;
    }
    bool rc = StopAllRunningAnimations();
    FEATSTD_UNUSED(rc);
    mAnimationTimeDispatcher = Candera::Animation::AnimationTimeDispatcher::SharedPointer();

    Candera::TextRendering::FontEngine::Instance().Shutdown();

    mViewContainer.Finalize();
    if (0 != mRenderer) {
        mRenderer->Finalize();
        mRenderer = 0;
    }
    if (0 != mAssetProvider) {
        mAssetProvider->Finalize();
        mAssetProvider = 0;
    }
    mViewAccessor.Finalize();

    AssetAccessor & assetAccessor = AssetAccessor::GetInstance();
    assetAccessor.Finalize();

#if defined(COURIER_RENDERING_MONITOR_ENABLED)
    if (0 != mRenderingMonitor) {
        mRenderingMonitor->Finalize();
    }
#endif

    mFinalizeRenderVisitor = 0;
    mCurrentLoadingViewScene = 0;
    mCurrentLoadingViewSceneForceUpload = true;

#ifdef COURIER_SCRIPTING_ENABLED
    Scripting::CourierLuaScripting::UnbindFromViewHandler();
#endif
}

// ------------------------------------------------------------------------
bool ViewHandler::Init(ViewFactory * viewFactory, Candera::AssetConfig * assetConfiguration, Renderer * renderer)
{
    return Init(viewFactory, assetConfiguration, renderer, "");
}

// ------------------------------------------------------------------------
bool ViewHandler::Init(ViewFactory * viewFactory, Candera::AssetConfig * assetConfiguration, Renderer * renderer, const Char * rootPath)
{
    FEATSTD_DEBUG_ASSERT(viewFactory!=0);
    FEATSTD_DEBUG_ASSERT(assetConfiguration!=0);
    FEATSTD_DEBUG_ASSERT(renderer!=0);
    FEATSTD_DEBUG_ASSERT(rootPath!=0);

    COURIER_LOG_INFO("Using COURIER Version %s", Courier::GetVersion());

    if(viewFactory==0) {
        COURIER_LOG_ERROR("ViewFactory not set.");
        return false;
    }
    if(assetConfiguration==0) {
        COURIER_LOG_ERROR("AssetConfiguration not set.");
        return false;
    }
    if(assetConfiguration->GetAssetRepositoryCount()==0) {
        COURIER_LOG_ERROR("AssetConfiguration has no Candera::AssetRepository added.");
        return false;
    }

    if(renderer==0) {
        COURIER_LOG_ERROR("Renderer not set.");
        return false;
    }

    mViewFactory = viewFactory;
    mViewFactory->SetViewHandler(this);
    mRenderer = renderer;

    Candera::TextRendering::FontStore* fontStore = (0 == sFontStoreProviderCallback) ? GetDefaultFontStore() : (*sFontStoreProviderCallback)();
    if (0 == fontStore) {
        COURIER_LOG_ERROR("FontStore initialization failed.");
        return false;
    }

    bool rc = Candera::TextRendering::FontEngine::Instance().Init(fontStore);
    if (!rc) {
        COURIER_LOG_ERROR("FontEngine initialization failed.");
        return false;
    }

    AssetAccessor & assetAccessor = AssetAccessor::GetInstance();
    rc = assetAccessor.Init(assetConfiguration);
    if(! rc) {
        COURIER_LOG_ERROR("Asset initialization failed.");
        return false;
    }

    mAssetProvider = assetAccessor.GetDefaultAssetProvider();
    rc = (mAssetProvider!=0);
    if(! rc) {
        COURIER_LOG_ERROR("AssetProvider initializated by AssetAccessor.");
        return false;
    }

    mAnimationTimeDispatcher = Candera::Animation::AnimationTimeDispatcher::Create();
    rc = (mAnimationTimeDispatcher != 0);
    if (!rc) {
        COURIER_LOG_ERROR("Animation dispatcher initialization failed.");
        return false;
    }

#ifdef CANDERA_TRANSITIONS_ENABLED
    mCanderaTrHandler.Init(this, mAssetProvider);
    mCanderaTrHandler.SetAnimationTimeDispatcher(mAnimationTimeDispatcher);
#endif

    const AssetDescriptor &assetDescriptor = mAssetProvider->GetAssetDescriptor();

    const Char * platformName = assetDescriptor.GetPlatformName();
    bool isForTarget = assetDescriptor.IsForTarget();
    FEATSTD_UNUSED2(platformName, isForTarget); // if logging is disabled, variable/parameter is not used
    COURIER_LOG_INFO("Asset was generated for platform \"%s\" %s\n",
                      ((platformName!=0) ? platformName : "unknown"),
                      (isForTarget ? "TARGET" : "HOST"));

    rc = mRenderer->Init(mAssetProvider);
    if(! rc) {
        COURIER_LOG_ERROR("Render target initialization failed.");
        return false;
    }

    rc = mViewContainer.Init(this,rootPath);
    if(! rc) {
        COURIER_LOG_ERROR("View Container initialization failed.");
        return false;
    }

    rc = mAnimationInvalidator.Init(this);
    if(! rc) {
        COURIER_LOG_ERROR("AnimationInvalidator initialization failed.");
        return false;
    }

    rc = mCameraGroupHandler.Init(this);
    if(! rc) {
        COURIER_LOG_ERROR("CameraGroupHandler initialization failed.");
        return false;
    }

#ifdef CANDERA_TRANSITIONS_ENABLED
    if (0 != mAssetProvider) {
        if(!mCanderaTrHandler.SetTransitionRuleSet(mAssetProvider->GetTransitionRuleCollection())) {
            COURIER_LOG_WARN("Setting the rule set for CanderaTrasitionHandler faild.");
        }
    }
#endif

    COURIER_LOG_INFO("ViewScene::ClearOnSceneLoading is %s, ViewScene::InvalidateCamerasOnSceneClearing is %s,"
                     " ViewFacade::LoadSceneOnViewActivation is %s, ViewFacade::LoadSceneOnViewEnableRendering is %s",
                     (ViewScene::GetClearOnSceneLoading() ? "TRUE": "FALSE"),
                     (ViewScene::GetInvalidateCamerasOnSceneClearing() ? "TRUE": "FALSE"),
                     (ViewFacade::GetLoadSceneOnViewActivation() ? "TRUE": "FALSE"),
                     (ViewFacade::GetLoadSceneOnViewEnableRendering() ? "TRUE": "FALSE"));

#ifdef COURIER_SCRIPTING_ENABLED
    Scripting::CourierLuaScripting::BindToViewHandler(this);
#endif

    return true;
}

// ------------------------------------------------------------------------
Candera::ContentLoader * ViewHandler::GetContentLoader()
{
    FEATSTD_DEBUG_ASSERT(0!=AssetAccessor::GetInstance().GetContentLoader());
    return AssetAccessor::GetInstance().GetContentLoader();
}

// ------------------------------------------------------------------------
Candera::AssetProvider * ViewHandler::GetAssetProvider()
{
    FEATSTD_DEBUG_ASSERT(mAssetProvider!=0);
    return mAssetProvider;
}

// ------------------------------------------------------------------------
Candera::Animation::AnimationTimeDispatcher * ViewHandler::GetAnimationTimeDispatcher()
{
    FEATSTD_DEBUG_ASSERT(mAnimationTimeDispatcher!=0);
    return mAnimationTimeDispatcher.GetPointerToSharedInstance();
}

// ------------------------------------------------------------------------
Candera::Animation::AnimationTimeDispatcher::SharedPointer ViewHandler::GetAnimationTimeDispatcherSharedPointer()
{
    FEATSTD_DEBUG_ASSERT(mAnimationTimeDispatcher.GetPointerToSharedInstance()!=0);
    return mAnimationTimeDispatcher;
}

// ------------------------------------------------------------------------
Renderer * ViewHandler::GetRenderer()
{
    FEATSTD_DEBUG_ASSERT(mRenderer!=0);
    return mRenderer;
}

// ------------------------------------------------------------------------
CameraGroupHandler * ViewHandler::GetCameraGroupHandler()
{
    FEATSTD_LINT_NEXT_EXPRESSION(1536, "we expose the object, thats what we want")
    return &mCameraGroupHandler;
}

// ------------------------------------------------------------------------
void ViewHandler::SetBeginRenderVisitor(ViewVisitor * viewVisitor)
{
    mBeginRenderVisitor = viewVisitor;
}

// ------------------------------------------------------------------------
void ViewHandler::SetFinalizeRenderVisitor(ViewVisitor * viewVisitor)
{
    mFinalizeRenderVisitor = viewVisitor;
}

// ------------------------------------------------------------------------
void ViewHandler::SetViewControllerFactory(ViewControllerFactory * viewControllerFactory)
{
    mViewControllerFactory = viewControllerFactory;
}

// ------------------------------------------------------------------------
void ViewHandler::SetTransitionHandler(TransitionHandler * transitionHandler)
{
    mTransitionHandler = transitionHandler;
}

// ------------------------------------------------------------------------
void ViewHandler::SetViewHandlerSession(ViewHandlerSession * viewHandlerSession)
{
    mViewHandlerSession = viewHandlerSession;
    if(mViewHandlerSession!=0) {
        (void)mViewHandlerSession->Init(this);
    }
}

// ------------------------------------------------------------------------
#if defined(COURIER_RENDERING_MONITOR_ENABLED)
void ViewHandler::SetRenderingMonitor(RenderingMonitor * renderingMonitor)
{
    mRenderingMonitor = renderingMonitor;
}
#endif

// ------------------------------------------------------------------------
bool ViewHandler::ExecuteAnimationAction(AnimationAction::Enum action, const ViewId & viewId, const CompositePath & compositePath, const ItemId & animationId, const AnimationProperties & properties)
{
    // TODO: use composite path and viewId as soon as it is available in the animation interface
    if (mFinalizing) {
        return false;
    }
    FEATSTD_DEBUG_ASSERT(mAssetProvider!=0);
    FEATSTD_DEBUG_ASSERT(mAnimationTimeDispatcher!=0);

    if ((mAssetProvider!=0) && (mAnimationTimeDispatcher!=0) && (mRenderer!=0)) {
        if (action==AnimationAction::StopAll) {
            return StopAllRunningAnimations();
        }
        else {
            bool isAnimationGroup = IsAnimationGroup(animationId);
            Candera::Animation::AnimationPlayer * singlePlayer = 0;
            Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> animationPlayer;
            Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationGroupPlayer> groupPlayer;
            if (!isAnimationGroup) {
                if (viewId == ViewId()) {
                    Candera::Id id = Candera::Internal::AssetProviderFunctions::GetIdByName(mAssetProvider, AnimationLib, animationId.CStr());
                    animationPlayer = mAssetProvider->GetAnimationById(id);
                }
                else if (compositePath == CompositePath()) {
                    COURIER_LOG_ERROR("Scene local animations are not supported '%s'/''/'%s'", viewId.CStr(), animationId.CStr());
                }
                else {
                    View* view = FindView(viewId);
                    if (0 != view) {
                        animationPlayer = view->GetAnimation(compositePath, animationId);
                    }
                    else {
                        // TODO: support StringIdentifier in Log
                        const FeatStd::Char* compositePathName = (0 == compositePath.ToCString())?"StringIdentifierNotSupportedInLog":compositePath.ToCString();
                        FEATSTD_UNUSED(compositePathName); // if logging is disabled, variable/parameter is not used
                        COURIER_LOG_ERROR("Falied to locate the view of the animation request '%s'/'%s'/'%s'", viewId.CStr(), compositePathName, animationId.CStr());
                    }
                }
                if (!animationPlayer.PointsToNull()) {
                    if ((action==AnimationAction::Start) || (action==AnimationAction::Load) || (action==AnimationAction::ToBegin) || (action==AnimationAction::ToEnd)) {
                        // set AnimationPlayer (not group) specific values here
                        singlePlayer = static_cast<Candera::Animation::AnimationPlayer*>(animationPlayer.GetPointerToSharedInstance());
                        if(action==AnimationAction::ToEnd) {
                            // we artificially change the direction so that we are immediately at the end of the animation
                            AnimationProperties & uglycast = const_cast< AnimationProperties &>(properties);
                            uglycast.RevertDirection();
                        }
                        properties.ApplyProperties(singlePlayer);
                    }
                }
            }
            else {
                Candera::Id id = Candera::Internal::AssetProviderFunctions::GetIdByName(mAssetProvider, AnimationGroupLib, animationId.CStr());
                groupPlayer = Dynamic_Cast<Animation::AnimationGroupPlayer::SharedPointer>(mAssetProvider->GetAnimationById(id));
                animationPlayer = groupPlayer;
            }

            if (animationPlayer!=0) {
                bool success = false;
                switch(action) {
                    case AnimationAction::Load:
                    case AnimationAction::Start:
                    case AnimationAction::ToBegin:
                    case AnimationAction::ToEnd: {
                        RemoveAnimationPlayer(animationPlayer);
                        if (AddAnimationPlayer(animationPlayer)) {
                            // in case we have a group player also add the child animations
                            if(groupPlayer!=0) {
                                AddPlayerIteration(groupPlayer, groupPlayer->GetFirstSuccessor(Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayer>(0), Candera::Animation::AnimationGroupPlayer::WithPrevious));
                            }
                            // set values both for AnimationPlayer and AnimationGroupPlayer
                            Candera::Animation::AnimationPlayerBase * playerBase = static_cast<Candera::Animation::AnimationPlayerBase*>(animationPlayer.GetPointerToSharedInstance());
                            properties.ApplyProperties(playerBase);
                            animationPlayer->SetName(animationId.CStr());
                            (void)animationPlayer->AddAnimationPlayerListener(&mAnimationPlayerListener);
                        }
                        if(action==AnimationAction::Start) {
                            DefineWorldTime();
                            mRenderer->WakeUpAllRenderComponents();
                            return animationPlayer->Start();
                        }
                        if((singlePlayer!=0) && ((action==AnimationAction::ToBegin) || (action==AnimationAction::ToEnd))) {
                            DefineWorldTime();
                            mRenderer->WakeUpAllRenderComponents();
                            bool rc = animationPlayer->Start();
                            RenderHintAll renderHint;
                            DetectAnimationScenes(&renderHint, false);
                            if(rc) {
                                rc = animationPlayer->Stop();
                            }
                            if(action==AnimationAction::ToEnd) {
                                AnimationProperties & uglycast = const_cast< AnimationProperties &>(properties);
                                // we change the direction back, so that we have the original position later on
                                uglycast.RevertDirection();
                                uglycast.ApplyProperties(singlePlayer);
                            }
                            bool scenesInvalidated = mAnimationInvalidator.Invalidate(&renderHint);
                            // In this case render all as before.
                            if (!scenesInvalidated) {
                                // we shall render all enabled cameras and therefore its rendertargets.
                                // no dirty area information is available for animations.
                                // therefore, the complete area has to be marked as dirty.
                                mViewContainer.Invalidate(FeatStd::Optional<Candera::Rectangle>());
                            }
                            return rc;
                        }
                        success = true;
                        break;
                    }
                    case AnimationAction::Stop:
                        success = animationPlayer->Stop();
                        break;
                    case AnimationAction::Finish: {
                        FeatStd::UInt32 timeToFinish = properties.GetTimeToFinish();
                        success = (animationPlayer->Finish((timeToFinish==0) ? 1 : timeToFinish));
                        break;
                    }
                    case AnimationAction::Pause:
                        success = animationPlayer->Pause();
                        break;
                    case AnimationAction::Resume:
                        success = animationPlayer->Resume();
                        break;
                    case AnimationAction::Unload:
                        RemoveAnimationPlayer(animationPlayer);
                        success = true;
                        break;
                    case AnimationAction::Change: {
                        if (! isAnimationGroup) {
                            Candera::Animation::AnimationPlayer * singleAnimPlayer = static_cast<Candera::Animation::AnimationPlayer*>(animationPlayer.GetPointerToSharedInstance());
                            properties.ApplyProperties(singleAnimPlayer);
                        } else {
                            Candera::Animation::AnimationPlayerBase * animPlayerBase = static_cast<Candera::Animation::AnimationPlayerBase*>(animationPlayer.GetPointerToSharedInstance());
                            properties.ApplyProperties(animPlayerBase);
                        }
                        success = true;
                        break;
                    }

                    // because of lint
                    default:
                        success = false;
                        break;
                }
                return success;
            }
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool ViewHandler::IsAnimationGroup(const ItemId & animationId) const
{
    if (0 != mAssetProvider) {
        const AssetDescriptor& assetDescriptor = mAssetProvider->GetAssetDescriptor();
        for (AssetDescriptor::AssetIdIterator animationGroupIt = assetDescriptor.GetAssetIdIterator(AnimationGroupLib); animationGroupIt.IsValid(); ++animationGroupIt) {
            const Char* animationGroupName = mAssetProvider->GetNameById(AnimationGroupLib, *animationGroupIt, 0);
            if (animationGroupName == 0)
            {
                FEATSTD_LOG_DEBUG("GetNameById is not valid");
            }
            if (animationId == ItemId(animationGroupName)) {
                return true;
            }
        }
    }
    return false;
}

// ------------------------------------------------------------------------
void ViewHandler::AddPlayerIteration(Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationGroupPlayer> group, Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> player)
{
    if (player != 0) {
        if(mAnimationTimeDispatcher!=0) {
            if(mAnimationPlayers.Add(player)) {
                (void)mAnimationTimeDispatcher->AddPlayer(player);
            }
        }
        AddPlayerIteration(group, group->GetFirstSuccessor(player, Candera::Animation::AnimationGroupPlayer::WithPrevious));
        AddPlayerIteration(group, group->GetFirstSuccessor(player, Candera::Animation::AnimationGroupPlayer::AfterPrevious));

        Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationGroupPlayer> innerGroup = Dynamic_Cast<FeatStd::MemoryManagement::SharedPointer<Animation::AnimationGroupPlayer> >(player);
        if (!innerGroup.PointsToNull()) {
            AddPlayerIteration(innerGroup, innerGroup->GetFirstSuccessor(Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayer>(0), Candera::Animation::AnimationGroupPlayer::WithPrevious));
        }

        AddPlayerIteration(group, group->GetNext(player));
    }
}

// ------------------------------------------------------------------------
bool ViewHandler::ExecuteTransitionAction(TransitionAction::Enum action, const ItemId & transitionId,
                                          const ViewId & firstViewId, const ViewId & secondViewId, const Payload & optionalPayload)
{
    if (mFinalizing) {
        return false;
    }
    if(mTransitionHandler!=0) {
        DefineWorldTime();
        View * firstView = (firstViewId == ViewId()) ? 0 : FindView(firstViewId);
        View * secondView = (secondViewId == ViewId()) ? 0 : FindView(secondViewId);
        Transition::SetLastTransitionError(Transition::GeneralTransitionError);
        // we allow zero views !!!
        switch(action) {
            case TransitionAction::Start: {
                Transition * transition = mTransitionHandler->Create(transitionId);
                if (0 != transition) {
                    return transition->Start(firstView, secondView, optionalPayload);
                }
                else {
                    Transition::SetLastTransitionError(Transition::CreateTransitionError);
                }
                break;
            }
            case TransitionAction::Reset: {
                Transition * transition = mTransitionHandler->Get(transitionId,firstView,secondView);
                if (0 != transition) {
                    return transition->Reset();
                }
                else {
                    Transition::SetLastTransitionError(Transition::MissingTransitionError);
                }
                break;
            }
            case TransitionAction::Finish: {
                Transition * transition = mTransitionHandler->Get(transitionId,firstView,secondView);
                if (0 != transition) {
                    return transition->Finish();
                }
                else {
                    Transition::SetLastTransitionError(Transition::MissingTransitionError);
                }
                break;
            }
        }
    }
    return false;
}

#ifdef CANDERA_TRANSITIONS_ENABLED
bool ViewHandler::ExecuteCanderaTransitionReqAction(CanderaTransitionRequestAction::Enum action, const Candera::Transitions::Identifier& identifier, const Candera::Transitions::Hint& hint)
{
    mCanderaTrHandler.CreateTransitionRequestFragment(action, identifier, hint);
    return true;
}

bool ViewHandler::ExecuteCanderaTransitionControlFlowAction(CanderaTransitionControlFlowAction::Enum action)
{
    if (CanderaTransitionControlFlowAction::End == action) {
        UpdateCanderaTr();
        mRenderer->WakeUpAllRenderComponents();
    }
    mCanderaTrHandler.CreateControlFlowRequest(action);
    return true;
}
#endif

// ------------------------------------------------------------------------
bool ViewHandler::StopAllRunningAnimations()
{
    bool rc = true;
    if(mAnimationTimeDispatcher!=0) {
        for (Vector<Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> >::Iterator it = mAnimationPlayers.Begin(); it != mAnimationPlayers.End(); ++it) {
            Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> animationPlayer = *it;
            if (animationPlayer!=0) {
                if(mAnimationTimeDispatcher!=0) {
                    (void)mAnimationTimeDispatcher->RemovePlayer(animationPlayer);
                }
                rc = animationPlayer->Stop() && rc;
            }
        }
        mAnimationPlayers.Clear();
    }
    return rc;
}

// ------------------------------------------------------------------------
void ViewHandler::RemoveAnimationPlayer(Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> player)
{
    FEATSTD_DEBUG_ASSERT(player!=0);
    Int idx = 0;
    Int found = -1;
    for (Vector<Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> >::Iterator it = mAnimationPlayers.Begin(); it != mAnimationPlayers.End(); ++it) {
        Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> animationPlayer = *it;
        if (animationPlayer==player) {
            if(mAnimationTimeDispatcher!=0) {
                // mAnimationPlayerListener may not be registered, because animation was only added from outside
                (void)animationPlayer->RemoveAnimationPlayerListener(&mAnimationPlayerListener);
                (void)mAnimationTimeDispatcher->RemovePlayer(animationPlayer);
            }
            found = idx;
            break;
        }
        ++idx;
    }
    if(found>=0) {
        static RenderHintAll hint;
        mAnimationInvalidator.ReleaseScenes(&hint,player.GetPointerToSharedInstance());
        (void)mAnimationPlayers.Remove(found);
        COURIER_LOG_INFO("Removed AnimationPlayer '%s'",player.GetPointerToSharedInstance()->GetName());
    }
}

// ------------------------------------------------------------------------
bool ViewHandler::AddAnimationPlayer(Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> player)
{
    FEATSTD_DEBUG_ASSERT(player!=0);
    if((player!=0) && (mAnimationTimeDispatcher!=0)) {
        if(mAnimationPlayers.Add(player)) {
            COURIER_LOG_INFO("Added AnimationPlayer '%s' ",player.GetPointerToSharedInstance()->GetName());
            return mAnimationTimeDispatcher->AddPlayer(player);
        }
    }
    return false;
}

// ------------------------------------------------------------------------
SharedPointer<Candera::Animation::AnimationPlayerBase> ViewHandler::GetAnimationPlayer(const ItemId & animationId)
{
    for (Vector<Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> >::Iterator it = mAnimationPlayers.Begin(); it != mAnimationPlayers.End(); ++it) {
        Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> animationPlayer = *it;
        if (animationPlayer!=0) {
            if(animationId==ItemId(animationPlayer->GetName())) {
                return animationPlayer;
            }
        }
    }
    return Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase>();
}

// ------------------------------------------------------------------------
void ViewHandler::Invalidate(RenderTarget * renderTarget, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if(renderTarget!=0) {
        mViewContainer.Invalidate(renderTarget, dirtyArea);
    }
}

// ------------------------------------------------------------------------
void ViewHandler::InvalidateAll(bool forceInvalidateAllRendertargets, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    mViewContainer.Invalidate(dirtyArea);
    if((mRenderer!=0) && forceInvalidateAllRendertargets) {
        mRenderer->InvalidateAll(forceInvalidateAllRendertargets);
    }
}

// ------------------------------------------------------------------------
bool ViewHandler::ActivateLayer(Int layerId, bool activate)
{
    if(mRenderer!=0) {
        return mRenderer->ActivateLayer(layerId,activate);
    }
    return false;
}

// ------------------------------------------------------------------------
void ViewHandler::SetAnimationSpeedFactor(FeatStd::Float speedFactor)
{
    COURIER_LOG_INFO("Animation Speed factor set to %.3f",speedFactor);
    UInt32 currentWorldTime = Platform::Ticks(Platform::TicksType::Now).GetTime();
    UpdateRefWorldTime(currentWorldTime);
    mAnimationSpeedFactor = speedFactor;
}

// ------------------------------------------------------------------------
void ViewHandler::UpdateRefWorldTime(UInt32 worldTime)
{
    if (0 == mAnimationRefWT) {
        mAnimationRefWT = worldTime;
    } else {
        Float animationTimeDiff  = Float(worldTime - mRefWT) * mAnimationSpeedFactor + mAnimationTimeFragment;
        mAnimationRefWT += UInt32(animationTimeDiff);
        mAnimationTimeFragment = animationTimeDiff - static_cast<Float>(UInt32(animationTimeDiff));
    }
    mRefWT = worldTime;
}

// ------------------------------------------------------------------------
void ViewHandler::DefineWorldTime()
{
    mWorldTime = Platform::Ticks(Platform::TicksType::Now).GetTime();
    if ((mWorldTime - mRefWT) > UPDATE_TIME_PERIOD) {
        UpdateRefWorldTime(mWorldTime);
    }

    Float animationTimeDiff  = Float(mWorldTime - mRefWT) * mAnimationSpeedFactor + mAnimationTimeFragment;
    UInt32 animationWorldTime = mAnimationRefWT + UInt32(animationTimeDiff);

    if (mAnimationTimeDispatcher != 0) {
        mAnimationTimeDispatcher->SetWorldTimeMs(animationWorldTime);
        mAnimationTimeDispatcher->DispatchTime();
    }
}

#ifdef CANDERA_TRANSITIONS_ENABLED
void ViewHandler::UpdateCanderaTr()
{
    const UInt32 worldTime = Platform::Ticks(Platform::TicksType::Now).GetTime();
    const UInt32 deltaTime = (0 != mCanderaTransitionTime) ? static_cast<UInt32>(worldTime - mCanderaTransitionTime) : 0;
    mCanderaTrHandler.Update(deltaTime);
    mCanderaTransitionTime = worldTime;
}
#endif
// ------------------------------------------------------------------------
UInt32 ViewHandler::GetWorldTime()
{
    return mWorldTime;
}

// ------------------------------------------------------------------------
View * ViewHandler::FindView(const ViewId & viewId)
{
    if (viewId == ViewId()) {
        return &mViewContainer;
    }
    return const_cast<View*>(mViewAccessor.Lookup(viewId.GetHashId()));
}

// ------------------------------------------------------------------------
static void GetParentPath(View::Name & pathName, const Char * viewName)
{
    const UInt cBufferSize = 256;
    Char buffer[cBufferSize];
    FEATSTD_DEBUG_ASSERT(cBufferSize>View::Name::TCharCapacity());
    FeatStd::Internal::String::CopyPartial(buffer,viewName,cBufferSize-1);
    Int idx = static_cast<Int>(FeatStd::Internal::String::Length(buffer));
    do {
        if((buffer[idx]=='#') || (idx==0)) {
            buffer[idx] = '\0';
            idx = 0;
        }
        idx--;
    } while(idx>=0);
    pathName = buffer;
}

// ------------------------------------------------------------------------
View * ViewHandler::FindParentView(const ViewId & viewId)
{
    View::Name pathName;
    GetParentPath(pathName, viewId.CStr());
    return FindView(ViewId(pathName.CStr()));
}

// ------------------------------------------------------------------------
View * ViewHandler::CreateParentView(const ViewId & viewId)
{
    View::Name pathName;
    GetParentPath(pathName, viewId.CStr());

    if(0==FeatStd::Internal::String::Length(pathName.CStr())) {
        return &mViewContainer;
    }
    if(0==FeatStd::Internal::String::CompareStrings(mViewContainer.GetId().CStr(),pathName.CStr())) {
        return &mViewContainer;
    }
    View* view = (0 == mViewFactory) ? 0 : mViewFactory->Create(ViewId(pathName.CStr()));
    if (0 == view) {
        view = CreateViewContainer();
    }
    else {
        mForceAnimationInvalidator = true;
    }
    view = InitializeViewImpl(view, pathName.CStr(), true);
    if (0 == view) {
        COURIER_LOG_ERROR("ViewContainer '%s' was not created.",pathName.CStr());
    }
    return view;
}

// ------------------------------------------------------------------------
bool ViewHandler::ExecuteViewAction(ViewAction::Enum action, const ViewId & viewId, bool loadScene, bool initScene)
{
    if (mFinalizing) {
        return false;
    }
    FEATSTD_DEBUG_ASSERT(mViewFactory!=0);
    bool rc = false;
    if(mViewFactory!=0) {
        switch(action) {
            case ViewAction::Create: {
                // first of all check if the view is already existing
                View * view = FindView(viewId);
                if ((0 != view) && (&mViewContainer != view)) {
                    COURIER_LOG_DEBUG("ViewAction::Create: '%s' already created.",viewId.CStr());
                    rc = true;
                } else {
                    // create the view object
                    view = mViewFactory->Create(viewId);
                    if (0 != view) {
                        mForceAnimationInvalidator = true;
                        const Char* viewName = view->GetId().CStr();
                        if ((0 == viewName) || (0 == viewName[0])) {
                            viewName = 0;
                            SceneEntryVector sceneVector;
                            if (RetrieveSceneEntries(sceneVector)) {
                                FeatStd::SizeType viewStrLen = FeatStd::Internal::String::Length(viewId.CStr());
                                for (SizeType i = 0; (i < sceneVector.Size()) && (0 == viewName); ++i) {
                                    const SceneEntry& sceneEntry = sceneVector[i];
                                    if (0 == FeatStd::Internal::String::ComparePartial(viewId.CStr(), sceneEntry.mSceneName, viewStrLen)) {
                                        FeatStd::SizeType sceneNameLen = FeatStd::Internal::String::Length(sceneEntry.mSceneName);
                                        if (sceneNameLen == viewStrLen) {
                                            viewName = sceneEntry.mSceneName;
                                        }
                                    }
                                }
                            }
                        }
                        if (0 == viewName) {
                            viewName = viewId.CStr();
                            FEATSTD_LOG_WARN("ViewAction::Create: view id '%s' is not part of the asset and not initialized by the view factory!", viewName);
                        }
                        view = InitializeViewImpl(view, viewName, initScene);
                    }
                    else {
                        SceneEntryVector sceneVector;
                        if (RetrieveSceneEntries(sceneVector)) {
                            FeatStd::SizeType viewStrLen = FeatStd::Internal::String::Length(viewId.CStr());
                            for (SizeType i = 0; i < sceneVector.Size(); ++i) {
                                const SceneEntry& sceneEntry = sceneVector[i];
                                if ((0 == viewStrLen) || (0 == FeatStd::Internal::String::ComparePartial(viewId.CStr(), sceneEntry.mSceneName, viewStrLen))) {
                                    FeatStd::SizeType sceneNameLen = FeatStd::Internal::String::Length(sceneEntry.mSceneName);
                                    if (sceneNameLen == viewStrLen) {
                                        view = CreateViewScene(sceneEntry.mIs2D);
                                        if (0 != view) {
                                            view = InitializeViewImpl(view, sceneEntry.mSceneName, initScene);
                                        }
                                        break;
                                    }
                                    if ((sceneNameLen > viewStrLen) && ('#' == sceneEntry.mSceneName[viewStrLen])) {
                                        view = CreateViewContainer();
                                        if (0 != view) {
                                            view = InitializeViewImpl(view, viewId.CStr(), initScene);
                                        }
                                        break;
                                    }
                                }
                            }
                        } else {
                            COURIER_LOG_ERROR("ViewAction::Create: Cannot load scene names.");
                        }
                    }

                    // if we have a view and possibly a view controller the make an init
                    if (0 == view) {
                        COURIER_LOG_ERROR("ViewAction::Create: cannot create '%s'.",viewId.CStr());
                    }
                    else {
                        rc = true;
                    }
                }
                // if everythings okay and we have a view and shall upload the content then upload
                if(rc && loadScene && (0 != view)) {
                    rc = view->LoadContent(true, true);
                    if(! rc) {
                        COURIER_LOG_ERROR("ViewAction::Create: cannot upload scene '%s'.",viewId.CStr());
                    }
                }
                break;
            }

            case ViewAction::CreateAll: {
                View * view = FindView(viewId);
                if ((view != 0) && (&mViewContainer != view)) {
                    COURIER_LOG_DEBUG("ViewAction::CreateAll: '%s' already created.",viewId.CStr());
                    rc = true;
                } else {
                    if (FeatStd::Internal::String::Length(viewId.CStr()) == 0) {
                        (void)StopAllRunningAnimations();
                        mViewContainer.Finalize();
                        mViewAccessor.Finalize();
                    }
                    SceneEntryVector sceneVector;
                    rc = RetrieveSceneEntries(sceneVector);
                    if(! rc) {
                        COURIER_LOG_ERROR("ViewAction::CreateAll: Cannot load scene names.");
                    }
                    rc = rc && CreateViews(sceneVector,viewId,initScene);
                    if(! rc) {
                        COURIER_LOG_ERROR("ViewAction::CreateAll: Cannot create views.");
                    }
                    if(rc && loadScene) {
                        if (FeatStd::Internal::String::Length(viewId.CStr()) == 0) {
                            rc = mViewContainer.LoadContent(true, true);
                        }
                        else {
                            view = FindView(viewId);
                            if (view != 0) {
                                rc = view->LoadContent(true, true);
                            }
                        }
                        if(! rc) {
                            COURIER_LOG_ERROR("ViewAction::CreateAll: Cannot load scenes.");
                        }
                    }
                }
                // if everythings okay and we have a view and shall upload the content then upload
                if(rc && loadScene && (view!=0)) {
                    rc = view->LoadContent(true, true);
                    if(! rc) {
                        COURIER_LOG_ERROR("ViewAction::Create: cannot upload scene '%s'.",viewId.CStr());
                    }
                }
                break;
            }

            case ViewAction::Destroy: {
                View * view = FindView(viewId);
                if ((0 != view) && (&mViewContainer != view)) {
                    DestroyView(view);
                    rc = true;
                }
                else {
                    COURIER_LOG_ERROR("ViewAction::Destroy: cannot destroy '%s'.",viewId.CStr());
                }
                break;
            }

            case ViewAction::DestroyAll: {
                (void)StopAllRunningAnimations();
                mAnimationInvalidator.Clear();
                mViewContainer.Finalize();
                mViewAccessor.Finalize();
                mCurrentLoadingViewScenes.Clear();
                rc = true;
                break;
            }

            case ViewAction::Clear: {
                View * view = FindView(viewId);
                if(view!=0) {
                    view->Clear(false, FeatStd::Optional<Candera::Rectangle>());
                } else {
                    COURIER_LOG_ERROR("ViewAction::Clear: cannot clear '%s'.",viewId.CStr());
                }
                rc = (view!=0);
                break;
            }
            case ViewAction::Invalidate: {
                View * view = FindView(viewId);
                if(view!=0) {
                    view->Invalidate(FeatStd::Optional<Candera::Rectangle>());
                } else {
                    COURIER_LOG_ERROR("ViewAction::Invalidate: cannot invalidate '%s'.",viewId.CStr());
                }
                rc = (view!=0);
                break;
            }
            case ViewAction::ClearCameraViewports: {
                View * view = FindView(viewId);
                if ( 0 != view) {
                    view->Clear(true, FeatStd::Optional<Candera::Rectangle>());
                } else {
                    COURIER_LOG_ERROR("ViewAction::ClearCameraViewports: cannot clear '%s'.",viewId.CStr());
                }
                rc = ( 0 != view);
                break;
           }
        }
        return rc;
    }
    return false;
}

// ------------------------------------------------------------------------
void ViewHandler::DestroyView(View * view)
{
    FEATSTD_DEBUG_ASSERT(view!=0);
    FEATSTD_DEBUG_ASSERT(mViewFactory!=0);
    mAnimationInvalidator.RemoveView(view);
    if ((0 != view) && (&mViewContainer != view)) {
        for (SizeType i = 0; i < mCurrentLoadingViewScenes.Size();) {
            if (view == mCurrentLoadingViewScenes[i].mScene) {
                static_cast<void>(mCurrentLoadingViewScenes.Remove(i));
            }
            else {
                ++i;
            }
        }
        if (0 != mViewFactory) {
            view->Finalize(); // first finalize, then remove view
            // remove it from accessors and from parent view
            if(RemoveView(view)) {
                // destroy the optional view controller
                if(mViewControllerFactory!=0) {
                    ViewController * vc = view->GetViewController();
                    if(vc!=0) {
                        view->ReleaseViewController();
                        mViewControllerFactory->Destroy(vc);
                    }
                }
                // destroy the view and check if is managed by the viewhandler or not
                if(! view->IsViewHandlerManaged()) {
                    return mViewFactory->Destroy(view);
                } else {
                    return mViewFactory->DestroyView(view);
                }
            }
        }
    }
}

// ------------------------------------------------------------------------
bool ViewHandler::AddView(View * view)
{
    if ((0 != view) && (&mViewContainer != view)) {
        View * parentView = FindParentView(view->GetId());
        if(parentView==0) {
            parentView = CreateParentView(view->GetId());
            if(parentView==0) {
                return false;
            }
        }
        if(mViewAccessor.AddItem(view)) {
            bool rc = parentView->AddView(view);
            if(rc) {
                view->OnScopeMaskChanged();
            }
            return rc;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool ViewHandler::RemoveView(View * view)
{
    if ((0 != view) && (&mViewContainer != view)) {
        if(mViewAccessor.RemoveItem(view)) {
            const ViewId & path = view->GetPathId();
            View * parentView = FindView(path);
            if(parentView==0) {
                parentView = &mViewContainer;
            }
            return parentView->RemoveView(view);
        }
        FEATSTD_DEBUG_FAIL();
    }
    return false;
}

// ------------------------------------------------------------------------
FrameworkWidget * ViewHandler::FindWidget(const ViewId & viewId, const CompositePath & compositePath, const ItemId & widgetId)
{
    View * view = FindView(viewId);
    if(view!=0) {
        return view->GetFrameworkWidget(compositePath, widgetId);
    }
    return 0;
}

// ------------------------------------------------------------------------
//! [COURIER_TouchEventPreprocessor_SearchTouchWidget]
bool ViewHandler::OnMessage(const Message & msg)
{
    bool rc = false;

    const DelayedUnloadMsg* delayedUnloadMsg = Courier::message_cast<const DelayedUnloadMsg*>(&msg);
    if (0 != delayedUnloadMsg) {
        if (!m_delayedUnloadRequest.Add(DelayedUnloadRequest(delayedUnloadMsg->GetViewId(), cCOURIER_DEFAULT_RENDER_COUNT))) {
            COURIER_LOG_FATAL("Adding DelayedUnloadRequest to container failed.\n");
        }
        return true;
    }

    if(0 != mViewHandlerSession) {
        rc = mViewHandlerSession->ProcessMessage(msg);
        if(rc) {
            return true;
        }
    }

    if(msg.IsFocus()) {
        const Courier::TouchMsg * msgPtr = Courier::message_cast<const Courier::TouchMsg *>(&msg);
        if(msgPtr!=0) {
            const TouchInfo info(msgPtr->GetXPos(), msgPtr->GetYPos(), msgPtr->GetTimeStamp(), msgPtr->GetPointerId(), msgPtr->GetSourceId());
            FrameworkWidget * touchedWidget = mViewContainer.GetTouchedWidget(info);
            if(touchedWidget!=0) {
                const Char * widgetName =  static_cast<WidgetBase*>(touchedWidget)->GetName();
                COURIER_LOG_DEBUG("TouchMsg for Widget(%s) received", widgetName);
                View * view = touchedWidget->GetParentView();
                if(view!=0) {
                    if(SetFocus(touchedWidget)) {
                        rc = mFocusedFrameworkWidget->OnMessage(msg);
                    }
                }
            }
        } else {
            if(mFocusedFrameworkWidget!=0) {
                COURIER_LOG_DEBUG("FocusMessage(%s) for Widget(%s) received", msg.GetName(),static_cast<WidgetBase*>(mFocusedFrameworkWidget)->GetName());
                rc = mFocusedFrameworkWidget->OnMessage(msg);
            }
        }
    } else {
        rc = OnMessageViewTreeOnly(msg);
    }
    if(! rc) {
        COURIER_LOG_DEBUG("No Entity has consumed Message %s", msg.GetName());
    }
    return rc;
}
//! [COURIER_TouchEventPreprocessor_SearchTouchWidget]

// ------------------------------------------------------------------------
bool ViewHandler::OnMessageViewTreeOnly(const Message & msg)
{
    bool rc = mViewContainer.OnMessage(msg);
    return rc;
}

// ------------------------------------------------------------------------
bool ViewHandler::IsAnimationEnabled()
{
    for (Vector<Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> >::Iterator it = mAnimationPlayers.Begin(); it != mAnimationPlayers.End(); ++it) {
        Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> animationPlayer = *it;
        if (animationPlayer!=0) {
            if(animationPlayer->IsEnabled()) {
                return true;
            }
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool ViewHandler::ChangeTheme(const ItemId & themeId)
{
    if(mAssetProvider!=0) {
        Candera::Id id = Candera::Internal::AssetProviderFunctions::GetIdByName(mAssetProvider, ThemeLib, (themeId != ItemId()) ? themeId.CStr() : 0);
        mAssetProvider->SetCurrentThemeById(id);
        return true;
    }
    return false;
}

// ------------------------------------------------------------------------
void ViewHandler::Update(RenderHint * renderHint)
{
    // Update
    DefineWorldTime();
    StateMachineBehavior* stateMachine = StateMachineBehavior::GetGlobalStateMachine();
    if (0 != stateMachine) {
        stateMachine->Update();
    }
    mViewContainer.Update(renderHint);

#ifdef CANDERA_TRANSITIONS_ENABLED
    // Update Candera Transition Strategy
    UpdateCanderaTr();
#endif
}

// ------------------------------------------------------------------------
bool ViewHandler::PartialLoad()
{
    bool result = false;
    for (SizeType i = 0; i < mCurrentLoadingViewScenes.Size(); ++i) {
        if (0 != mCurrentLoadingViewScenes[i].mScene) {
            if (mCurrentLoadingViewScenes[i].mScene->PartialLoad(mCurrentLoadingViewScenes[i].mForceUpload)) {
                result = true;
    }
            else {
                mCurrentLoadingViewScenes[i].mScene = 0; // mark for deletion
            }
        }
    }
    // TODO: more efficient deletion of finished entries
    for (SizeType i = 0; i < mCurrentLoadingViewScenes.Size();) {
        if (0 == mCurrentLoadingViewScenes[i].mScene) {
            static_cast<void>(mCurrentLoadingViewScenes.Remove(i));
        }
        else {
            ++i;
        }
    }
    return result;
}

void ViewHandler::SchedulePartialLoad(Courier::ViewScene* view, bool forceUpload)
{
    static_cast<void>(mCurrentLoadingViewScenes.Add(PartialLoadData(view, forceUpload)));
}

// ------------------------------------------------------------------------
void ViewHandler::Accept(ViewVisitor* visitor)
{
    if((visitor!=0) && visitor->OnBegin()) {
        mViewContainer.Accept(*visitor);
        visitor->OnEnd();
    }
}

// ------------------------------------------------------------------------
void ViewHandler::DetectAnimationScenes(RenderHint * renderHint, bool force)
{
    for (Vector<Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> >::Iterator it = mAnimationPlayers.Begin(); it != mAnimationPlayers.End(); ++it) {
        Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> animationPlayer = *it;
        if (animationPlayer!=0) {
            mAnimationInvalidator.ProcessScenes(renderHint, animationPlayer.GetPointerToSharedInstance(), force);
        }
    }
}

// ------------------------------------------------------------------------
bool ViewHandler::Render(RenderHint * renderHint)
{
    return Render(renderHint,false);
}

// ------------------------------------------------------------------------
bool ViewHandler::Render(RenderHint * renderHint, bool renderAll)
{
    FEATSTD_DEBUG_ASSERT(mRenderer!=0);

    FeatStd::SizeType index = 0;
    while (index < m_delayedUnloadRequest.Size()) {
        if (m_delayedUnloadRequest[index].m_delayCounter > 0) {
            --m_delayedUnloadRequest[index].m_delayCounter;
        }
        bool increment = true;
        if (m_delayedUnloadRequest[index].m_delayCounter == 0) {
            Courier::LoadReqMsg* msgPtr = COURIER_MESSAGE_NEW(Courier::LoadReqMsg)(m_delayedUnloadRequest[index].m_viewId, false);
            if ((0 != msgPtr) && (msgPtr->Post())) {
                static_cast<void>(m_delayedUnloadRequest.Remove(index));
                increment = false;
            }
        }
        if (increment) {
            ++index;
        }
    }

    OnPreRender(renderHint);

#if defined(COURIER_RENDERING_MONITOR_ENABLED)
    if(mRenderingMonitor!=0) {
        mRenderingMonitor->OnRenderBegin();
    }
    CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::Frame));
#endif

    Accept(mBeginRenderVisitor);

    // Detect possible scenes for invalidation
    DetectAnimationScenes(renderHint, mForceAnimationInvalidator);
    mForceAnimationInvalidator = false;
    // AnimationInvalidator returns false, if there was a property setter which could not have been processed.
    bool scenesInvalidated = mAnimationInvalidator.Invalidate(renderHint);

    // used for invalidation for Candera transitions
    bool isCanderaTransitionIvalidateAll = false;
#ifdef CANDERA_TRANSITIONS_ENABLED
    FEATSTD_LINT_NEXT_EXPRESSION(838, "MISRA C++ 2008 Required Rule 0-1-9: previously assigned value is used in case CANDERA_TRANSITIONS_ENABLED not defined.")
    bool isCanderaTransitionOngoing = mCanderaTrHandler.IsCanderaTransitionOngoing();
    if (isCanderaTransitionOngoing) {
        isCanderaTransitionIvalidateAll = mCanderaTrHandler.IsInvalidateAll();
        if (!isCanderaTransitionIvalidateAll) {
            mCanderaTrHandler.InvalidateTransitionViews();
        }
    }
#endif

    // In this case render all as before.
    if(renderAll || ((! scenesInvalidated) && IsAnimationEnabled()) || (isCanderaTransitionIvalidateAll)) {
        // we shall render all enabled cameras and therefore its rendertargets.
        // no dirty area information is available for animations.
        // therefore, the complete area has to be marked as dirty.
        mViewContainer.Invalidate(FeatStd::Optional<Candera::Rectangle>());
    }

    // Render
    bool rc = mRenderer->Render(renderHint);

    // Compute the fps after rendering.
    mRenderer->ComputeFramesPerSecond(renderHint);

    Accept(mFinalizeRenderVisitor);

    // reset invalidation flags of rendertargets
    mRenderer->InvalidateAll(false);

#if defined(COURIER_RENDERING_MONITOR_ENABLED)
    if(mRenderingMonitor!=0) {
        mRenderingMonitor->OnRenderEnd();
    }
#endif

    OnPostRender(renderHint);

    return rc;
}

// ------------------------------------------------------------------------
void ViewHandler::OnPreRender(RenderHint * renderHint)
{
    FEATSTD_UNUSED(renderHint);
}

// ------------------------------------------------------------------------
void ViewHandler::OnPostRender(RenderHint * renderHint)
{
    FEATSTD_UNUSED(renderHint);
}

// ------------------------------------------------------------------------
UInt16 ViewHandler::GetFramesPerSecond()
{
#if defined(CANDERA_3D_ENABLED)
    FEATSTD_DEBUG_ASSERT(mRenderer!=0);
    if(mRenderer!=0) {
        return mRenderer->GetMeasure3D().mFramesPerSecond;
    }
#endif
    return 0;
}

// ------------------------------------------------------------------------
FeatStd::SizeType ViewHandler::GetLastFrameDuration()
{
#if defined(CANDERA_3D_ENABLED)
    FEATSTD_DEBUG_ASSERT(mRenderer!=0);
    if(mRenderer!=0) {
        return mRenderer->GetMeasure3D().mLastFrameDuration;
    }
#endif
    return 0;
}

// ------------------------------------------------------------------------
UInt16 ViewHandler::GetFramesPerSecond2D()
{
#if defined(CANDERA_2D_ENABLED)
    FEATSTD_DEBUG_ASSERT(mRenderer!=0);
    if(mRenderer!=0) {
        return mRenderer->GetMeasure2D().mFramesPerSecond;
    }
#endif
    return 0;
}

// ------------------------------------------------------------------------
FeatStd::SizeType ViewHandler::GetLastFrameDuration2D()
{
#if defined(CANDERA_2D_ENABLED)
    FEATSTD_DEBUG_ASSERT(mRenderer!=0);
    if(mRenderer!=0) {
        return mRenderer->GetMeasure2D().mLastFrameDuration;
    }
#endif
    return 0;
}

// ------------------------------------------------------------------------
bool ViewHandler::SetWidgetProperty(const ViewId & viewId, const CompositePath & compositePath, const ItemId & widgetId, const ItemId & propertyId, const Char * value)
{
    FEATSTD_DEBUG_ASSERT(value!=0);

    WidgetBase * widget = static_cast<WidgetBase*>(FindWidget(viewId, compositePath, widgetId));
    if((widget!=0) && (value!=0)) {
        const Char * widgetType = widget->GetTypeName();
        if(widgetType!=0) {
            Candera::MetaInfo::WidgetSetMetaInfo * widgetSet = GetWidgetSet();
            if (widgetSet!=0) {
                Candera::MetaInfo::WidgetMetaInfo * metaInfo = widgetSet->LookupItem(widgetType);
                if (metaInfo != 0) {
                    Candera::MetaInfo::PropertyMetaInfo<WidgetBase> * propertyMetaInfo = metaInfo->LookupItem(propertyId.CStr());
                    if (propertyMetaInfo!=0) {
                        return propertyMetaInfo->Set(widget, value);
                    }
                }
            }
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool ViewHandler::RetrieveSceneEntries(SceneEntryVector & sceneEntryVector) const
{
    if(0!=mAssetProvider) {
        bool rc = true;
        const AssetDescriptor& assetDescriptor = mAssetProvider->GetAssetDescriptor();
        UInt32 count2D = 0;
        UInt32 count3D = 0;
        for (AssetDescriptor::AssetIdIterator scene2dIt = assetDescriptor.GetAssetIdIterator(Scene2DLib); scene2dIt.IsValid(); ++scene2dIt) {
            const Char* scene2dName = mAssetProvider->GetNameById(Scene2DLib, *scene2dIt, 0);
            if(scene2dName == 0)
            {
                FEATSTD_LOG_DEBUG("GetNameById is not valid");
            }
            rc = sceneEntryVector.Add(SceneEntry(scene2dName, true)) && rc;
            ++count2D;
        }
        for (AssetDescriptor::AssetIdIterator scene3dIt = assetDescriptor.GetAssetIdIterator(SceneLib); scene3dIt.IsValid(); ++scene3dIt) {
            const Char* scene3dName = mAssetProvider->GetNameById(SceneLib, *scene3dIt, 0);
            if(scene3dName == 0)
            {
                FEATSTD_LOG_DEBUG("GetNameById is not valid");
            }
            rc = sceneEntryVector.Add(SceneEntry(scene3dName, false)) && rc;
            ++count3D;
        }
        COURIER_LOG_INFO("Retrieved Scenes Count 2D(%d), 3D(%d)", count2D, count3D);
        return true;
    }
    return false;
}

// ------------------------------------------------------------------------
bool ViewHandler::CreateViews(const SceneEntryVector & sceneEntryVector, const ViewId & viewId, bool initScene)
{
    FeatStd::SizeType viewStrLen = FeatStd::Internal::String::Length(viewId.CStr());
    for(SizeType i=0; i<sceneEntryVector.Size(); i++) {
        if((viewStrLen==0) || (0==FeatStd::Internal::String::ComparePartial(viewId.CStr(),sceneEntryVector[i].mSceneName,viewStrLen))) {
            if ((0 == viewStrLen) || (viewStrLen == FeatStd::Internal::String::Length(sceneEntryVector[i].mSceneName)) || ('#' == sceneEntryVector[i].mSceneName[viewStrLen])) {
                if(! CreateView(sceneEntryVector[i], initScene)) {
                    return false;
                }
            }
        }
    }
    return true;
}

// ------------------------------------------------------------------------
bool ViewHandler::CreateView(const SceneEntry & sceneEntry, bool initScene)
{
    View* view = (0 == mViewFactory) ? 0 : mViewFactory->Create(ViewId(sceneEntry.mSceneName));
    if (0 == view) {
        view = CreateViewScene(sceneEntry.mIs2D);
    }
    if (0 != view) {
        mForceAnimationInvalidator = true;
        view = InitializeViewImpl(view, sceneEntry.mSceneName, initScene);
    }
    if (0 == view) {
        COURIER_LOG_ERROR("View '%s' cannot be created.",sceneEntry.mSceneName);
        return false;
    }
    return true;
}

// ------------------------------------------------------------------------
ViewScene * ViewHandler::CreateViewScene(bool is2D)
{
    FEATSTD_DEBUG_ASSERT(mViewFactory!=0);
    ViewScene * scene = 0;
    if(mViewFactory!=0) {
        scene = mViewFactory->CreateViewScene(is2D);
        if (0 != scene) {
            mForceAnimationInvalidator = true;
        }
    }
    return scene;
}

// ------------------------------------------------------------------------
ViewContainer * ViewHandler::CreateViewContainer()
{
    FEATSTD_DEBUG_ASSERT(mViewFactory!=0);
    ViewContainer * container = 0;
    if(mViewFactory!=0) {
        container = mViewFactory->CreateViewContainer();
    }
    return container;
}

// ------------------------------------------------------------------------
bool ViewHandler::LoseFocus(FrameworkWidget * widget)
{
    if((widget!=0) && (mFocusedFrameworkWidget!=0) && (widget==mFocusedFrameworkWidget)) {
        if(mFocusedFrameworkWidget->GetParentView()!=0) {
            const ViewId & focusedViewId = mFocusedFrameworkWidget->GetParentView()->GetId();
            WidgetBase * focusedWidget = static_cast<WidgetBase *>(mFocusedFrameworkWidget);
            const LostFocusEvent lostFocusEvent;
            mFocusedFrameworkWidget->DispatchEvent(lostFocusEvent);
            mFocusedFrameworkWidget = 0;
            const StringIdentifier* compositeIdentifier = focusedWidget->GetStringId();
            if (0 != compositeIdentifier) {
                // the owner is the identifer of the composite
                compositeIdentifier = compositeIdentifier->GetOwner();
                // null pointer is ok since it will produce the same result as CompositePath()
            }
            Message * postMsg = (COURIER_MESSAGE_NEW(LostFocusIndMsg)(focusedViewId, CompositePath(compositeIdentifier), ItemId(focusedWidget->GetName())));
            return (postMsg!=0) && postMsg->Post();
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool ViewHandler::SetFocus(FrameworkWidget* frameworkWidget)
{
    bool rc = true;
    WidgetBase * widget = static_cast<WidgetBase*>(frameworkWidget);
    if (widget != mFocusedFrameworkWidget) {
        if ((0 != mFocusedFrameworkWidget) && (0 != mFocusedFrameworkWidget->GetParentView())) {
            const ViewId & focusedViewId = mFocusedFrameworkWidget->GetParentView()->GetId();
            WidgetBase * focusedWidget = static_cast<WidgetBase *>(mFocusedFrameworkWidget);
            const LostFocusEvent lostFocusEvent;
            mFocusedFrameworkWidget->DispatchEvent(lostFocusEvent);
            mFocusedFrameworkWidget = 0;
            const StringIdentifier* compositeIdentifier = focusedWidget->GetStringId();
            if (0 != compositeIdentifier) {
                // the owner is the identifer of the composite
                compositeIdentifier = compositeIdentifier->GetOwner();
                // null pointer is ok since it will produce the same result as CompositePath()
            }
            Message * postMsg = (COURIER_MESSAGE_NEW(LostFocusIndMsg)(focusedViewId, CompositePath(compositeIdentifier), ItemId(focusedWidget->GetName())));
            rc = (postMsg!=0) && postMsg->Post();
        }
    }
    rc = (widget!=0) && rc;
    if(rc) {
        mFocusedFrameworkWidget = widget;
        if(widget!=0) {
            const FocusEvent focusEvent;
            widget->DispatchEvent(focusEvent);
        }
    }
    return rc;
}

bool ViewHandler::SetFocus(const ViewId & viewId, const CompositePath & compositePath, const ItemId & widgetId)
{
    return SetFocus(FindWidget(viewId, compositePath, widgetId));
}

// ------------------------------------------------------------------------
void ViewHandler::AnimationFinishedCallback(Int32 completedIterationsCount, Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> player, void * userData)
{
    FEATSTD_UNUSED2(userData, completedIterationsCount);
    const Char * playerName = (player->GetName()!=0) ? player->GetName() : "";
    const StringIdentifier* compositeIdentifier = player->GetStringId();
    if (0 != compositeIdentifier) {
        // the owner is the identifer of the composite
        compositeIdentifier = compositeIdentifier->GetOwner();
        // null pointer is ok since it will produce the same result as CompositePath()
    }
    // TODO: get view id and composite path from player
    Message * postMsg = (COURIER_MESSAGE_NEW(AnimationIndMsg)(ViewId(), CompositePath(compositeIdentifier), ItemId(playerName)));
    if(postMsg!=0) {
        (void)postMsg->Post();
    }
}

// ------------------------------------------------------------------------
bool ViewHandler::SetCurrentCulture(const Char * locale)
{
    bool rc = Candera::Globalization::CultureManager::GetInstance().SetCurrentCulture(locale);
    if(rc) {
        COURIER_LOG_INFO("Culture changed to '%s'.",locale);
    } else {
        COURIER_LOG_ERROR("Culture '%s' could not be set.",locale);
    }
    return rc;
}

// ------------------------------------------------------------------------
void ViewHandler::CultureChangeCallback(const Candera::Globalization::Culture & culture)
{
    Candera::Globalization::Culture::SharedPointer sPtr(const_cast<Candera::Globalization::Culture *>(&culture));
    Message * postMsg = (COURIER_MESSAGE_NEW(CultureChangeIndMsg)(sPtr));
    if(postMsg!=0) {
        (void)postMsg->Post();
    }
}
// ------------------------------------------------------------------------
bool ViewHandler::SetCurrentLoadingViewSceneInternal(ViewScene * viewScene, bool forceUpload)
{
    mCurrentLoadingViewSceneForceUpload = forceUpload;
    if(0 == mCurrentLoadingViewScene) {
        mCurrentLoadingViewScene = viewScene;
        return true;
    }

    if(mCurrentLoadingViewScene==viewScene) {
        return true;
    }

    if(0 == viewScene) {
        mCurrentLoadingViewScene = 0;
        return true;
    }

    return false;
}

// ------------------------------------------------------------------------
ViewScene * ViewHandler::GetCurrentLoadingViewSceneInternal()
{
    return mCurrentLoadingViewScene;
}

View * ViewHandler::InitializeViewImpl(View* view, const Char* viewId, bool initScene)
{
    if (0 != view) {
        ViewController * viewController = 0;
        if (0 != mViewControllerFactory) {
            viewController = mViewControllerFactory->Create(ViewId(viewId));
        }
        if (view->Init(this, viewId, viewController)) {
            if (AddView(view)) {
                if ((!initScene) || view->InitContent(true)) {
                    mCameraGroupHandler.OnViewInitContent(true, *view);
                    return view;
                }
                static_cast<void>(RemoveView(view));
            }
        }
        view->ReleaseViewController();
        if ((0 != mViewControllerFactory) && (0 != viewController)) {
            mViewControllerFactory->Destroy(viewController);
        }
        if (view->IsViewHandlerManaged()) {
            mViewFactory->DestroyView(view);
        }
        else {
            mViewFactory->Destroy(view);
        }
    }
    return 0;
}

}
