//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_ViewFacade_h)
#define Courier_Visualization_ViewFacade_h

#include <Courier/Base.h>
#include <Courier/Visualization/View.h>
#include <Courier/Visualization/ViewPlaceholder.h>

namespace Courier { 
    class Message;
    class IViewHandler;
    class RenderHint;
}

// ----------------------------------------------------------------------------
namespace Courier { 
/// @addtogroup COURIER_VISUALIZATION
/// @{
/** The ViewFacade class is the facade used by the ViewComponent.
    There are two entry points which has to be called in a cyclic manner: 
        - The OnMessage method which distributes messages to Views and their Widgets.   
        - The Render method which is responsible for rendering using Candera library.
    The ViewFacade is intercepting general messages for view creation, animation maintenance and 
    other purposes and is using the ViewHandler instance for executing those operations.
*/
class ViewFacade  {
    public:
        ///
        ViewFacade();

        ///
        virtual ~ViewFacade();

        /** Initializes the ViewFacade.
            @param viewHandler IViewHandler used for rendering and message distribution.
            @return <em>true</em> if initialization was successful,
                    <em>false</em> otherwise. */
        bool Init(IViewHandler * viewHandler);

        ///
        bool PartialLoad();

        ///
        void Update(RenderHint * renderHint);

        /** Renders all Views and their scenes.
            @return <em>true</em> if rendering was successful,
                    <em>false</em> otherwise. */
        bool Render(RenderHint * renderHint);

        /** Distributes messages and processes defined messages, like render request.
            @return <em>true</em> if message was processed,
                    <em>false</em> otherwise. */
        bool OnRenderComponentMessage(ComponentId id, RenderHint * renderHint, const Message & msg);

        /** Distributes messages and processes defined messages, like view creation etc.
            @return <em>true</em> if message was processed,
                    <em>false</em> otherwise. */
        bool OnViewComponentMessage(const Message & msg);

        /// Enables or disables the automatic loading of the view if the view gets activated (default = true).
        static void SetLoadSceneOnViewActivation(bool enable) { mLoadSceneOnViewActivation = enable; }

        /** Enables or disables the automatic loading of the view if the rendering of the view gets enabled (default = true).
            If this flag is enabled the LoadSceneOnViewActivation flag has no meaning anymore. Turning of LoadSceneOnViewEnableRendering
            LoadSceneOnViewActivation flag is processed again for backwards compatibility <= Courier 2.1.0.x
        */
        static void SetLoadSceneOnViewEnableRendering(bool enable) { mLoadSceneOnViewEnableRendering = enable; }

        /// 
        static bool GetLoadSceneOnViewActivation() { return mLoadSceneOnViewActivation; }
        /// 
        static bool GetLoadSceneOnViewEnableRendering() { return mLoadSceneOnViewEnableRendering; }

        /// @return the ViewHandler of the view facade.
        IViewHandler * GetViewHandler() const
        {
            return mViewHandler;
        }

    private:
        /// The used ViewHandler
        IViewHandler * mViewHandler;
        /// The flag which controls the automatic loading if the view is activated
        static bool mLoadSceneOnViewActivation;
        /// The flag which controls the automatic loading if the rendering is enabled, if set to true mLoadSceneOnViewActivation has no meaning.
        static bool mLoadSceneOnViewEnableRendering;

        Courier::Vector<Internal::ViewPlaceholder> mViewPlaceholderVct;

        FeatStd::SizeType FindViewPlaceholder(const ViewId& viewId);
};
    /// @}
}

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {
    template<> UInt32 StringBufferAppender<Candera::Globalization::TextDirection>::Append(StringBuffer& stringBuffer, Candera::Globalization::TextDirection const & object);
    template<> UInt32 StringBufferAppender<Candera::Globalization::Culture>::Append(StringBuffer& stringBuffer, Candera::Globalization::Culture const & object);
}
#endif

#endif
