//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ViewContainer.h"

#include "IViewHandler.h"
#include "ViewVisitor.h"
#include "ViewController.h"
#include "VisualizationMsgs.h"

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

using namespace Candera;

// ------------------------------------------------------------------------
ViewContainer::ViewContainer(bool managed) : Base(managed)
{
}

// ------------------------------------------------------------------------
ViewContainer::~ViewContainer()
{
    SetParentView(0);
}

// ------------------------------------------------------------------------
bool ViewContainer::Init(IViewHandler * viewHandler, const Char * viewName)
{
    return ViewContainer::Init(viewHandler,viewName,0);
}

// ------------------------------------------------------------------------
bool ViewContainer::Init(IViewHandler * viewHandler, const Char * viewName, ViewController * viewController)
{
    FEATSTD_DEBUG_ASSERT(viewHandler!=0);
    FEATSTD_DEBUG_ASSERT(viewName!=0);
    bool rc = (viewHandler!=0) && (viewName!=0);
    if(rc) {
        // copy the container name into Name object, this is why we don�t have a valid container name outside
        mName = viewName;
        if(FeatStd::Internal::String::Length(viewName)>FeatStd::Internal::String::Length(mName.CStr())) {
            COURIER_LOG_ERROR("Name '%s' of ViewContainer exceeds View::Name capacity(%u).",viewName, View::Name::TCharCapacity());
            return false;
        }
        // mContainerName now refers to the mName
        mContainerName = ViewId(mName.CStr());
        rc = Base::Init(viewHandler, viewName, viewController);
    }
    return rc;
}

// ------------------------------------------------------------------------
const ViewId & ViewContainer::GetId() const
{
    return mContainerName;
}

// ------------------------------------------------------------------------
bool ViewContainer::IsInitialized()
{
    FEATSTD_DEBUG_ASSERT(GetViewHandler()!=0);
    FEATSTD_DEBUG_ASSERT(GetViewHandler()->GetContentLoader()!=0);
    return (GetViewHandler()!=0) && (GetViewHandler()->GetContentLoader()!=0);
}

// ------------------------------------------------------------------------
bool ViewContainer::AddView(View * view)
{
    FEATSTD_DEBUG_ASSERT(view!=0);
    ResetInvalidationFlag();
    View* parent = GetParentView();
    while (0 != parent) {
        parent->ResetInvalidationFlag();
        parent = parent->GetParentView();
    }

    bool rc = (view!=0);
    if(rc) {
        view->SetParentView(this);
        bool inserted = false;
        for (ViewLinkedList::Iterator it = mViews.Begin(); (it!=mViews.End()) && (! inserted); ++it) {
            if(view->GetId()<it->GetId()) {
                mViews.Insert(view,&*it);
                inserted = true;
            }
        }
        if(! inserted) {
            mViews.Append(view);
        }
        OnScopeMaskChanged();
        return true;
    }
    return false;
}

// ------------------------------------------------------------------------
bool ViewContainer::RemoveView(View * view)
{
    FEATSTD_DEBUG_ASSERT(view!=0);
    bool rc = (view!=0);
    if (rc) {
        view->SetParentView(0);
        rc = mViews.Remove(view);
        OnScopeMaskChanged();
    }
    return rc;
}

// ------------------------------------------------------------------------
bool ViewContainer::GetRenderTargetPtrVector(RTPtrVector & vector)
{
    return ViewContainer::GetRenderTargetPtrVector(vector,true);
}

// ------------------------------------------------------------------------
bool ViewContainer::GetRenderTargetPtrVector(RTPtrVector & vector, bool clear)
{
    if(clear) {
        vector.Clear();
    }
    bool rc = true;
    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        rc = it->GetRenderTargetPtrVector(vector,false) && rc;
    }
    return rc;
}

// ------------------------------------------------------------------------
void ViewContainer::Finalize()
{
    ViewPtrVector deletionViews;
    ViewLinkedList::Iterator it = mViews.Begin();
    while (it != mViews.End()) {
        (void)deletionViews.Add(&*it);
        ++it;
    }

    // necessary to do the deletion via temp vector because of list implementation:
    // removing an item from a list may invalidate existing iterator objects.
    for(FeatStd::SizeType i = 0; i < deletionViews.Size(); ++i) {
        GetViewHandler()->DestroyView(deletionViews[i]); // calls finalize and remove
    }

    mViews.Clear();
}

// ------------------------------------------------------------------------
bool ViewContainer::InitContent(bool init)
{
    bool rc = View::InitContent(init);
    if (0 != mViewController) {
        const ParentViewInitContentEvent initContentEvent(init);
        mViewController->DispatchEvent(initContentEvent);
    }
    return rc;
}

// ------------------------------------------------------------------------
bool ViewContainer::LoadContent(bool load, bool forceUpload)
{
    bool rc = View::LoadContent(load, forceUpload);
    const ParentViewLoadEvent loadEvent(load);

    if((0 != mViewController) && load) {
        mViewController->DispatchEvent(loadEvent);
    }

    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        rc = it->LoadContent(load, forceUpload) && rc;
    }

    if((0 != mViewController) && (! load)) {
        mViewController->DispatchEvent(loadEvent);
    }

    return rc;
}

// ------------------------------------------------------------------------
bool ViewContainer::IsContentInitialized() const
{
    for (ViewLinkedList::ConstIterator it = mViews.Begin(); it != mViews.End(); ++it) {
        if (it->IsContentInitialized()) {
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool ViewContainer::IsContentLoaded() const
{
    for (ViewLinkedList::ConstIterator it = mViews.Begin(); it != mViews.End(); ++it) {
        if (it->IsContentLoaded()) {
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
void ViewContainer::EnableRendering(bool enable)
{
    const ParentViewRenderingEnabledEvent renderingEvent(enable);

    if((0 != mViewController) && enable) {
        mViewController->DispatchEvent(renderingEvent);
    }

    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        it->EnableRendering(enable);
    }

    if((0 != mViewController) && (! enable)) {
        mViewController->DispatchEvent(renderingEvent);
    }
}

// ------------------------------------------------------------------------
bool ViewContainer::IsRenderingEnabled() const
{
    for (ViewLinkedList::ConstIterator it = mViews.Begin(); it != mViews.End(); ++it) {
        if (it->IsRenderingEnabled()) {
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
void ViewContainer::Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if (!IsAlreadyInvalidated(ItemId(), dirtyArea)) {
        View::InvalidateImpl(ItemId(), dirtyArea);
        for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
            it->Invalidate(dirtyArea);
        }
    }
}

// ------------------------------------------------------------------------
void ViewContainer::Invalidate(RenderTarget * renderTarget, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        it->Invalidate(renderTarget, dirtyArea);
    }
}

// ------------------------------------------------------------------------
void ViewContainer::Invalidate(UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if (!IsAlreadyInvalidated(ItemId(), dirtyArea)) {
        View::InvalidateImpl(ItemId(), dirtyArea);
        for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
            it->Invalidate(renderCounter, dirtyArea);
        }
    }
}

// ------------------------------------------------------------------------
void ViewContainer::Invalidate(void * excludeCamera, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{ 
    COURIER_UNUSED(excludeCamera); 
    COURIER_UNUSED(dirtyArea);
}

// ------------------------------------------------------------------------
void ViewContainer::Clear(bool onlyCameraViewPorts, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        it->Clear(onlyCameraViewPorts, dirtyArea);
    }
}

// ------------------------------------------------------------------------
bool ViewContainer::IsActive() const
{
    for (ViewLinkedList::ConstIterator it = mViews.Begin(); it != mViews.End(); ++it) {
        if (it->IsActive()) {
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
void ViewContainer::Update(RenderHint * renderHint)
{
    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        it->Update(renderHint);
    }
}

// ------------------------------------------------------------------------
FrameworkWidget * ViewContainer::GetTouchedWidget(const TouchInfo & info)
{
    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        if (it->IsActive())  {
            FrameworkWidget * fw = it->GetTouchedWidget(info);
            if(fw!=0) {
                return fw;
            }
        }
    }
    return 0;
}

// ------------------------------------------------------------------------
void ViewContainer::Activate(bool activate)
{
    const ParentViewActivateEvent activateEvent(activate);

    if ((0 != mViewController) && activate) {
        mViewController->DispatchEvent(activateEvent);
    }

    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        it->Activate(activate);
    }

    if ((0 != mViewController) && activate) {
        mViewController->DispatchEvent(activateEvent);
    }
}

// ------------------------------------------------------------------------
FrameworkWidget * ViewContainer::GetFrameworkWidget(const CompositePath & compositePath, const ItemId & widgetId)
{
    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        FrameworkWidget * fw = it->GetFrameworkWidget(compositePath, widgetId);
        if(fw!=0) {
            return fw;
        }
    }
    return 0;
}

// ------------------------------------------------------------------------
Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> ViewContainer::GetAnimation(const CompositePath & compositePath, const ItemId & animationId)
{
    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> animationPlayer = it->GetAnimation(compositePath, animationId);
        if (!animationPlayer.PointsToNull()) {
            return animationPlayer;
        }
    }
    return Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase>();

}

// ------------------------------------------------------------------------
bool ViewContainer::DistributeMessage(const Message & msg)
{
    bool lMsgConsumed = false;
    for (ViewLinkedList::Iterator it = mViews.Begin(); (it!=mViews.End()) && (! lMsgConsumed); ++it) {
        if (it->IsActive()) {
            if(msg.InScopeOf(it->GetEffectiveScopeMask())) {
                lMsgConsumed = it->OnMessage(msg);
            }
        }
    }
    return lMsgConsumed;
}

// ------------------------------------------------------------------------
View * ViewContainer::FindView(const ViewId & viewId)
{
    if (viewId==GetId()) {
        return this;
    }

    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        View * view = it->FindView(viewId);
        if(view!=0) {
            return view;
        }
    }
    return 0;
}

// ------------------------------------------------------------------------
void ViewContainer::Accept(ViewVisitor & visitor)
{
    if(visitor.Visit(this)) {
        for (ViewLinkedList::Iterator it = mViews.Begin(); it!=mViews.End(); ++it) {
            it->Accept(visitor);
        }
    }
}

// ------------------------------------------------------------------------
void ViewContainer::OnTransitionStarted(const TransitionStartedEvent & msg)
{
    for (ViewLinkedList::Iterator it = mViews.Begin(); it!=mViews.End(); ++it) {
        it->OnTransitionStarted(msg);
    }
}

// ------------------------------------------------------------------------
void ViewContainer::OnTransitionFinished(const TransitionFinishedEvent & msg)
{
    for (ViewLinkedList::Iterator it = mViews.Begin(); it!=mViews.End(); ++it) {
        it->OnTransitionFinished(msg);
    }
}

// ------------------------------------------------------------------------
void ViewContainer::OnScopeMaskChanged()
{
    Message::ScopeMask effectiveScopeMask = GetScopeMask();
    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        effectiveScopeMask |= it->GetEffectiveScopeMask();
    }
    if (0 != mViewController) {
        effectiveScopeMask |= mViewController->GetScopeMask();
    }
    if (mEffectiveScopeMask != effectiveScopeMask) {
        mEffectiveScopeMask = effectiveScopeMask;
        Base::OnScopeMaskChanged();
    }
}

void ViewContainer::OnInvalidationIdOverflow()
{
    View::OnInvalidationIdOverflow();
    for (ViewLinkedList::Iterator it = mViews.Begin(); it != mViews.End(); ++it) {
        static_cast<void>(it->OnInvalidationIdOverflow());
    }
}

}
