//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ViewAccessor.h"
#include "ViewScene3D.h"
#include "ViewScene2D.h"

#include <Courier/Diagnostics/Log.h>


namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Foundation);

// ------------------------------------------------------------------------
ViewAccessor::ViewAccessor()
{
}

// ------------------------------------------------------------------------
ViewAccessor::~ViewAccessor()
{
}

// ------------------------------------------------------------------------
void ViewAccessor::Finalize()
{
    mItemVector.Clear();
}

// ------------------------------------------------------------------------
bool ViewAccessor::AddItem(const View * view)
{
    if(view!=0) {
        FeatStd::SizeType size = mItemVector.Size();
        HashId key(view->GetId().CStr());
        if(size==0) {
            return mItemVector.Add(Item(view,key));
        }
        Item * item = &mItemVector[0];
        if(key<item->GetHashId()) {
            return mItemVector.Insert(0,Item(view,key));
        }
        item = &mItemVector[size-1];
        if(key>item->GetHashId()) {
            return mItemVector.Add(Item(view,key));
        }
        FeatStd::OffsetType mid = 0;
        if(size>0) {
            FeatStd::OffsetType first = 0;
            FeatStd::OffsetType last = static_cast<FeatStd::OffsetType>(size-1);
            while (first<=last) {
                mid = (first + last) / 2;
                const Item * low = &mItemVector[mid];
                const Item * high = &mItemVector[static_cast<FeatStd::SizeType>(mid+1)];
                if ((low->GetHashId()<key) && (key<high->GetHashId())) {
                    return mItemVector.Insert(static_cast<FeatStd::SizeType>(mid + 1), Item(view, key));
                }
                if (key>low->GetHashId()) {
                    first = mid + 1;
                } else if (key<low->GetHashId()) {
                    last = mid - 1;
                } else {
                    COURIER_LOG_FATAL("ViewAccessor AddItem failed due to exising view with same identifier: %s", view->GetId().CStr());
                    return false;
                }
            }
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool ViewAccessor::RemoveItem(const View * view)
{
    if(view!=0) {
        FeatStd::OffsetType pos = LookupPos(HashId(view->GetId().CStr()));
        if(pos>=0) {
            FEATSTD_DEBUG_ASSERT(mItemVector[pos].GetView()==view);
            return mItemVector.Remove(pos);
        }
    }
    return false;
}

// ------------------------------------------------------------------------
const View * ViewAccessor::Lookup(const HashId & hashId) const
{
    FeatStd::OffsetType pos = LookupPos(hashId);
    return (pos>=0) ? mItemVector[pos].GetView() : 0;
}

// ------------------------------------------------------------------------
FeatStd::OffsetType ViewAccessor::LookupPos(const HashId & hashId) const
{
    FeatStd::SizeType size = mItemVector.Size();
    if(size>0) {
        FeatStd::OffsetType first = 0;
        FeatStd::OffsetType last = FeatStd::OffsetType(size)-1;
        while (first<=last) {
            FeatStd::OffsetType mid = (first + last) / 2;
            const Item * item = &mItemVector[mid];
            FEATSTD_DEBUG_ASSERT(item!=0);
            if (hashId > item->GetHashId()) {
                first = mid + 1;
            } else if (hashId < item->GetHashId()) {
                last = mid - 1;
            } else {
                return mid;
            }
        }
    }
    return -1;
}

// ------------------------------------------------------------------------
bool ViewAccessor::AssertValid() const
{
    for (Int i=0; i < static_cast<Int>(mItemVector.Size()-1); i++) {
        const Item * item1 = &mItemVector[i];
        const Item * item2 = &mItemVector[static_cast<FeatStd::SizeType>(i + 1)];
        if(! ((item1->GetView()!=0) && (item2->GetView()!=0))) {
            return false;
        }
        if(! (item1->GetHashId()<item2->GetHashId())) {
            return false;
        }
        if(item1->GetHashId()==item2->GetHashId()) {
            return false;
        }
    }
    return true;
}

// ------------------------------------------------------------------------
void ViewAccessor::GetEnabledViewSceneVector(bool is2D, ViewPtrVector & viewVector)
{
    viewVector.Clear();
    for (FeatStd::SizeType i = 0; i < mItemVector.Size(); i++) {
        Item * item = &mItemVector[i];
        View * view = const_cast<View*>(item->GetView());
        if(view!=0) {
            if((is2D && view->Is2D()) || ((! is2D) && view->Is3D())) {
                if(view->IsContentLoaded() && view->IsRenderingEnabled()) {
                    (void)viewVector.Add(view);
                }
            }
        }
    }
}

}

