//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_TransitionHandler_h)
#define Courier_Visualization_TransitionHandler_h

#include <Courier/Util/Util.h>
#include <Courier/Visualization/Transition.h>

namespace Courier { 
    class IViewHandler;
    class TransitionFactory;
}

namespace Courier { 

/// @addtogroup COURIER_VISUALIZATION
/// @{
/** The TransitionHandler object is reponsible for holding a list of possible transition implementations.
    Each transition is identified by its id and can therefore found using the transition handler. Besides
    holding the list it is also responsible for sending the TransitionIndMsg when an transition has been finished.
*/
class TransitionHandler 
{
    public:
        ///
        TransitionHandler();

        /// 
        virtual ~TransitionHandler();

        /** Initializes the TransitionHandler.
            @param viewHandler IViewHandler used for rendering and message distribution. 
            @param factory TransitionFactory used for creating transition objects. 
            @return <em>true</em> if initialization was successful,
                    <em>false</em> otherwise. */
        bool Init(IViewHandler * viewHandler, TransitionFactory * factory);

        /** Finishes all running transition to bring the transition handler in a finalized state. */
        void Finalize();

        /// IViewHandler access method. 
        IViewHandler * GetViewHandler() { 
            return mViewHandler; 
        }

        /** Create a transition object of type 'transitionId'.
            @param transitionId the Id of the transition. 
            @return <em>Transition</em> transition with the specified Id is created,
                    <em>0</em> otherwise. */
        Transition * Create(const ItemId & transitionId);

        /** Lookup for a transition specified by its Id and views.
            @param transitionId the Id of the transition. 
            @param firstView the first View of the transition. 
            @param secondView the second View of the transition. 
            @return <em>Transition</em> if an transition with the specified Id is existing,
                    <em>0</em> otherwise. */
        Transition * Get(const ItemId & transitionId, const View * firstView, const View * secondView);

        /** This method is called by the transition in all use cases.
            If the transition was finished automatically, if the transition was reset and if it was stopped.
            In case it was reset the boolean value has to be set to true. When this method is called
            it sends an asynchrounous TransitionIndMsg message with all four parameters.
            @param transition the transition which has finished.
            @param firstView the first view.
            @param secondView the second optional view.
            @param reset if the transition was reset. 
            @return <em>true</em> if finishing was successful,
                    <em>false</em> otherwise. */
        bool OnTransitionFinished(const Transition * transition, const View * firstView, const View * secondView, bool reset) const;

    private:
        /// The used ViewHandler.
        IViewHandler * mViewHandler;

        /// The used TransitionFactory;
        TransitionFactory * mTransitionFactory;            

        /// This array is holding the registered transitions.
        typedef Courier::Vector< Transition * > TransitionVector;
        ///
        TransitionVector mTransitions;

};

/// @}
}

#endif
