//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_TransitionFactory_h)
#define Courier_Visualization_TransitionFactory_h

#include <Courier/Base.h>
#include <Courier/Util/StaticObjectBuffer.h>

namespace Courier {
    class Transition;
    class ItemId;
}

namespace Courier {
/// @addtogroup COURIER_VISUALIZATION
/// @{
/** A TransitionFactory is responsible for creating Transition objects.
    Transitions might be allocated dynamically using FEATSTD_NEW and FEATSTD_DELETE,
    or statically using the macros \ref COURIER_STATIC_TRANSITION_CONFIGURATION, \ref COURIER_STATIC_TRANSITION_NEW
    and \ref COURIER_STATIC_TRANSITION_FREE. In this case memory is "preallocated" and memory is not allocated on heap.
    For source code samples refer to the tutorial section of the documentation.
*/
class TransitionFactory  {
    public:
        ///
        TransitionFactory();

        ///
        virtual ~TransitionFactory();

        /** Returns a Transition object identified by its ItemId id.
            This method has to be overwritten by the derived factory and it is upto the
            customer if the ViewController instances are allocated dynamically or statically.
            @param transitionId the ItemId (type) of the transition.
            @return <em>Transition</em> if the Transition was created,
                    <em>0</em> otherwise. */
        virtual Transition * Create(const ItemId & transitionId) = 0;

        /** Destroys a Transition.
            This method has to be overwritten by the derived factory.
            @param transition the pointer to the Transition. */
        virtual void Destroy(Transition * transition) = 0;
};

/** COURIER_STATIC_TRANSITION_CONFIGURATION has to be use if transitions shall not be allocated dynamically.
    To declare a fixed sized preallocated transition object buffer use it like: COURIER_STATIC_TRANSITION_CONFIGURATION(HideSampleTransition,2);
*/
#define COURIER_STATIC_TRANSITION_CONFIGURATION COURIER_STATIC_OBJECT_BUFFER_CONFIGURATION

/** COURIER_STATIC_TRANSITION_NEW returns a pointer to an allocated transition object of a given type.
    Use it in your code like E.g. transitionPtr = COURIER_STATIC_TRANSITION_NEW(HideSampleTransition)
*/
#define COURIER_STATIC_TRANSITION_NEW(type) new (StaticObjectBuffer< type >::Alloc()) type

/** COURIER_STATIC_TRANSITION_FREE frees a previously allocated transition object of a given type.
    Use it in your code like E.g. COURIER_STATIC_TRANSITION_FREE(HideSampleTransition,transitionPtr)
*/
#define COURIER_STATIC_TRANSITION_FREE(type, transition) \
    static_cast<type*>(transition)->~type(); \
    StaticObjectBuffer< type >::Free(static_cast<type*>(transition));


/// @}
}

#endif
