//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Transition.h"
#include "TransitionHandler.h"
#include "View.h"
#include "VisualizationMsgs.h"

namespace Courier {


// ------------------------------------------------------------------------
Transition::TransitionError Transition::mLastTransitionError = Transition::TransitionSuccess;

// ------------------------------------------------------------------------
Transition::Transition() : mTransitionHandler(0), mFirstView(0), mSecondView(0), mIsActive(false), mWasReset(false)
{
}

// ------------------------------------------------------------------------
Transition::~Transition()
{
    mTransitionHandler = 0;
    mFirstView = 0;
    mSecondView = 0;
}

// ------------------------------------------------------------------------
TransitionHandler * Transition::GetTransitionHandler()
{
    return mTransitionHandler;
}

// ------------------------------------------------------------------------
void Transition::SetTransitionHandler(TransitionHandler * handler)
{
    // might be also zero, Transition is deregistered
    mTransitionHandler = handler;
}

// ------------------------------------------------------------------------
FEATSTD_LINT_CURRENT_SCOPE(1762, "No we don't want to make it const")
bool Transition::IsActive()
{
    return mIsActive;
}

// ------------------------------------------------------------------------
bool Transition::Start(View * firstView, View * secondView, const Payload & optionalPayload)
{
    FEATSTD_DEBUG_ASSERT(mTransitionHandler!=0);
    if(IsActive()) {
        SetLastTransitionError(Transition::StartOnActiveTransitionError);
        return false;
    }

    mFirstView = firstView;
    mSecondView = secondView;
    mWasReset = false;
    mIsActive = false;
    if(mTransitionHandler!=0) {
        mIsActive = true;
        SetLastTransitionError(Transition::TransitionSuccess);
        if(OnExecute(firstView, secondView, optionalPayload)) {
            const View::Ptr ptr1(firstView);
            const View::Ptr ptr2(secondView);
            const TransitionStartedEvent startedEvent(ptr1,ptr2);
            if(firstView!=0) {
                firstView->OnTransitionStarted(startedEvent);
            }
            if(secondView!=0) {
                secondView->OnTransitionStarted(startedEvent);
            }
            return true;
        } else {
            mIsActive = false;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool Transition::Reset()
{
    FEATSTD_DEBUG_ASSERT(mTransitionHandler!=0);

    if(! IsActive()) {
        SetLastTransitionError(Transition::ResetOnInactiveTransitionError);
        return false;
    }

    if(mTransitionHandler!=0) {
        mWasReset = true;
        SetLastTransitionError(Transition::TransitionSuccess);
        return OnReset();
    }
    SetLastTransitionError(Transition::GeneralTransitionError);
    return false;
}

// ------------------------------------------------------------------------
bool Transition::Finish()
{
    FEATSTD_DEBUG_ASSERT(mTransitionHandler!=0);

    if(! IsActive()) {
        SetLastTransitionError(Transition::FinishOnInactiveTransitionError);
        return false;
    }

    if(mTransitionHandler!=0) {
        SetLastTransitionError(Transition::TransitionSuccess);
        return OnFinish();
    }
    SetLastTransitionError(Transition::GeneralTransitionError);
    return false;
}

// ------------------------------------------------------------------------
bool Transition::OnTransitionFinished()
{
    bool rc = false;
    if(mTransitionHandler!=0) {
        const View::Ptr ptr1(mFirstView);
        const View::Ptr ptr2(mSecondView);
        const TransitionFinishedEvent finishedEvent(ptr1,ptr2,mWasReset);
        if(mFirstView!=0) {
            // use mFirstView as parameter because it might be distributed to its childs
            mFirstView->OnTransitionFinished(finishedEvent);
        }

        if(mSecondView!=0) {
            // use mSecondView as parameter because it might be distributed to its childs
            mSecondView->OnTransitionFinished(finishedEvent);
        }
        rc = mTransitionHandler->OnTransitionFinished(this,mFirstView,mSecondView,mWasReset);
    }
    mFirstView = 0;
    mSecondView = 0;
    mIsActive = false;
    return rc;
}
}

