//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_RenderHandler_h)
#define Courier_Visualization_RenderHandler_h

#include <Candera/Candera.h>
FEATSTD_LINT_FILE(967, Renderer.h, "false positive - Renderer.h has valid include guard.")

#include <Courier/Util/Util.h>
#include <Courier/Visualization/Gdu.h>
#include <Courier/Visualization/View.h>
#include <Courier/Visualization/RenderJob.h>
#include <Courier/Visualization/RenderConfiguration.h>
#include <Courier/Platform/RendererHelper.h>

namespace Candera {
    class DefaultAssetProvider;
}

namespace Courier {
    namespace UnitTest {
        class RendererWrapper;
    }

/// @addtogroup COURIER_VISUALIZATION
/// @{
/**
 * The RenderJobStrategy class is used to sort the RenderJobs. The default sort criterias are:
 * 1) offscreen render jobs before Display render jobs.
 * 2) 2d render jobs before 3d render jobs
 * 3) clear render jobs before normal view render jobs
 * 4) lower camera sequence number before higher camera sequence number
 * For custom sort criteria simply derive from RenderJobStrategy and use the Renderer::SetRenderJobStrategy.
 */
class RenderJobStrategy
{
public:
    FEATSTD_RTTI_DECLARATION();

    /**
     * The RenderJobAction defines the next step in the processing of the render jobs for a specific render job.
     */
    typedef enum {
        /**
         * The render job will be processed. Rendering is performed. Partial rendering of nodes is the responsibility of the Candera CameraRenderStrategy.
         */
        ProceedRenderJob,
        /**
        * The render job will not be processed. No rendering is performed.
        * NOTE: The Gdu render state will be RenderTargetIncomplete if a render job is skipped.
        */
        SkipRenderJob
    } RenderJobAction;

    /**
     * Create a render job strategy.
     *  @param earlySwap Set this flag to true if the Gdu has to be swapped as soon as the last render job is completed.
     *                   Otherwise, the Gdu's will be swapped
     */
    RenderJobStrategy(bool earlySwap = false) :
        mEarlySwap(earlySwap)
    {
    }

    /**
     * Any custom render job order can be implemented by implementiong this operator.
     * By default this sorting criterias are used:
     * 1) offscreen render target render jobs before display render target render jobs.
     * 2) 2d render jobs before 3d render jobs.
     * 3) Courier clear render jobs before scene render jobs.
     * 4) camera sequence number (if enabled)
     * @ return true if render job a has to be rendered before b.
     */
    virtual bool operator()(const RenderJob& a, const RenderJob& b) const;

protected:
    friend class RenderJobs;
    friend class Renderer;
    friend class View;

    struct InvalidationDependencyData
    {
        typedef enum {
#ifdef CANDERA_2D_ENABLED
            CameraData2D,
#endif
#ifdef CANDERA_3D_ENABLED
            CameraData3D,
#endif
            CameraDataNone
        } Type;

        InvalidationDependencyData(const Courier::View& view, const Courier::View::InvalidationDependency& invalidationDependency) :
            m_view(view),
            m_type(CameraDataNone),
            m_camera(),
            m_invalidationDependency(invalidationDependency)
        {
        }

#ifdef CANDERA_2D_ENABLED
        InvalidationDependencyData(const Courier::View& view, Candera::Camera2D const* const camera, const Courier::View::InvalidationDependency& invalidationDependency) :
            m_view(view),
            m_type(CameraData2D),
            m_camera(camera),
            m_invalidationDependency(invalidationDependency)
        {
        }
#endif

#ifdef CANDERA_3D_ENABLED
        InvalidationDependencyData(const Courier::View& view, Candera::Camera const* const camera, const Courier::View::InvalidationDependency& invalidationDependency) :
            m_view(view),
            m_type(CameraData3D),
            m_camera(camera),
            m_invalidationDependency(invalidationDependency)
        {
        }
#endif

        const Courier::View& GetView() const
        {
            return m_view;
        }

        Type GetCameraType() const
        {
            return m_type;
        }

#ifdef CANDERA_2D_ENABLED
        Candera::Camera2D const* GetCamera2D() const
        {
            return (m_type == CameraData2D) ? m_camera.m_2D : 0;
        }
#endif

#ifdef CANDERA_3D_ENABLED
        Candera::Camera const* GetCamera3D() const
        {
            return (m_type == CameraData3D) ? m_camera.m_3D : 0;
        }
#endif

        const Courier::View::InvalidationDependency& GetInvalidationDependency() const
        {
            return m_invalidationDependency;
        }

    private:
        union CameraData {
            CameraData() :
                m_void(0)
            {
            }

#ifdef CANDERA_2D_ENABLED
            CameraData(Candera::Camera2D const* const camera) :
                m_2D(camera)
            {
            }
#endif

#ifdef CANDERA_3D_ENABLED
            CameraData(Candera::Camera const* const camera) :
                m_3D(camera)
            {
            }
#endif

            void const* m_void;
#ifdef CANDERA_2D_ENABLED
            Candera::Camera2D const* m_2D;
#endif
#ifdef CANDERA_3D_ENABLED
            Candera::Camera const* m_3D;
#endif
        private:
            CameraData(const CameraData&);
            CameraData& operator=(const CameraData&);
        };

        const Courier::View& m_view;
        const Type m_type;
        const CameraData m_camera;
        const Courier::View::InvalidationDependency& m_invalidationDependency;
    };

    /**
     * Will be called before the render pass (before first render job).
     * @param renderJobs the render jobs of this render pass.
     */
    virtual void OnPreRenderPass(RenderJobVector& /*renderJobs*/)
    {
    }

    /**
     * Will be called after the render pass (after the last render job).
     * @param renderJobs the render jobs of this render pass.
     */
    virtual void OnPostRenderPass(RenderJobVector& /*renderJobs*/)
    {
    }

    /**
     * Will be called before a render job is rendered (not called if the render job is skipped).
     * @param renderJob the render job.
     */
    virtual void OnPreRenderJob(RenderJob& /*renderJob*/)
    {
    }

    /**
     * Will be called after a render job has been rendered (not called if the render job is skipped).
     * @param renderJob the render job.
     */
    virtual void OnPostRenderJob(RenderJob& /*renderJob*/)
    {
    }

    /**
     * Will be called for each render job before it is processed.
     * @param renderJob the render job that will be processed next.
     * @return The render job action that has to be performed for the given render job.
     */
    virtual RenderJobAction GetRenderJobAction(RenderJob& /*renderJob*/)
    {
        return ProceedRenderJob;
    }

    /**
     * Will be called after the last render job of a Gdu has been completed.
     * @param gdu the Gdu that should be swapped.
     * @return true if it should swap early.
     */
    virtual bool OnSwapBuffer(Gdu& /*gdu*/)
    {
        return EarlySwap();
    }

     /**
     * Is called after SwapBuffer is done.
     * @param gdu the Gdu that was swapped.
     */
    virtual void OnPostSwapBuffer(Gdu& gdu) { COURIER_UNUSED(gdu); }

    /**
     * Will be called after a render job has been processed.
     * @param renderJob the render job.
     * @return true if the render job is finished and can be removed.
     */
    virtual bool OnRenderJobFinished(RenderJob& renderJob)
    {
        return (0 == renderJob.GetRenderCounter());
    }

    /**
     * Transform the dirty area of the view to the dirty area that is required for invalidation dependency. By default the dirty area of the invalidation dependency is used as dirty area.
     */
    virtual void TransformDirtyAreaToInvalidationDependency(const FeatStd::Optional<Candera::Rectangle>& /*dirtyArea*/, FeatStd::Optional<Candera::Rectangle>& transformedDirtyArea,
        const Courier::View& /*view*/, const RenderJobStrategy::InvalidationDependencyData& invalidationDependencyData)
    {
        transformedDirtyArea = invalidationDependencyData.GetInvalidationDependency().GetDirtyArea();
    }

    /**
     * Transform the dirty area of the view to the dirty area that is required for invalidation dependency. By default the dirty area of the invalidation dependency is used as dirty area.
     */
    virtual void TransformDirtyAreaToOffscreenReferringViews(const Courier::View& /*view*/, const Gdu& /*offscreenGdu*/, const FeatStd::Optional<Candera::Rectangle>& /*dirtyArea*/, FeatStd::Optional<Candera::Rectangle>& transformedDirtyArea)
    {
        transformedDirtyArea = FeatStd::Optional<Candera::Rectangle>();
    }

#if defined(CANDERA_3D_ENABLED)
    /**
     * Transform the dirty area of the view to the dirty area that is required for invalidation dependency. By default the complete area is used as dirty area.
     */
    virtual void TransformDirtyAreaToCompositionCamera(ViewScene* /*view*/, Courier::Gdu& /*gdu*/, const FeatStd::Optional<Candera::Rectangle>& /*dirtyArea*/, FeatStd::Optional<Candera::Rectangle>& /*transformedDirtyArea*/)
    {
    }
#endif

    /**
     * Merge the new dirty area into the list of existing dirty areas. The default behavior is: the a union of the first existing dirty area and the new dirty area
     * is calculated and the first dirty area is replaced by this union.
     * NOTE: An empty existingDirtyAreas indicates no invalidation has yet occurred. An entry where no rectangle is set indicates the complete render target area is dirty.
     */
    virtual void AddDirtyArea(ViewScene* /*view*/, const FeatStd::Optional<Candera::Rectangle>& newDirtyArea, Vector<FeatStd::Optional<Candera::Rectangle> >& existingDirtyAreas);

    /**
     * Returns the effective render counter for the given render target (the view and camera are provided to enable a easier mapping for specific use cases).
     * By default the given render counter is returned if the gdu is not an offscreen render target. For offscreen render targets the the returned default render count is 1.
     */
    virtual UInt8 GetEffectiveRenderCounter(const ViewScene* view, const Candera::CanderaObject& camera, const Gdu& gdu, UInt8 renderCounter);

    /**
     * Perform the final dirty area processing before rendering the frame. The default behavior merges all dirty areas as one union of all dirty areas.
     * NOTE: An empty result list will indicates that the complete render target area is dirty. Each entry will be separately processed without further preprocessing!
     */
    virtual void ProcessDirtyAreas(const RenderJob& renderJob, const Vector<Vector<FeatStd::Optional<Candera::Rectangle> > >& markedDirtyAreas, Vector<FeatStd::Optional<Candera::Rectangle> >& renderDirtyAreas);

    /**
     * @return the early swap flag.
     */
    bool EarlySwap() const
    {
        return mEarlySwap;
    }

    /**
     * The earls swap flag.
     */
    bool mEarlySwap;
};

/** The Renderer object is responsible for maintaining the Gdu render objects which are loaded from the asset.
    It implements an invalidation mechanism which is used by the ViewHandler and its Views. A customized class
    might be derived from this implementation and used by the ViewHandler. This derived class might override
    the OnXXX virtual methods or override other methods.
*/
class Renderer
{
    friend class ViewHandler;

            public:
        ///
        Renderer();

        ///
        virtual ~Renderer();

        /** Frees resources used by the Gdu objects.
            This method additionally calls OnFinalize before it is executed.
            It Unloads the GraphicDeviceUnit objects and deletes the Gdu array. */
        virtual void Finalize();

        /** Initializes all RenderTargets, which are defined inside the assets.
            @param assetProvider AssetProvider to be used.
            @return <em>true</em> if initializing was successful,
                    <em>false</em> otherwise. */
        virtual bool Init(Candera::DefaultAssetProvider * assetProvider);

        /** Render all invalidated Cameras and swaps all invalidated RenderTargets which fulfill RenderHint conditions.
            This method might be overwritten to customize the render behavior.
            @param renderHint RenderHint defines if 2D or 3D or both shall be rendered.
            @return <em>true</em> if at least one RenderTarget was rendered,
                    <em>false</em> otherwise. */
        virtual bool Render(RenderHint * renderHint);

        /** Loads or Unloads a render target for a certain layer.
            If a RenderTarget on a layer shall be enabled, it will be loaded if not have been already loaded.
            If it shall be disabled and reference count get zero it will be unloaded. If another RenderTarget wants
            to use the layer, the used RenderTarget will be handled depending on the platform. <br/>
            The used RenderTarget will be unloaded and the new one will be loaded.
            This method might be overwritten to customize the render behavior.
            @param enable enable or disable a render target.
            @param renderTarget RenderTarget to be loaded.
            @param forceEnable*/
        virtual void EnableLayer(bool enable, Candera::RenderTarget * renderTarget, bool forceEnable);

        /** Lookup to the Gdu object identified by its RenderTarget. This allows
            accessing/casting the GraphicDeviceUnit class for special purposes, e.g. changing properties of the surface.
            @param renderTarget RenderTarget used for lookup.
            @return <em>Gdu</em> if a fitting Gdu object exists,
                    <em>0</em> otherwise. */
        virtual Gdu * GetGdu(Candera::RenderTarget * renderTarget);

        /** Invalidating a render target. Calling this method marks a render target as invalidated
            and will therefore be processed when Render method is called.
            This method is usually called by the view invalidation methods.
            @param renderTarget RenderTarget to be marked as invalidated. */
        virtual void Invalidate(Candera::RenderTarget* renderTarget);

        /** Invalidating a list of render targets.
            @param renderTargetVector RenderTargets to be marked as invalidated. */
        virtual void Invalidate(RTPtrVector& renderTargetVector);

        /** Invalidating or clearing invalidation flag of all render targets.
            @note If marking all Gdu objects as invalidated this will cause swapping of all render targets. */
        virtual void InvalidateAll(bool invalidate);

        /** Activates or deactivates a layer. All RenderTargets with the specified layer Id
            activated or deactivated.
            This method might be overwritten to customize the render behavior.
            @param layerId the layer id of the render target specified in the asset.
            @param activate boolean flag for enabling disabling the layer.
            @return <em>true</em> if activating was successful,
                    <em>false</em> otherwise. */
        virtual bool ActivateLayer(Int layerId, bool activate);

#if defined(CANDERA_2D_ENABLED)
        /** Queues the 2D camera which shall be rendered.
           @param view
           @param camera2D pointer to the camera which shall be rendered.
           @param renderCounter how often the camera shall be rendered.
           @param dirtyArea
           @param clear true indicates if this is a camera for clearing.
           @param doOffscreenInvalidation true indicates that also related offscreen cameras shall be invalidated.
        */
        virtual void Queue2DJob(ViewScene* view, Candera::Camera2D * camera2D, UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea, bool clear, bool doOffscreenInvalidation = true);

        /** Clears all 2D render targets. */
        void Clear2D(const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());
#endif

#if defined(CANDERA_3D_ENABLED)
        /** Queues the 3D camera which shall be rendered.
           @param view
           @param camera3D pointer to the camera which shall be rendered.
           @param renderCounter how often the camera shall be rendered.
           @param dirtyArea
           @param clear true indicates if this is a camera for clearing.
           @param doOffscreenInvalidation true indicates that also related offscreen cameras shall be invalidated.
        */
        void Queue3DJob(ViewScene* view, Candera::Camera * camera3D, UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea, bool clear, bool doOffscreenInvalidation = true);

        /** Clears all 3D render targets. */
        void Clear3D(const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());
#endif
        /// Delete Render Job
        void DeleteRenderJob(const Candera::CanderaObject* camera);

        /// Allows changing the RenderConfiguration which controls the rendering behavior.
        static void SetRenderConfiguration(const RenderConfiguration & configuration);

        // Returns the configuration used by the Renderer.
        static const RenderConfiguration & GetRenderConfiguration() { return mRenderConfiguration; }

        ///
        struct ViewGduRelation {
            ViewGduRelation(View * view, Gdu* gdu) : mView(view), mGdu(gdu) {}
            ViewGduRelation(const ViewGduRelation & obj) : mView(obj.mView), mGdu(obj.mGdu) {}
            bool operator==(const ViewGduRelation & obj) const {
                return (obj.mGdu==mGdu) && (obj.mView==mView);
            }

            View * mView;
            Gdu* mGdu;
        };
        typedef Courier::Vector< ViewGduRelation > ViewGduRelations;

        ///
        ViewGduRelations mViewGduRelations;
        ///
        void AddViewGduRelation(const ViewGduRelation & relation);
        ///
        void RemoveViewGduRelation(const ViewGduRelation & relation);
        ///
        void RemoveViewGduRelations(const View * view);

        /// Wakes render component.
        static void WakeUpAllRenderComponents();

        /// Delete Render Jobs
        void DeleteRenderJobs(const Candera::RenderTarget * renderTarget, bool /*is2D*/)
        {
            DeleteRenderJobs(renderTarget);
        }
        /// Delete Render Jobs
        void DeleteRenderJobs(const Candera::RenderTarget * renderTarget);
        /// Delete Render Jobs
        void DeleteRenderJobs(const Gdu* gdu);

        /// Sets flag for determining if load/unload on a certain layer is requested by 2D View
        void SetIs2DRequest(bool is2D) { mIs2DRequest = is2D; }

        /// Checks if a a load/unload of a certain layer is requested by 2D View
        bool Is2DRequest() const { return mIs2DRequest; }

        /// Set a custom RenderJobStrategy.
        void SetRenderJobStrategy(RenderJobStrategy* renderJobStrategy);
        /// Returns the custom RenderJobStrategy or the default implementation if no custom RenderJobStrategy is set.
        RenderJobStrategy& GetRenderJobStrategy();

#if defined(CANDERA_GPU_SIMULATION)
        /// Triggers a repaint of the display content. Only for Simulation.
       void ForceKickDisplay() { mForceKick = true; }
#endif

        /// Return a const reference to the GDUs
        const Courier::Vector<Gdu*> &GetGDUs() const { return mGDU; }

    protected:
        friend class RenderTargetDestroyEventListener;
        class RenderTargetDestroyEventListener : FeatStd::EventListener
        {
        public:
            RenderTargetDestroyEventListener();
            void Init(Renderer* renderer);
            void Finalize();
            virtual FeatStd::EventResult::Enum OnEvent(const FeatStd::Event& event);

        private:
            Renderer* mRenderer;
        };

        /// RenderTargetDestroyEventListener for recognizing the culture changes.
        RenderTargetDestroyEventListener mRenderTargetDestroyEventListener;


        struct Measure {
            Measure() : mFramesPerSecond(0), mFrameCount(0), mFramesTimespan(0), mPreviousFrameTimeStamp(0), mLastFrameDuration(0) {}
            UInt16 mFramesPerSecond;
            UInt16 mFrameCount;
            FeatStd::SizeType mFramesTimespan;
            FeatStd::SizeType mPreviousFrameTimeStamp;
            FeatStd::SizeType mLastFrameDuration;
        };

        /** Might be overwritten by a derived class. It is called after the Init method is executed.
            @return <em>true</em> if Init method shall return success,
                    <em>false</em> otherwise. */
        virtual bool OnInitialize();

        /// Might be overwritten by a derived class. It is called before the Finalize method is executed.
        virtual void OnFinalize();

        /** Might be overwritten by a derived class. It is called if the activation state of a layer is changed.
            @param layerId the layer Id.
            @param activate true if activated, false otherwise. */
        virtual bool OnLayerActivationChanged(Int layerId, bool activate);

        /// Invalidate all found Views which are using Offscreen render targets.
        void InvalidateOffscreenReferringViews(const Gdu* offscreenGdu, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

        /// A derived class may access the asset provider directly.
        Candera::DefaultAssetProvider * mAssetProvider;

        typedef Gdu* GduPtr;

        typedef Courier::Vector< GduPtr > GraphicDeviceUnitVector;

        /// A derived class may access the Gdu vector directly.
        GraphicDeviceUnitVector mGDU;

        ///
        struct OffScreenGdu {
            OffScreenGdu() : mOffScreenGdu(0), mRefCount(0) {}
            OffScreenGdu(Gdu* gdu) : mOffScreenGdu(gdu), mRefCount(1) {}
            OffScreenGdu(const OffScreenGdu& ref) : mOffScreenGdu(ref.mOffScreenGdu), mRefCount(ref.mRefCount) {}
            ~OffScreenGdu() { mOffScreenGdu = 0; mRefCount = 0; }

            Gdu* mOffScreenGdu;
            Int mRefCount;
        };

        typedef Courier::Vector< OffScreenGdu > OffScreenGduVector;
        ///
        OffScreenGduVector mOffscreenGduVector;
        ///
        void AcquireOffScreenGdu(Gdu* gdu);
        ///
        void ReleaseOffScreenGdu(const Gdu* gdu);

        /// Wakes render component.
        void WakeUpRenderComponent(bool is2D) const;

        /// Helper method which swaps a buffer.
        bool SwapBuffer(Gdu * gdu, RenderHint * renderHint, bool force2D, bool force3D);

        void EnableGdu(bool enable, Gdu* lookupGdu, bool forceEnable);

        /// The render jobs which shall be rendered.
        RenderJobs mRenderJobs;


#if defined(CANDERA_GPU_SIMULATION)
        bool mForceKick;
#endif

        // The configuration provided by the framework.
        static RenderConfiguration mRenderConfiguration;

#if defined(CANDERA_3D_ENABLED)
        const Measure& GetMeasure3D() const { return mMeasure3D; }
#endif

#if defined(CANDERA_2D_ENABLED)
        const Measure& GetMeasure2D() const { return mMeasure2D; }
#endif

        /** Will be called on every render cycle and computes frames per second and last frame duration.
            @param hint defines if 2D, 3D or both shall be measured. */
        virtual void ComputeFramesPerSecond(RenderHint* hint);

    private:
        friend class UnitTest::RendererWrapper;

        friend class RenderJobs;
        void SwapBuffer(Gdu* gdu, RenderHint* renderHint);
        static void ComputeFramesPerSecond(Measure& measure);

#if defined(CANDERA_3D_ENABLED)
        Measure mMeasure3D;
#endif

#if defined(CANDERA_2D_ENABLED)
        Measure mMeasure2D;
#endif
        bool mContinueExecution;
        RenderJobStrategy* mRenderJobStrategy;
        bool mIs2DRequest;
};

/// @}
}

#endif
