//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Renderer.h"

#include "RenderHint.h"
#include <Courier/Diagnostics/Log.h>
#include <Courier/Messaging/MessagingMsgs.h>
#include <Courier/Visualization/ViewScene.h>
#include <CanderaPlatform/Device/Common/Base/Display.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaPlatform/OS/TimePlatform.h>
#include <CanderaAssetLoader/AssetLoader.h>
#include <FeatStd/Util/StaticObject.h>
#include <Courier/Visualization/ComponentWakeupHelper.h>

#if defined(COURIER_RENDERING_MONITOR_ENABLED)
    #include "RenderingMonitor.h"
#endif

#include <CanderaPlatform/Device/Common/Base/SimulatedDisplay.h>
FEATSTD_LINT_NEXT_EXPRESSION(537, "this repeated include file is dependent of the used platform")
#include <CanderaPlatform/Device/Common/Base/DevicePackageInterface.h>
#include "AssetAccessor.h"

namespace Courier {
    FEATSTD_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);
namespace Internal {
#if !defined(COURIER_DEFAULT_GDU_COUNT)
static const FeatStd::UInt32 cCOURIER_DEFAULT_GDU_COUNT = 8;
#define COURIER_DEFAULT_GDU_COUNT Courier::Internal::cCOURIER_DEFAULT_GDU_COUNT
#else
static const FeatStd::UInt32 cCOURIER_DEFAULT_GDU_COUNT = COURIER_DEFAULT_GDU_COUNT;
#endif

static void unused(bool value)
{
   FEATSTD_UNUSED(value);
}

static bool forceInitRendererComponentWakeupHelper =
    ComponentWakeupHelper<ComponentId(ComponentType::Renderer)>::Init()
#if defined(CANDERA_2D_ENABLED)
    && ComponentWakeupHelper<ComponentId(ComponentType::Renderer2D)>::Init()
#endif
#if defined(CANDERA_3D_ENABLED)
    && ComponentWakeupHelper<ComponentId(ComponentType::Renderer3D)>::Init()
#endif
    ;

} // namespace Internal

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

using namespace Candera;

// ------------------------------------------------------------------------
Renderer::RenderTargetDestroyEventListener::RenderTargetDestroyEventListener() :
mRenderer(0)
{
}

void Renderer::RenderTargetDestroyEventListener::Init(Renderer* renderer)
{
    mRenderer = renderer;
    if (0 != mRenderer) {
        if (!Candera::Internal::InternalEventSourceAccess<Candera::RenderTarget>::GetEventSource().AddEventListener(this)) {
            FEATSTD_LOG_WARN("Error adding render target destroy event listener to event source!");
        }
    }
}

void Renderer::RenderTargetDestroyEventListener::Finalize()
{
    if (0 != mRenderer) {
        if (!Candera::Internal::InternalEventSourceAccess<Candera::RenderTarget>::GetEventSource().RemoveEventListener(this)) {
            FEATSTD_LOG_WARN("Error removing render target destroy event listener from event source!");
        }
        mRenderer = 0;
    }
}

FeatStd::EventResult::Enum Renderer::RenderTargetDestroyEventListener::OnEvent(const FeatStd::Event& event)
{
    if (0 != mRenderer) {
        typedef Candera::Internal::RenderTargetEventListenerCollection<Candera::RenderTarget>::TEvent<Candera::Internal::RenderTargetEventListenerCollection<Candera::RenderTarget>::Destroy> DestroyEvent;
        const DestroyEvent* destroyEvent = Candera::Dynamic_Cast<const DestroyEvent*>(&event);
        if (0 != destroyEvent) {
            Candera::RenderTarget* renderTarget = destroyEvent->GetRenderTarget();
            if (0 != renderTarget) {
                SizeType i = 0;
                while (i < mRenderer->mGDU.Size()) {
                    if (mRenderer->mGDU[i]->GetRenderTarget() == renderTarget) {
                        FEATSTD_DELETE(mRenderer->mGDU[i]); mRenderer->mGDU[i] = 0;
                        if (!mRenderer->mGDU.Remove(i)) {
                            FEATSTD_LOG_WARN("Error removing GDU from renderer!");
                        }
                    }
                    else {
                        ++i;
                    }
                }
                mRenderer->DeleteRenderJobs(renderTarget);
            }
        }
    }
    return FeatStd::EventResult::Proceed;
}

// ------------------------------------------------------------------------

RenderConfiguration Renderer::mRenderConfiguration;

static SimulatedDisplay* FindSimulatedDisplay(Int displayId)
{
    Display* display = DevicePackageInterface::GetDisplay(displayId);
    return (0 != display) ? display->ToSimulatedDisplay() : 0;
}

// ------------------------------------------------------------------------
void RenderJobStrategy::AddDirtyArea(ViewScene* /*view*/, const FeatStd::Optional<Candera::Rectangle>& newDirtyArea, Vector<FeatStd::Optional<Candera::Rectangle> >& existingDirtyAreas)
{
    if (newDirtyArea.IsSet()) {
        if (existingDirtyAreas.Size() < 1) {
            bool rc = existingDirtyAreas.Reserve(1);
            FEATSTD_DEBUG_ASSERT(rc);
            rc = existingDirtyAreas.Add(newDirtyArea) && rc;
            FEATSTD_DEBUG_ASSERT(rc);
            FEATSTD_UNUSED(rc);
        }
        else if (existingDirtyAreas[0].IsSet()) {
            Candera::Rectangle rectangle(*existingDirtyAreas[0]);
            rectangle.Union(*newDirtyArea);
            existingDirtyAreas[0] = rectangle;
        }
        else {
            // the complete area is already dirty.
        }
    }
    else {
        // the complete area is dirty.
        if (existingDirtyAreas.Size() != 1) {
            FEATSTD_DEBUG_ASSERT(existingDirtyAreas.Size() == 0);
            bool rc = existingDirtyAreas.Reserve(1);
            FEATSTD_DEBUG_ASSERT(rc);
            existingDirtyAreas.Clear();
            rc = existingDirtyAreas.Add(newDirtyArea) && rc;
            FEATSTD_DEBUG_ASSERT(rc);
            FEATSTD_UNUSED(rc);
        }
        else {
            existingDirtyAreas[0] = newDirtyArea;
        }
    }
}

// ------------------------------------------------------------------------
UInt8 RenderJobStrategy::GetEffectiveRenderCounter(const ViewScene* /*view*/, const Candera::CanderaObject& /*camera*/, const Gdu& gdu, UInt8 renderCounter)
{
    return (gdu.IsOffscreen()) ? 1 : renderCounter;
}

// ------------------------------------------------------------------------
void RenderJobStrategy::ProcessDirtyAreas(const RenderJob& /*renderJob*/, const Vector<Vector<FeatStd::Optional<Candera::Rectangle> > >& markedDirtyAreas, Vector<FeatStd::Optional<Candera::Rectangle> >& renderDirtyAreas)
{
    renderDirtyAreas.Clear();
    bool completeAreaIsDirty = false;
    bool hasRenderDirtyArea = false;
    Candera::Rectangle renderDirtyArea;
    for (FeatStd::SizeType i = 0; i < markedDirtyAreas.Size(); ++i) {
        const Vector<FeatStd::Optional<Candera::Rectangle> >& currentDirtyAreas = markedDirtyAreas[i];
        for (FeatStd::SizeType j = 0; j < currentDirtyAreas.Size(); ++j) {
            const FeatStd::Optional<Candera::Rectangle>& currentDirtyArea = currentDirtyAreas[j];
            if (currentDirtyArea.IsSet()) {
                if (hasRenderDirtyArea) {
                    renderDirtyArea.Union(*currentDirtyArea);
                }
                else {
                    hasRenderDirtyArea = true;
                    renderDirtyArea = *currentDirtyArea;
                }
            } else {
                completeAreaIsDirty = true;
                break;
            }
        }
    }
    if ((!completeAreaIsDirty) && hasRenderDirtyArea) {
        bool rc = renderDirtyAreas.Add(FeatStd::Optional<Candera::Rectangle>(renderDirtyArea));
        FEATSTD_DEBUG_ASSERT(rc);
        FEATSTD_UNUSED(rc);
    }
}

// ------------------------------------------------------------------------
Renderer::Renderer() :
    mAssetProvider(0)
#if defined(CANDERA_GPU_SIMULATION)
    , mForceKick(false)
#endif
    , mContinueExecution(false)
    , mRenderJobStrategy(0)
    , mIs2DRequest(false)
{
    if (!mGDU.Reserve(Courier::Internal::cCOURIER_DEFAULT_GDU_COUNT)) {
        FEATSTD_PANIC("Can't reserve memory for Gdu objects");
    }
    mRenderTargetDestroyEventListener.Init(this);
}

// ------------------------------------------------------------------------
Renderer::~Renderer()
{
    mAssetProvider = 0;
    mRenderTargetDestroyEventListener.Finalize();
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1740, Courier::Renderer::mRenderJobStrategy, "Normally this is a singleton object. If an own object is used, it has to be freed by the application which creates the instance")
}

// ------------------------------------------------------------------------
class GduDescendeOrderByUploadIdx
{
public:
    bool operator()(const Gdu * lhs, const Gdu * rhs) const {
        return lhs->GetUploadRequestNo() > rhs->GetUploadRequestNo();
    }
};


// ------------------------------------------------------------------------
void Renderer::Finalize()
{
    mRenderTargetDestroyEventListener.Finalize();
    OnFinalize();
    static GduDescendeOrderByUploadIdx sortObj;
    mGDU.Sort(sortObj);
    for (SizeType i = 0 ; i < mGDU.Size(); ++i) {
        Gdu * gdu = mGDU[i];
        if (0 != gdu) {
            gdu->Finalize();
            FEATSTD_DELETE(gdu);
        }
    }
    mGDU.Clear();
    Courier::Internal::RendererHelper::PlatformClean();
}

// ------------------------------------------------------------------------
bool Renderer::Init(Candera::DefaultAssetProvider * assetProvider)
{
    FEATSTD_DEBUG_ASSERT(assetProvider != 0);

    mAssetProvider = assetProvider;

#ifdef FEATSTD_LOG_ENABLED
    bool is2D = false;
#if defined(CANDERA_2D_ENABLED)
    FEATSTD_LINT_NEXT_EXPRESSION(838, "this is okay that we dont use previous is2D")
    is2D = true;
#endif

    bool is3D = false;
#if defined(CANDERA_3D_ENABLED)
    FEATSTD_LINT_NEXT_EXPRESSION(838, "this is okay that we dont use previous is3D")
    is3D = true;
#endif

    bool isMonitor = false;
#if defined(COURIER_RENDERING_MONITOR_ENABLED)
    FEATSTD_LINT_NEXT_EXPRESSION(838, "this is okay that we dont use previous isMonitor")
    isMonitor = true;
#endif

    COURIER_LOG_INFO("Build: CANDERA_2D_ENABLED is %s, CANDERA_3D_ENABLED is %s, COURIER_RENDERING_MONITOR_ENABLED is %s",
                        (is2D ? "ON": "OFF"), (is3D ? "ON": "OFF"), (isMonitor ? "ON": "OFF"));

    COURIER_LOG_INFO("RenderConfiguration: ShallLoadAllRenderTargetsAtStartup is %s, ShallUseInvalidation is %s, ShallUseCourierSwapping is %s,"
                     " ShallUseGlobalCameraSequenceNumber is %s, ShallUseWakeUpRenderMechanism is %s",
                        (mRenderConfiguration.ShallLoadAllRenderTargetsAtStartup() ? "TRUE": "FALSE"),
                        (mRenderConfiguration.ShallUseInvalidation() ? "TRUE": "FALSE"),
                        (mRenderConfiguration.ShallUseCourierSwapping() ? "TRUE": "FALSE"),
                        (mRenderConfiguration.ShallUseGlobalCameraSequenceNumber() ? "TRUE": "FALSE"),
                        (mRenderConfiguration.ShallUseWakeUpRenderMechanism() ? "TRUE": "FALSE")
                    );

#if defined(CANDERA_GPU_SIMULATION)
    COURIER_LOG_INFO("RenderConfiguration: ShallUseKickDisplayUpdate is %s",(mRenderConfiguration.ShallUseKickDisplayUpdate() ? "TRUE": "FALSE"));
#endif
#endif // FEATSTD_LOG_ENABLED

    FeatStd::SizeType gduCount = 0;
    if (0 != mAssetProvider) {
        const AssetDescriptor& assetDescriptor = mAssetProvider->GetAssetDescriptor();
        Int scenes2DCount = Internal::AssetLibHelper::GetCount(mAssetProvider, Candera::Scene2DLib);
        Int scenes3DCount = Internal::AssetLibHelper::GetCount(mAssetProvider, Candera::SceneLib);

        FEATSTD_UNUSED2(scenes2DCount, scenes3DCount); // if logging is disabled, variable/parameter is not used
            COURIER_LOG_INFO("Scenes Count 2D(%d), 3D(%d)", scenes2DCount, scenes3DCount);
        for (AssetDescriptor::AssetIdIterator displayIt = assetDescriptor.GetAssetIdIterator(DisplayLib); displayIt.IsValid(); ++displayIt) {
            const Char* displayName = mAssetProvider->GetNameById(DisplayLib, *displayIt, 0);
            if (0 == displayName) {
                FEATSTD_LOG_DEBUG("GetNameById is not valid");
                displayName = "INVALID";
            }
            Int width = -1;
            Int height = -1;
            Int displayId = mAssetProvider->GetDisplayDataById(*displayIt, width, height);
            COURIER_LOG_INFO("Display (#%d, %s(%d,%d)) detected.", displayId, displayName, width, height);
            if (mRenderConfiguration.ShallUseKickDisplayUpdate()) {
                SimulatedDisplay* simulatedDisplay = FindSimulatedDisplay(displayId);
                if (0 != simulatedDisplay) {
                    simulatedDisplay->SetAutoKickEnabled(false);
                }
            }
        }
        for (AssetDescriptor::AssetIdIterator renderTargetIt = assetDescriptor.GetAssetIdIterator(RenderTargetLib); renderTargetIt.IsValid(); ++renderTargetIt) {
            GraphicDeviceUnit* gdu = mAssetProvider->GetGraphicDeviceUnitById(*renderTargetIt);
            FEATSTD_LINT_CURRENT_SCOPE(429, "gduObj is freed at Finalize call")
            if (0 != gdu) {
                Gdu * gduObj = FEATSTD_NEW(Gdu)(gdu, mGDU.Size());
                FEATSTD_LINT_NEXT_EXPRESSION(774, "allocation may fail")
                if ((0 != gduObj) && mGDU.Add(gduObj)) {
                    gduObj->Log();
                    ++gduCount;
                }
            }
        }
        if (mRenderConfiguration.ShallLoadAllRenderTargetsAtStartup()) {
            for (SizeType i = 0; i < mGDU.Size(); ++i) {
                Gdu* gdu = mGDU[i];
                if ((0 != gdu) && (!gdu->IsLoaded())) {
                    EnableLayer(true, gdu->GetRenderTarget(), true);
                }
            }
        }
    }
    if (0 == gduCount) {
        COURIER_LOG_ERROR("No GraphicDeviceUnit(s) found in asset.");
    }
    else {
        COURIER_LOG_INFO("%d GraphicDeviceUnit(s) found in asset.", gduCount);
    }
    return ((gduCount > 0) && OnInitialize());
}

// ------------------------------------------------------------------------
void Renderer::SetRenderConfiguration(const RenderConfiguration & configuration)
{
    mRenderConfiguration = configuration;
    COURIER_LOG_INFO("RenderConfiguration changed.");
}

// ------------------------------------------------------------------------
void Renderer::SwapBuffer(Gdu* gdu, RenderHint * renderHint) {
    if ((0 != gdu) && mRenderConfiguration.ShallUseCourierSwapping()) {
        if (gdu->ShallSync()) {
            gdu->Sync();
        }
        bool force2D = mRenderConfiguration.ShallUseRenderComponent2DForSwapBuffer();
        bool force3D = mRenderConfiguration.ShallUseRenderComponent3DForSwapBuffer();
        if (SwapBuffer(gdu, renderHint, force2D, force3D)) {
            mContinueExecution = true;
        }
    }
}

// ------------------------------------------------------------------------
bool Renderer::Render(RenderHint * renderHint)
{
#if defined(CANDERA_3D_ENABLED)
    const_cast<Candera::Renderer::Statistics&>(Candera::Renderer::GetStatistics()).Reset();
#endif
#if defined(CANDERA_2D_ENABLED)
    const_cast<Candera::Renderer2D::Statistics&>(Candera::Renderer2D::GetStatistics()).Reset();
#endif

    bool rc = true;

#if defined(COURIER_RENDERING_MONITOR_ENABLED)
    if (mRenderConfiguration.ShallUseInvalidation()) {
#if defined(CANDERA_2D_ENABLED)
        static_cast<void>(Candera::Monitor::RTMonitor::HandlePaused(2));
#endif
#if defined(CANDERA_3D_ENABLED)
        static_cast<void>(Candera::Monitor::RTMonitor::HandlePaused(3));
#endif
    }
#endif
#if defined(CANDERA_GPU_SIMULATION)
    if (mRenderConfiguration.ShallUseKickDisplayUpdate()) {
        if (0 != mAssetProvider) {
            const AssetDescriptor& assetDescriptor = mAssetProvider->GetAssetDescriptor();
            for (AssetDescriptor::AssetIdIterator displayIt = assetDescriptor.GetAssetIdIterator(DisplayLib); displayIt.IsValid(); ++displayIt) {
                Int dummy;
                Int displayId = mAssetProvider->GetDisplayDataById(*displayIt, dummy, dummy);
                SimulatedDisplay* simulatedDisplay = FindSimulatedDisplay(displayId);
                if (0 != simulatedDisplay) {
                    simulatedDisplay->SetAutoKickEnabled(false);
                }
            }
        }
    }
#endif

#if defined(COURIER_RENDERING_MONITOR_ENABLED)
    MONITOR_MARKER_BEGIN_FRAME();
#endif

    mContinueExecution = !mRenderConfiguration.ShallUseCourierSwapping();
    if (mRenderConfiguration.ShallUseInvalidation()) {
#if defined(COURIER_RENDERING_MONITOR_ENABLED) 
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::Frame));
#endif
        rc = mRenderJobs.Render(renderHint, *this);
    }
    else {
#if defined(CANDERA_2D_ENABLED)
        if (renderHint->ShallRender2D()) {
            Candera::Renderer2D::RenderAllCameras();
        }
#endif
#if defined(CANDERA_3D_ENABLED)
        if (renderHint->ShallRender3D()) {
            rc = Candera::Renderer::RenderAllCameras();
        }
#endif
    }

#if defined(COURIER_RENDERING_MONITOR_ENABLED)
    MONITOR_MARKER_END_FRAME();
#endif

    if (!rc) {
        COURIER_LOG_ERROR("Rendering failed.");
    }
    if (mRenderConfiguration.ShallUseCourierSwapping() && (!GetRenderJobStrategy().EarlySwap())) {
        bool force2D = mRenderConfiguration.ShallUseRenderComponent2DForSwapBuffer();
        bool force3D = mRenderConfiguration.ShallUseRenderComponent3DForSwapBuffer();
        for (SizeType s = 0; s < mGDU.Size(); ++s) {
            Gdu* gdu = mGDU[s];
            if ((0 != gdu) && gdu->ShallSync()) {
                gdu->Sync();
                (void)SwapBuffer(gdu, renderHint, force2D, force3D);
            }
        }
        for (SizeType i = 0; i < mGDU.Size(); ++i) {
            Gdu* gdu = mGDU[i];
            if ((0 != gdu) && (!gdu->ShallSync())) {
                (void)SwapBuffer(gdu, renderHint, force2D, force3D);
            }
        }
    }

#if defined(CANDERA_GPU_SIMULATION)
    if (mRenderConfiguration.ShallUseKickDisplayUpdate() && (0 != mAssetProvider)) {
        const AssetDescriptor& assetDescriptor = mAssetProvider->GetAssetDescriptor();
        for (AssetDescriptor::AssetIdIterator displayIt = assetDescriptor.GetAssetIdIterator(DisplayLib); displayIt.IsValid(); ++displayIt) {
            bool kick = false;
            for (AssetDescriptor::AssetIdIterator renderTargetIt = assetDescriptor.GetAssetIdIterator(RenderTargetLib); renderTargetIt.IsValid(); ++renderTargetIt) {
                GraphicDeviceUnit * gdu = mAssetProvider->GetGraphicDeviceUnitById(*renderTargetIt);
                if (0 != gdu) {
                    Gdu tempGdu(gdu, 0);
                    Gdu* lookupGdu = GetGdu(tempGdu.GetRenderTarget());
                    if ((0 != lookupGdu) && lookupGdu->IsLoaded() && lookupGdu->IsActive()) { 
                        if (mRenderConfiguration.ShallUseInvalidation()) {
                            if (lookupGdu->IsInvalidated()) {
                                kick = true;
                                break;
                            }
                        }
                        else {
                            kick = true;
                            break;
                        }
                    }
                }
            }
            if (mForceKick) {
                mForceKick = false;
                kick = true;
            }
            if (kick) {
                Int dummy;
                Int displayId = mAssetProvider->GetDisplayDataById(*displayIt, dummy, dummy);
                SimulatedDisplay* simulatedDisplay = FindSimulatedDisplay(displayId);
                if (0 != simulatedDisplay) {
                    (void)simulatedDisplay->KickDisplayUpdate();
                }
            }
        }
    }
#endif

    if (mRenderConfiguration.ShallUseInvalidation()) {
        return mContinueExecution || (mRenderJobs.Size() > 0);
    }
    return true;
}

// ------------------------------------------------------------------------
bool Renderer::SwapBuffer(Gdu * gdu, RenderHint * renderHint, bool force2D, bool force3D)
{
    if (0 != gdu) {
        if (gdu->IsActive() && gdu->IsInvalidated()) {
            if (!gdu->HasCameraWithRenderStrategy()) {
                bool normalSwap = (gdu->Is2D() == renderHint->ShallRender2D()) || (gdu->Is3D() == renderHint->ShallRender3D());
                // make the check if the call shall effectively swap the buffers
                if (force2D || force3D) {
                    if ((force2D && renderHint->ShallRender2D()) || (force3D && renderHint->ShallRender3D())) {
                        normalSwap = normalSwap && gdu->GetRenderTarget()->Activate();
                        gdu->GetRenderTarget()->SwapBuffers();
                        GetRenderJobStrategy().OnPostSwapBuffer(*gdu);
                        mContinueExecution = true;
                    }
                }
                else if (normalSwap) {
                    normalSwap = gdu->GetRenderTarget()->Activate();
                    gdu->GetRenderTarget()->SwapBuffers();
                    GetRenderJobStrategy().OnPostSwapBuffer(*gdu);
                    mContinueExecution = true;
                }
                else {
                    // MISRA
                }
                // this check is for detecting that rendering shall continue (Execute of RenderComponent)
                if (normalSwap) {
                    return true;
                }
            }
        }
    }
    return false;
}

// ------------------------------------------------------------------------
Gdu * Renderer::GetGdu(Candera::RenderTarget* renderTarget)
{
    if (0 != renderTarget) {
        for (SizeType i = 0; i < mGDU.Size(); ++i) {
            Gdu* gdu = mGDU[i];
            if (0 != gdu) {
                RenderTarget* rt = gdu->GetRenderTarget3D();
                if (rt == renderTarget) {
                    return gdu;
                }
                else {
                    rt = gdu->GetRenderTarget2D();
                    if (rt == renderTarget) {
                        return gdu;
                    }
                }
            }
        }
    }
    return 0;
}

// ------------------------------------------------------------------------
bool Renderer::ActivateLayer(Int layerId, bool activate)
{
    bool fireEvent = false;
    for (SizeType i = 0; i < mGDU.Size(); ++i) {
        Gdu* gdu = mGDU[i];
        if (0 != gdu) {
            FEATSTD_LINT_NEXT_EXPRESSION(731, "We want compare boolean values !")
            if ((layerId == gdu->GetLayerId()) && (gdu->IsActive() != activate)) {
                gdu->SetActivate(activate);
                fireEvent = true;
            }
        }
    }
    if (fireEvent) {
        fireEvent = OnLayerActivationChanged(layerId, activate);
    }
    return fireEvent;
}

// ------------------------------------------------------------------------
void Renderer::WakeUpAllRenderComponents()
{
    if (mRenderConfiguration.ShallUseWakeUpRenderMechanism()) {
#if defined(CANDERA_3D_ENABLED)
        Internal::ComponentWakeupHelper<ComponentId(ComponentType::Renderer3D)>::Wakeup();
#endif
#if defined(CANDERA_2D_ENABLED)
        Internal::ComponentWakeupHelper<ComponentId(ComponentType::Renderer2D)>::Wakeup();
#endif
        Internal::ComponentWakeupHelper<ComponentId(ComponentType::Renderer)>::Wakeup();
    }
}

// ------------------------------------------------------------------------
void Renderer::WakeUpRenderComponent(bool is2D) const
{
    if (mRenderConfiguration.ShallUseWakeUpRenderMechanism()) {
        Int wakeups = 0;
        if (!is2D) {
            if (mRenderJobs.HasJob(false)) {
#if defined(CANDERA_3D_ENABLED)
                Internal::ComponentWakeupHelper<ComponentId(ComponentType::Renderer3D)>::Wakeup();
#endif
            }
            ++wakeups;
        }
        if (is2D) {
            if (mRenderJobs.HasJob(true)) {
#if defined(CANDERA_2D_ENABLED)
                Internal::ComponentWakeupHelper<ComponentId(ComponentType::Renderer2D)>::Wakeup();
#endif
            }
            ++wakeups;
        }
        if (0 != wakeups) {
            Internal::ComponentWakeupHelper<ComponentId(ComponentType::Renderer)>::Wakeup();
        }
    }
}

// ------------------------------------------------------------------------
void Renderer::InvalidateOffscreenReferringViews(const Gdu* offscreenGdu, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if ((0 != offscreenGdu) && offscreenGdu->IsOffscreen() && offscreenGdu->IsLoaded()) {
        for (SizeType i = 0; i < mViewGduRelations.Size(); ++i) {
            if (offscreenGdu == mViewGduRelations[i].mGdu) {
                View * scene = mViewGduRelations[i].mView;
                if (0 != scene) {
                    FeatStd::Optional<Candera::Rectangle> transformedDirtyArea;
                    GetRenderJobStrategy().TransformDirtyAreaToOffscreenReferringViews(*scene, *offscreenGdu, dirtyArea, transformedDirtyArea);
                    scene->InvalidateOffscreenHelper(transformedDirtyArea);
                }
            }
        }
    }
}

// ------------------------------------------------------------------------
void Renderer::AddViewGduRelation(const ViewGduRelation & relation) 
{
    for (SizeType i = 0; i < mViewGduRelations.Size(); ++i) {
        if (mViewGduRelations[i] == relation) {
            return;
        }
    }
    if (0 != relation.mView) {
        COURIER_LOG_INFO("Gdu '%s' (%08x) in View '%s' found", relation.mGdu->GetGdu()->GetName(), relation.mGdu, relation.mView->GetId().CStr());
        (void)mViewGduRelations.Add(relation); 
    }
}

// ------------------------------------------------------------------------
void Renderer::RemoveViewGduRelation(const ViewGduRelation & relation)
{
    for (SizeType i = 0; i < mViewGduRelations.Size(); ++i) {
        if (mViewGduRelations[i] == relation) {
            static_cast<void>(mViewGduRelations.Remove(i));
            return;
        }
    }
}

// ------------------------------------------------------------------------
void Renderer::RemoveViewGduRelations(const View * view)
{
    mRenderJobs.RemoveViewRelations(view);
    bool deleteEntry;
    do {
        deleteEntry = false;
        SizeType idx;
        for (idx = 0; idx < mViewGduRelations.Size(); ++idx) {
            if(mViewGduRelations[idx].mView == view) {
                deleteEntry = true;
                break;
            }
        }
        if (deleteEntry) {
            (void)mViewGduRelations.Remove(idx);
        }
    } while(deleteEntry);
}

static bool ValidateAndAdjustDirtyArea(FeatStd::Optional<Candera::Rectangle>& dirtyArea, Candera::RenderTarget& renderTarget)
{
    if (dirtyArea.IsSet()) {
        Candera::Rectangle dirtyAreaRectangle = *dirtyArea;
        // clip dirty area to render target
        if (dirtyAreaRectangle.GetLeft() < 0.0F) {
            dirtyAreaRectangle.SetWidth(dirtyAreaRectangle.GetWidth() + dirtyAreaRectangle.GetLeft());
            dirtyAreaRectangle.SetLeft(0.0F);
        }
        if (dirtyAreaRectangle.GetTop() < 0.0F) {
            dirtyAreaRectangle.SetHeight(dirtyAreaRectangle.GetHeight() + dirtyAreaRectangle.GetTop());
            dirtyAreaRectangle.SetTop(0.0F);
        }
        const Candera::Float right = dirtyAreaRectangle.GetLeft() + dirtyAreaRectangle.GetWidth();
        if (right > static_cast<Candera::Float>(renderTarget.GetWidth())) {
            dirtyAreaRectangle.SetWidth(dirtyAreaRectangle.GetWidth() - (right - static_cast<Candera::Float>(renderTarget.GetWidth())));
        }
        const Candera::Float bottom = dirtyAreaRectangle.GetTop() + dirtyAreaRectangle.GetHeight();
        if (bottom > static_cast<Candera::Float>(renderTarget.GetHeight())) {
            dirtyAreaRectangle.SetHeight(dirtyAreaRectangle.GetHeight() - (bottom - static_cast<Candera::Float>(renderTarget.GetHeight())));
        }

        // round dirty area rectangle to pixels
        Candera::Float flooredLeft = Candera::Math::Floor(dirtyAreaRectangle.GetLeft());
        dirtyAreaRectangle.SetWidth(Candera::Math::Ceil(dirtyAreaRectangle.GetWidth() + dirtyAreaRectangle.GetLeft() - flooredLeft));
        dirtyAreaRectangle.SetLeft(flooredLeft);
        Candera::Float flooredTop = Candera::Math::Floor(dirtyAreaRectangle.GetTop());
        dirtyAreaRectangle.SetHeight(Candera::Math::Ceil(dirtyAreaRectangle.GetHeight() + dirtyAreaRectangle.GetTop() - flooredTop));
        dirtyAreaRectangle.SetTop(flooredTop);
        dirtyArea = FeatStd::Optional<Candera::Rectangle>(dirtyAreaRectangle);
        return (dirtyAreaRectangle.GetWidth() > 0.0F) && (dirtyAreaRectangle.GetHeight() > 0.0F);
    }
    return true;
}

// ------------------------------------------------------------------------
#if defined(CANDERA_2D_ENABLED)

void Renderer::Queue2DJob(ViewScene* view, Candera::Camera2D* camera2D, UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea, bool clear, bool doOffscreenInvalidation)
{
    COURIER_LOG_DEBUG("Queue2DJob: view='%s', camera='%s'(%x), clear=%s, dirtyArea=%s {%f,%f,%f,%f}",
        (0 != view) ? view->GetId().CStr() : "",
        ((0 != camera2D) && (0 != camera2D->GetName())) ? camera2D->GetName() : "",
        camera2D,
        clear ? "true" : "false",
        (dirtyArea.IsSet()) ? "true" : "false",
        (*dirtyArea).GetLeft(),
        (*dirtyArea).GetTop(),
        (*dirtyArea).GetWidth(),
        (*dirtyArea).GetHeight());
    FeatStd::Optional<Candera::Rectangle> localDirtyArea = dirtyArea;

    doOffscreenInvalidation = doOffscreenInvalidation && mRenderConfiguration.ShallUseOffscreenInvalidation();

    FEATSTD_LINT_NEXT_EXPRESSION(777, "Deliberately testing floats for equality.")
    if ((dirtyArea.IsSet()) &&
        (((*dirtyArea).GetLeft() != (*dirtyArea).GetLeft()) ||
        ((*dirtyArea).GetTop() != (*dirtyArea).GetTop()) ||
        ((*dirtyArea).GetWidth() != (*dirtyArea).GetWidth()) ||
        ((*dirtyArea).GetHeight() != (*dirtyArea).GetHeight()))) {
        localDirtyArea = FeatStd::Optional<Candera::Rectangle>();
        COURIER_LOG_DEBUG("Queue3DJob: invalid dirty area detected. Using complete invalidation instead. view='%s', camera='%s'(%x), clear=%s, dirtyArea=%s {%f,%f,%f,%f}",
            (0 != view) ? view->GetId().CStr() : "",
            ((0 != camera2D) && (0 != camera2D->GetName())) ? camera2D->GetName() : "",
            camera2D,
            clear ? "true" : "false",
            (dirtyArea.IsSet()) ? "true" : "false",
            (*dirtyArea).GetLeft(),
            (*dirtyArea).GetTop(),
            (*dirtyArea).GetWidth(),
            (*dirtyArea).GetHeight());
    }
    if (mRenderConfiguration.ShallUseInvalidation()) {
        if (0 != camera2D) {
            if (mRenderConfiguration.ShallUseCourierSwapping() && (0 == camera2D->GetCameraRenderStrategy())) {
                camera2D->SetSwapEnabled(false);
            }
            RenderTarget* renderTarget = camera2D->GetRenderTarget();
            Gdu* usedGdu = (0 != renderTarget) ? GetGdu(camera2D->GetRenderTarget()) : 0;
            if (0 == usedGdu) {
                COURIER_LOG_ERROR("Queue2DJob failed, GDU not valid.");
            }
            else if (usedGdu->IsLoaded() && usedGdu->IsActive()) {
                if (ValidateAndAdjustDirtyArea(localDirtyArea, *renderTarget)) {
                    // invalidate all 
                    if (doOffscreenInvalidation) {
                        InvalidateOffscreenReferringViews(usedGdu, localDirtyArea);
                    }
                    usedGdu->Invalidate();
                    RenderJob newJob(view, camera2D, usedGdu, true, clear, GetRenderJobStrategy().GetEffectiveRenderCounter(view, *camera2D, *usedGdu, renderCounter));
                    if (clear) {
                        camera2D->SetRenderingEnabled(true);
                    }
                    Vector<FeatStd::Optional<Candera::Rectangle> >* dirtyAreas = 0;
                    bool rc = mRenderJobs.AddJob(newJob, dirtyAreas);
                    if (!rc) {
                        COURIER_LOG_ERROR("Failed to add 2D job");
                    }
                    if (rc && (0 != dirtyAreas)) {
                        GetRenderJobStrategy().AddDirtyArea(view, localDirtyArea, *dirtyAreas);
                    }
#if defined(CANDERA_3D_ENABLED)
                    Candera::Camera* compositionCamera = usedGdu->GetOwnerAccess().GetCompositionCamera();
                    if (0 != compositionCamera) {
                        FeatStd::Optional<Candera::Rectangle> transformedDirtyArea;
                        GetRenderJobStrategy().TransformDirtyAreaToCompositionCamera(view, *usedGdu, localDirtyArea, transformedDirtyArea);
                        COURIER_LOG_DEBUG("Queue2DJob: composition camera");
                        Queue3DJob(view, compositionCamera, renderCounter, transformedDirtyArea, false, false);
                    }
#endif
                    WakeUpRenderComponent(true);
                }
                else {
                    COURIER_LOG_DEBUG("Queue2DJob: dirty area outside of rendertarget.");
                }
            }
            else {
                COURIER_LOG_WARN("2D job not added. GDU not %s.", usedGdu->IsLoaded() ? "active" : "loaded");
            }
        }
    }
}

// ------------------------------------------------------------------------
void Renderer::Clear2D(const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if (mRenderConfiguration.ShallUseInvalidation()) {
        for (FeatStd::SizeType i = 0; i < mGDU.Size(); ++i) {
            Gdu* gdu = mGDU[i];
            if ((0 != gdu) && gdu->Is2D()) {
                Camera2D* camera = gdu->GetClearCamera2D();
                if (0 != camera) {
                    Queue2DJob(0, camera, cCOURIER_DEFAULT_RENDER_COUNT, dirtyArea, true, true);
                }
            }
        }
    }
}

#endif

// ------------------------------------------------------------------------
#if defined(CANDERA_3D_ENABLED)

void Renderer::Queue3DJob(ViewScene* view, Candera::Camera* camera3D, UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea, bool clear, bool doOffscreenInvalidation)
{
    COURIER_LOG_DEBUG("Queue3DJob: view='%s', camera='%s'(%x), clear=%s, dirtyArea=%s {%f,%f,%f,%f}",
        (0 != view) ? view->GetId().CStr() : "",
        ((0 != camera3D) && (0 != camera3D->GetName())) ? camera3D->GetName() : "",
        camera3D,
        clear ? "true" : "false",
        (dirtyArea.IsSet()) ? "true" : "false",
        (*dirtyArea).GetLeft(),
        (*dirtyArea).GetTop(),
        (*dirtyArea).GetWidth(),
        (*dirtyArea).GetHeight());
    FeatStd::Optional<Candera::Rectangle> localDirtyArea = dirtyArea;

    doOffscreenInvalidation = doOffscreenInvalidation && mRenderConfiguration.ShallUseOffscreenInvalidation();

    FEATSTD_LINT_NEXT_EXPRESSION(777, "Deliberately testing floats for equality.")
    if ((dirtyArea.IsSet()) &&
        (((*dirtyArea).GetLeft() != (*dirtyArea).GetLeft()) ||
        ((*dirtyArea).GetTop() != (*dirtyArea).GetTop()) ||
        ((*dirtyArea).GetWidth() != (*dirtyArea).GetWidth()) ||
        ((*dirtyArea).GetHeight() != (*dirtyArea).GetHeight()))) {
        localDirtyArea = FeatStd::Optional<Candera::Rectangle>();
        COURIER_LOG_DEBUG("Queue3DJob: invalid dirty area detected. Using complete invalidation instead. view='%s', camera='%s'(%x), clear=%s, dirtyArea=%s {%f,%f,%f,%f}",
            (0 != view) ? view->GetId().CStr() : "",
            ((0 != camera3D) && (0 != camera3D->GetName())) ? camera3D->GetName() : "",
            camera3D,
            clear ? "true" : "false",
            (dirtyArea.IsSet()) ? "true" : "false",
            (*dirtyArea).GetLeft(),
            (*dirtyArea).GetTop(),
            (*dirtyArea).GetWidth(),
            (*dirtyArea).GetHeight());
    }
    if (mRenderConfiguration.ShallUseInvalidation()) {
        if (0 != camera3D) {
            if (mRenderConfiguration.ShallUseCourierSwapping() && (0 == camera3D->GetCameraRenderStrategy())) {
                camera3D->SetSwapEnabled(false);
            }
            RenderTarget* renderTarget = camera3D->GetRenderTarget();
            Gdu* usedGdu = (0 != renderTarget) ? GetGdu(camera3D->GetRenderTarget()) : 0;
            if (0 == usedGdu) {
                COURIER_LOG_ERROR("Queue3DJob failed, GDU not valid.");
            }
            else if (usedGdu->IsLoaded() && usedGdu->IsActive()) {
                if (ValidateAndAdjustDirtyArea(localDirtyArea, *renderTarget)) {
                    if (doOffscreenInvalidation) {
                        InvalidateOffscreenReferringViews(usedGdu, localDirtyArea);
                    }
                    usedGdu->Invalidate();
                    RenderJob newJob(view, camera3D, usedGdu, false, clear, GetRenderJobStrategy().GetEffectiveRenderCounter(view, *camera3D, *usedGdu, renderCounter));
                    if (clear) {
                        camera3D->SetRenderingEnabled(true);
                    }
                    Vector<FeatStd::Optional<Candera::Rectangle> >* dirtyAreas = 0;
                    bool rc = mRenderJobs.AddJob(newJob, dirtyAreas);
                    if (!rc) {
                        COURIER_LOG_ERROR("Failed to add 3D job");
                    }
                    if (rc && (0 != dirtyAreas)) {
                        GetRenderJobStrategy().AddDirtyArea(view, localDirtyArea, *dirtyAreas);
                    }
                    Candera::Camera* compositionCamera = usedGdu->GetOwnerAccess().GetCompositionCamera();
                    if (0 != compositionCamera) {
                        FeatStd::Optional<Candera::Rectangle> transformedDirtyArea;
                        GetRenderJobStrategy().TransformDirtyAreaToCompositionCamera(view, *usedGdu, localDirtyArea, transformedDirtyArea);
                        COURIER_LOG_DEBUG("Queue3DJob: composition camera");
                        Queue3DJob(view, compositionCamera, renderCounter, transformedDirtyArea, false, false);
                    }
                    WakeUpRenderComponent(false);
                }
                else {
                    COURIER_LOG_DEBUG("Queue3DJob: dirty area outside of rendertarget.");
                }
            }
            else {
                COURIER_LOG_WARN("3D job not added. GDU not %s.", usedGdu->IsLoaded() ? "active" : "loaded");
            }
        }
    }
}

// ------------------------------------------------------------------------
void Renderer::Clear3D(const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if (mRenderConfiguration.ShallUseInvalidation()) {
        for (FeatStd::SizeType i = 0; i < mGDU.Size(); ++i) {
            Gdu* gdu = mGDU[i];
            if ((0 != gdu) && gdu->Is3D()) {
                Camera* camera = gdu->GetClearCamera3D();
                if (0 != camera) {
                    Queue3DJob(0, camera, cCOURIER_DEFAULT_RENDER_COUNT, dirtyArea, true, true);
                }
            }
        }
    }
}

#endif

// ------------------------------------------------------------------------
void Renderer::DeleteRenderJobs(const RenderTarget* renderTarget)
{
    if (0 != renderTarget) {
        mRenderJobs.DeleteJobs(renderTarget);
    }
}

// ------------------------------------------------------------------------
void Renderer::DeleteRenderJobs(const Gdu* gdu)
{
    if (0 != gdu) {
#if defined(CANDERA_2D_ENABLED)
        DeleteRenderJobs(gdu->GetRenderTarget2D());
#endif

#if defined(CANDERA_3D_ENABLED)
        DeleteRenderJobs(gdu->GetRenderTarget3D());
#endif
    }
}

// ------------------------------------------------------------------------
void Renderer::DeleteRenderJob(const Candera::CanderaObject* camera)
{
    mRenderJobs.DeleteJobs(camera);
}

// ------------------------------------------------------------------------
void Renderer::AcquireOffScreenGdu(Gdu* gdu)
{
    for (FeatStd::SizeType i = 0; i < mOffscreenGduVector.Size(); ++i) {
        OffScreenGdu* offScreenGdu = &mOffscreenGduVector[i];
        if (offScreenGdu->mOffScreenGdu == gdu) {
            // it might exist in list but already unloaded, so load it again
            if (0 == offScreenGdu->mRefCount) {
                if (gdu->Upload(this)) {
                    COURIER_LOG_INFO("Offscreen Target successful uploaded");
                }
                else {
                    COURIER_LOG_ERROR("Offscreen Target could not be uploaded");
                }
            }
            ++(offScreenGdu->mRefCount);
            gdu->Activate();
            return;
        }
    }
    // add it to list with refcount 1 and upload it
    (void)mOffscreenGduVector.Add(OffScreenGdu(gdu));
    if (gdu->Upload(this)) {
        COURIER_LOG_INFO("Offscreen Target successful uploaded");
    }
    else {
        COURIER_LOG_ERROR("Offscreen Target could not be uploaded");
    }
}

// ------------------------------------------------------------------------
void Renderer::ReleaseOffScreenGdu(const Gdu* gdu)
{
    for (FeatStd::SizeType i = 0; i < mOffscreenGduVector.Size(); ++i) {
        OffScreenGdu* offScreenGdu = &mOffscreenGduVector[i];
        if (offScreenGdu->mOffScreenGdu == gdu) {
            --(offScreenGdu->mRefCount);
            if (0 == offScreenGdu->mRefCount) {
                DeleteRenderJobs(offScreenGdu->mOffScreenGdu);
                offScreenGdu->mOffScreenGdu->Unload(this);
                COURIER_LOG_INFO("Offscreen Target unloaded");
            }
            return;
        }
    }
    COURIER_LOG_ERROR("Could not find Offscreen Target inside list");
}

// ------------------------------------------------------------------------
void Renderer::EnableGdu(bool enable, Gdu* lookupGdu, bool forceEnable)
{
    if (0 == lookupGdu) {
        return;
    }
    if (!lookupGdu->IsOffscreen()) {
        FEATSTD_LINT_CURRENT_SCOPE(550, "rc variable is used inside the macros")
        bool rc = false;
        Internal::unused(rc);
        if (enable) {
            rc = Courier::Internal::RendererHelper::PlatformUpload(lookupGdu, this, forceEnable);
            COURIER_LOG_DEBUG("Upload for Gdu returned %s", rc ? "true" : "false (this can also mean that the gdu is already uploaded)");
        }
        else {
#ifdef CANDERA_FONTENGINE_FREETYPE
            Candera::TextRendering::FontEngine::FlushCache();
#endif
            rc = Courier::Internal::RendererHelper::PlatformUnload(lookupGdu, this);
            COURIER_LOG_DEBUG("Unload for Gdu returned %s", rc ? "true" : "false (this can also mean that the ref count for gdu is > 0)");
        }
#if defined(CANDERA_GPU_SIMULATION)
        if (rc) {
            if (mRenderConfiguration.ShallUseKickDisplayUpdate()) {
                Int displayId = lookupGdu->GetGdu()->GetDisplay();
                if (0 != DevicePackageInterface::GetDisplay(displayId)) {
                    (void)DevicePackageInterface::GetDisplay(displayId)->ToSimulatedDisplay()->KickDisplayUpdate();
                }
            }
        }
#endif
    }
    else {
        if (enable) {
            AcquireOffScreenGdu(lookupGdu);
        }
        else {
            ReleaseOffScreenGdu(lookupGdu);
        }
    }
}

// ------------------------------------------------------------------------
void Renderer::EnableLayer(bool enable, Candera::RenderTarget * renderTarget, bool forceEnable)
{
    if (0 == renderTarget) {
        return;
    }
    Gdu * lookupGdu = GetGdu(renderTarget);
    if (lookupGdu==0) {
        return;
    }
    if (!enable) {
        EnableGdu(enable, lookupGdu, forceEnable);
    }
    Candera::Internal::GraphicDeviceUnitOwnerAccess gduOwnerAccess(lookupGdu->GetGdu());
    if (Candera::Internal::GraphicDeviceUnitOwnerAccess::GraphicDeviceUnitOwner == gduOwnerAccess.GetOwnerType()) {
        Candera::GraphicDeviceUnit* ownerGdu = gduOwnerAccess.GetGraphicDeviceUnitOwner();
        if (0 != ownerGdu) {
            EnableLayer(enable, ownerGdu->ToRenderTarget2D(), forceEnable);
            EnableLayer(enable, ownerGdu->ToRenderTarget3D(), forceEnable);
        }
    }
    if (enable) {
        EnableGdu(enable, lookupGdu, forceEnable);
    }
}

// ------------------------------------------------------------------------
void Renderer::Invalidate(Candera::RenderTarget* renderTarget)
{
    FEATSTD_DEBUG_ASSERT(0 != renderTarget);
    for (SizeType i = 0; i < mGDU.Size(); ++i) {
        Gdu* gdu = mGDU[i];
        if ((0 != gdu) && (renderTarget == gdu->GetRenderTarget())) {
            gdu->Invalidate();
        }
    }
}

// ------------------------------------------------------------------------
void Renderer::Invalidate(RTPtrVector& renderTargetVector)
{
    for (SizeType i = 0; i < renderTargetVector.Size(); ++i) {
        Invalidate(renderTargetVector[i]);
    }
}

// ------------------------------------------------------------------------
void Renderer::InvalidateAll(bool invalidate)
{
    for (SizeType i = 0; i < mGDU.Size(); ++i) {
        Gdu* gdu = mGDU[i];
        if (0 != gdu) {
            gdu->Invalidate(invalidate);
        }
    }
}

// ------------------------------------------------------------------------
bool Renderer::OnInitialize()
{
    return true;
}

// ------------------------------------------------------------------------
void Renderer::OnFinalize()
{
}

// ------------------------------------------------------------------------
FEATSTD_LINT_CURRENT_SCOPE(1762, "No we don't want to make it const because customer might overwrite and maintain content")
bool Renderer::OnLayerActivationChanged(Int layerId, bool activate)
{
    FEATSTD_UNUSED2(layerId, activate);
    COURIER_LOG_DEBUG("OnLayerActivationChanged called for Layer(%d) Activate(%d)", layerId, static_cast<Int>(activate));
    return true;
}

// ------------------------------------------------------------------------
void Renderer::ComputeFramesPerSecond(RenderHint* hint)
{
#if defined(CANDERA_3D_ENABLED)
    if (hint->ShallRender3D()) {
        ComputeFramesPerSecond(mMeasure3D);
    }
#endif

#if defined(CANDERA_2D_ENABLED)
    if (hint->ShallRender2D()) {
        ComputeFramesPerSecond(mMeasure2D);
    }
#endif
}

// ------------------------------------------------------------------------
void Renderer::ComputeFramesPerSecond(Measure& measure)
{
    const FeatStd::SizeType timeStamp = Candera::TimePlatform::GetMilliSeconds();
    measure.mLastFrameDuration = timeStamp - measure.mPreviousFrameTimeStamp;
    measure.mFramesTimespan += measure.mLastFrameDuration;
    ++measure.mFrameCount;

    if ((measure.mFramesTimespan > 500) && (measure.mFrameCount > 0) && (measure.mFramesTimespan > 0)) {
        measure.mFramesPerSecond = static_cast<UInt16>((1000L * static_cast<FeatStd::SizeType>(measure.mFrameCount)) / measure.mFramesTimespan);
        measure.mFramesTimespan = 0;
        measure.mFrameCount = 0;
    }

    measure.mPreviousFrameTimeStamp = timeStamp;
}

void Renderer::SetRenderJobStrategy(RenderJobStrategy* renderJobStrategy)
{
    mRenderJobStrategy = renderJobStrategy;
}

// ------------------------------------------------------------------------
RenderJobStrategy& Renderer::GetRenderJobStrategy()
{
    if (0 != mRenderJobStrategy) {
        return *mRenderJobStrategy;
    }
    FEATSTD_UNSYNCED_STATIC_OBJECT(RenderJobStrategy, sRenderJobStrategy, (false));
    return sRenderJobStrategy;
}

// ------------------------------------------------------------------------
FEATSTD_RTTI_BASECLASS_DEFINITION(RenderJobStrategy)

// ------------------------------------------------------------------------
bool RenderJobStrategy::operator()(const RenderJob& a, const RenderJob& b) const
{
    // Sorting criteria are evaluated using equivalent of (a.isX() && !b.isX()) || ( (a.isX() == b.isX()) && ( next criteria ) )
    const RenderConfiguration& renderConfiguration = Renderer::GetRenderConfiguration();
    const bool c_aIsOffscreen = const_cast<RenderJob&>(a).GetGdu()->IsOffscreen();
    const bool c_bIsOffscreen = const_cast<RenderJob&>(b).GetGdu()->IsOffscreen();
    return (c_aIsOffscreen && (!c_bIsOffscreen)) || ((c_aIsOffscreen == c_bIsOffscreen) && (
            (renderConfiguration.ShallUse2DBefore3D() && a.Is2D() && (!b.Is2D())) ||
            (((!renderConfiguration.ShallUse2DBefore3D()) || (a.Is2D() == b.Is2D())) && (
                (a.IsClear() && (!b.IsClear())) ||
                ((a.IsClear() == b.IsClear()) &&
                    (renderConfiguration.ShallUseGlobalCameraSequenceNumber() && (a.GetSequenceNumber() < b.GetSequenceNumber()))
                )
            ))
        ));
}

}
