//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "RenderJob.h"
#include "Renderer.h"
#include "RenderHint.h"
#include <Courier/Diagnostics/Log.h>
#include <Courier/Visualization/ViewScene.h>

namespace Courier {
    COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

// ------------------------------------------------------------------------
RenderJob & RenderJob::operator = (const RenderJob& renderJob)
{
    if (this != &renderJob) {
        mDirtyAreas = renderJob.mDirtyAreas;
        FEATSTD_LINT_NEXT_EXPRESSION(1555, "Direct pointer copy of member, thats what we want")
        mView = renderJob.mView;
        FEATSTD_LINT_NEXT_EXPRESSION(1555, "Direct pointer copy of member, thats what we want")
        mCamera = renderJob.mCamera;
        FEATSTD_LINT_NEXT_EXPRESSION(1555, "Direct pointer copy of member, thats what we want")
        mGdu = renderJob.mGdu;
        mFrameId = renderJob.mFrameId;
        mCurrentDirtyArea = renderJob.mCurrentDirtyArea;
        mRenderState = renderJob.mRenderState;
        mRenderCounter = renderJob.mRenderCounter;
        mIs2D = renderJob.mIs2D;
        mIsClear = renderJob.mIsClear;
    }
    return *this;
}

// ------------------------------------------------------------------------
bool RenderJob::DecRenderCounter()
{
    if (mDirtyAreas.Size() > 0) {
        mCurrentDirtyArea = (mCurrentDirtyArea + 1) % mDirtyAreas.Size();
        mDirtyAreas[mCurrentDirtyArea].Clear();
    }
    if (mRenderCounter > 0) {
        --mRenderCounter;
        return true;
    }
    return false;
}

// ------------------------------------------------------------------------
void RenderJob::Validate() const
{
#if defined (CANDERA_LAYOUT_ENABLED)
    if (Renderer::GetRenderConfiguration().ShallUseValidatingLayout()) {
        Candera::AbstractNodePointer scene = mCamera.GetScene();
#if defined(CANDERA_2D_ENABLED)
        Candera::Scene2D* scene2D = Candera::Dynamic_Cast<Candera::Scene2D*>(scene.ToCanderaObject());
        if (0 != scene2D) {
            scene2D->ValidateLayout();
            return;
        }
#endif
#if defined(CANDERA_3D_ENABLED)
        Candera::Scene* scene3D = Candera::Dynamic_Cast<Candera::Scene*>(scene.ToCanderaObject());
        if (0 != scene3D) {
            scene3D->ValidateLayout();

        }
#endif
    }
#endif
}

// ------------------------------------------------------------------------
RenderJob::RenderState RenderJob::Render(const Vector<FeatStd::Optional<Candera::Rectangle> >& dirtyAreas)
{
    RenderJob::RenderState result = RenderPassFailed;
    if (0 != mGdu) {
        // mark gdu for swapping
        mGdu->Invalidate();
    }
    if (mCamera.IsValid()) {
#if defined(CANDERA_2D_ENABLED)
        if ((Is2D())) {
            Candera::Camera2D* camera = Candera::Dynamic_Cast<Candera::Camera2D*>(mCamera.ToCanderaObject());
            if (dirtyAreas.Size() > 0) {
                for (FeatStd::SizeType i = 0; i < dirtyAreas.Size(); ++i) {
                    const FeatStd::Optional<Candera::Rectangle>& dirtyArea = dirtyAreas[i];
                    COURIER_LOG_DEBUG("RenderCamera2D: view='%s', camera='%s'(%x), clear=%s, dirtyArea=%s {%f,%f,%f,%f}",
                        (0 != mView) ? mView->GetId().CStr() : "",
                        (0 != camera->GetName()) ? camera->GetName() : "", 
                        camera,
                        (mIsClear == true) ? "true" : "false",
                        (dirtyArea.IsSet()) ? "true" : "false",
                        (*dirtyArea).GetLeft(),
                        (*dirtyArea).GetTop(),
                        (*dirtyArea).GetWidth(),
                        (*dirtyArea).GetHeight());
                    if (dirtyArea.IsSet()) {
                        Candera::Renderer2D::RenderCamera(camera, /* resetStatistics */ false, *dirtyArea);
                    }
                    else {
                        Candera::Renderer2D::RenderCamera(camera);
                    }
                }
            }
            else {
                COURIER_LOG_DEBUG("RenderCamera2D: view='%s', camera='%s'(%x), clear=%s",
                    (0 != mView) ? mView->GetId().CStr() : "",
                    (0 != camera->GetName()) ? camera->GetName() : "",
                    camera,
                    (mIsClear == true) ? "true" : "false");
                Candera::Renderer2D::RenderCamera(camera);
            }
            result = ((0 == camera->GetCameraRenderStrategy()) || camera->GetCameraRenderStrategy()->IsRenderPassCompleted()) ? RenderPassCompleted : RenderPassIncomplete;
        }
#endif
#if defined(CANDERA_3D_ENABLED)
        if (!Is2D()) {
            Candera::Camera* camera = Candera::Dynamic_Cast<Candera::Camera*>(mCamera.ToCanderaObject());
            Candera::RenderTarget* renderTarget = camera->GetRenderTarget();
            bool renderPassFailed = false;
            if ((dirtyAreas.Size() > 0) && (0 != renderTarget)) {
                for (FeatStd::SizeType i = 0; i < dirtyAreas.Size(); ++i) {
                    const FeatStd::Optional<Candera::Rectangle>& dirtyArea = dirtyAreas[i];
                    COURIER_LOG_DEBUG("RenderCamera: view='%s', camera='%s'(%x), clear=%s, dirtyArea=%s {%f,%f,%f,%f}",
                        (0 != mView) ? mView->GetId().CStr() : "",
                        (0 != camera->GetName()) ? camera->GetName() : "",
                        camera,
                        (mIsClear == true) ? "true" : "false",
                        (dirtyArea.IsSet()) ? "true" : "false",
                        (*dirtyArea).GetLeft(),
                        (*dirtyArea).GetTop(),
                        (*dirtyArea).GetWidth(),
                        (*dirtyArea).GetHeight());
                    if (dirtyArea.IsSet()) {
                        Candera::Rectangle normalizedDirtyArea;
                        Candera::Math3D::NormalizeRectangle(*renderTarget, *dirtyArea, normalizedDirtyArea);
                        if (!Candera::Renderer::RenderCamera(camera, /* resetStatistics */ false, normalizedDirtyArea)) {
                            renderPassFailed = true;
                        }
                    }
                    else {
                        if (!Candera::Renderer::RenderCamera(camera)) {
                            renderPassFailed = true;
                        }
                    }
                }
            }
            else {
                COURIER_LOG_DEBUG("RenderCamera: view='%s', camera='%s'(%x), clear=%s",
                    (0 != mView) ? mView->GetId().CStr() : "",
                    (0 != camera->GetName()) ? camera->GetName() : "",
                    camera,
                    (mIsClear == true) ? "true" : "false");
                if (!Candera::Renderer::RenderCamera(camera)) {
                    renderPassFailed = true;
                }
            }
            if (renderPassFailed) {
                result = RenderPassFailed;
            }
            else {
                result = ((0 == camera->GetCameraRenderStrategy()) || camera->GetCameraRenderStrategy()->IsRenderPassCompleted()) ? RenderPassCompleted : RenderPassIncomplete;
            }
        }
#endif
    }
    mRenderState = result;
    return result;
}

// ------------------------------------------------------------------------
const FeatStd::Char* RenderJob::GetSceneName() const
{
    return mCamera.GetScene().GetName();
}

// ------------------------------------------------------------------------
const FeatStd::Char* RenderJob::GetCameraName() const
{
    return mCamera.GetName();
}

// ------------------------------------------------------------------------
void RenderJob::DisableIfClearCamera() const
{
    if (mIsClear == true) {
        mCamera.SetRenderingEnabled(false);
    }
}

// ------------------------------------------------------------------------
Int32 RenderJob::GetSequenceNumber() const
{
#if defined(CANDERA_2D_ENABLED)
    Candera::Camera2D* camera2D = Candera::Dynamic_Cast<Candera::Camera2D*>(mCamera.ToCanderaObject());
    if (0 != camera2D) {
        return camera2D->GetSequenceNumber();
    }
#endif
#if defined(CANDERA_3D_ENABLED)
    Candera::Camera* camera = Candera::Dynamic_Cast<Candera::Camera*>(mCamera.ToCanderaObject());
    if (0 != camera) {
        return camera->GetSequenceNumber();
    }
#endif
    return 0;
}

// ------------------------------------------------------------------------
// RenderJobs methods
// ------------------------------------------------------------------------
bool RenderJobs::AddJob(const RenderJob & newjob, Vector<FeatStd::Optional<Candera::Rectangle> >*& dirtyAreas)
{
    // Keep the order of the jobs in the order they are added.
    // This means we do not replace a job with the same camera, we append it instead.
    // This is the default use case, cameras get rendered in the order they get added. 
    // (RenderConfiguration UseGlobalCameraSequenceNumber(true) sorts them before rendering)

    FEATSTD_DEBUG_ASSERT(0 != mActiveVector);
    if (0 == mActiveVector) {
        return false;
    }
    FeatStd::SizeType size = mActiveVector->Size();
    if (size > 0) {
        RenderJob& job = (*mActiveVector)[size - 1];
        // check if the last job has the same camera, if so then replace it
        if (newjob.GetCamera() == job.GetCamera()) {
            job.mRenderCounter = newjob.mRenderCounter;
            dirtyAreas = &job.mDirtyAreas[job.mCurrentDirtyArea];
            return true;
        }
        // search from last position to the first occurence of the camera
        // in case we found it remove it, this will cause packing the vector
        // but ensures that vector is not growing when invalidating too often
        // by the application.
        for (FeatStd::OffsetType r = static_cast<FeatStd::OffsetType>(size-1); r >= 0; --r) {
            RenderJob& currentJob = (*mActiveVector)[r];
            if (newjob.GetCamera() == currentJob.GetCamera()) {
                currentJob.mRenderCounter = newjob.mRenderCounter;
                if (!Renderer::GetRenderConfiguration().ShallUseGlobalCameraSequenceNumber()) {
                    RenderJob tempJob = currentJob;
                    (void)mActiveVector->Remove(r);
                    // adding to the end of the list is required for Courier configurations where no sorting of the render jobs is performed
                    if (mActiveVector->Add(tempJob)) {
                        RenderJob& jobRef = (*mActiveVector)[mActiveVector->Size() - 1];
                        dirtyAreas = &jobRef.mDirtyAreas[jobRef.mCurrentDirtyArea];
                        return true;
                    }
                    return false;
                }
                dirtyAreas = &currentJob.mDirtyAreas[currentJob.mCurrentDirtyArea];
                return true;
            }
        }
    }
    // add the new one to the end.
    if (mActiveVector->Add(newjob)) {
        RenderJob& jobRef = (*mActiveVector)[mActiveVector->Size() - 1];
        dirtyAreas = &jobRef.mDirtyAreas[jobRef.mCurrentDirtyArea];
        return true;
    }
    return false;
}

// ------------------------------------------------------------------------
bool RenderJobs::Render(RenderHint * renderHint, Renderer& renderer)
{
    FEATSTD_DEBUG_ASSERT(0 != mActiveVector);
    FEATSTD_DEBUG_ASSERT(0 != renderHint);
    if (0 == mActiveVector) {
        return false;
    }
    if (0 == renderHint) {
        return false;
    }

    for (FeatStd::SizeType r = 0; r < mActiveVector->Size(); ++r) {
        RenderJob & job = (*mActiveVector)[r];
        Gdu * gdu = job.GetGdu();
        if ((job.GetRenderCounter() > 0) && job.GetCamera().IsValid() && (0 != gdu) && gdu->IsLoaded()) {
            job.Validate();
        }
    }

    RenderJobStrategy& renderJobStrategy = renderer.GetRenderJobStrategy();
    renderJobStrategy.OnPreRenderPass(*mActiveVector);
    if (Renderer::GetRenderConfiguration().ShallUseGlobalCameraSequenceNumber()) {
        if (mActiveVector->Size() > 1) {
            (*mActiveVector).Sort(renderJobStrategy);
        }
    }
    bool hasEmptyEntry = false;

    for (FeatStd::SizeType r = 0; r < mActiveVector->Size(); ++r) {
        RenderJob& job = (*mActiveVector)[r];
        if (((renderHint->ShallRender2D() && job.Is2D()) || (renderHint->ShallRender3D() && (!job.Is2D()))) &&
                (job.GetRenderCounter() > 0) &&
                job.GetCamera().IsValid() &&
                (0 != job.GetGdu()) &&
                (job.GetRenderCounter() > 0) &&
                job.GetGdu()->IsLoaded()) {
            Gdu* gdu = job.GetGdu();
            if (Gdu::RenderTargetValid != gdu->GetRenderState()) {
                // mark gdu for swapping
                gdu->SetRenderState(Gdu::RenderTargetCompleted);
            }
            gdu->SetLastRenderJob(&job);
        }
    }

    for (FeatStd::SizeType r = 0; r < mActiveVector->Size(); ++r) {
        RenderJob& job = (*mActiveVector)[r];
        if (((renderHint->ShallRender2D() && job.Is2D()) || (renderHint->ShallRender3D() && (!job.Is2D()))) &&
                (job.GetRenderCounter() > 0) &&
                job.GetCamera().IsValid() &&
                (0 != job.GetGdu()) &&
                (job.GetRenderCounter() > 0) &&
                job.GetGdu()->IsLoaded()) {
            Gdu* gdu = job.GetGdu();
            RenderJob::RenderState renderState = job.GetRenderState();
            if (renderJobStrategy.GetRenderJobAction(job) == RenderJobStrategy::ProceedRenderJob) {
                renderJobStrategy.OnPreRenderJob(job);
                renderJobStrategy.ProcessDirtyAreas(job, job.mDirtyAreas, mDirtyAreas);
                renderState = job.Render(mDirtyAreas);
                if (RenderJob::RenderPassCompleted == renderState) {
                    (void)job.DecRenderCounter();
                }
                renderJobStrategy.OnPostRenderJob(job);
            }
            gdu->Invalidate();
            if (Gdu::RenderTargetValid == gdu->GetRenderState()) {
                // mark gdu for swapping.
                gdu->SetRenderState(Gdu::RenderTargetCompleted);
            }
            if (RenderJob::RenderPassIncomplete == renderState) {
                // mark gdu as incomplete. swapping has to occur in another frame.
                gdu->SetRenderState(Gdu::RenderTargetIncomplete);
            }
            if (gdu->GetLastRenderJob() == &job) {
                if ((Gdu::RenderTargetCompleted == gdu->GetRenderState()) && renderJobStrategy.OnSwapBuffer(*gdu)) {
                    renderer.SwapBuffer(gdu, renderHint);
                }
            }
        }
        else {
            while (job.GetRenderCounter() > 0) {
                (void)job.DecRenderCounter();
            }
        }
        if (renderJobStrategy.OnRenderJobFinished(job)) {
            job.DisableIfClearCamera();
            job.Clear();
            hasEmptyEntry = true;
        }
    }
    if (hasEmptyEntry) {
        SwapVectors();
    }
    if (mActiveVector->Size() > 0) {
        renderer.mContinueExecution = true;
    }
    renderJobStrategy.OnPostRenderPass(*mActiveVector);
    return true;
}

// ------------------------------------------------------------------------
void RenderJobs::DeleteJobs(const Candera::RenderTarget * renderTarget)
{
    FEATSTD_DEBUG_ASSERT(0 != mActiveVector);
    if (0 != mActiveVector) {
        bool hasEmptyEntry = false;
        for (FeatStd::SizeType r = 0; r < mActiveVector->Size(); ++r) {
            RenderJob& job = (*mActiveVector)[r];
            Gdu* gdu = job.GetGdu();
            if ((0 != gdu) && (gdu->GetRenderTarget() == renderTarget)) {
                job.Clear();
                hasEmptyEntry = true;
            }
        }
        if (hasEmptyEntry) {
            SwapVectors();
        }
    }
}

// ------------------------------------------------------------------------
void RenderJobs::DeleteJobs(const Candera::CanderaObject* camera)
{
    FEATSTD_DEBUG_ASSERT(0 != mActiveVector);
    if (0 != mActiveVector) {
        bool hasEmptyEntry = false;
        for (FeatStd::SizeType r = 0; r < mActiveVector->Size(); ++r) {
            RenderJob& job = (*mActiveVector)[r];
            if (job.GetCamera().ToCanderaObject() == camera) {
                job.Clear();
                hasEmptyEntry = true;
            }
        }
        if (hasEmptyEntry) {
            SwapVectors();
        }
    }
}

// ------------------------------------------------------------------------
void RenderJobs::RemoveViewRelations(const View* view)
{
    FEATSTD_DEBUG_ASSERT(0 != mActiveVector);
    if (0 != mActiveVector) {
        for (FeatStd::SizeType r = 0; r < mActiveVector->Size(); ++r) {
            RenderJob& job = (*mActiveVector)[r];
            if (job.GetView() == view) {
                job.SetView(0);
            }
        }
    }
}

// ------------------------------------------------------------------------
void RenderJobs::SwapVectors()
{
    FEATSTD_DEBUG_ASSERT(0 != mActiveVector);
    FEATSTD_DEBUG_ASSERT(0 != mInactiveVector);

    if ((0 != mActiveVector) && (0 != mInactiveVector)) {
        mInactiveVector->Clear();
        for (FeatStd::SizeType r = 0; r < mActiveVector->Size(); ++r) {
            RenderJob& job = (*mActiveVector)[r];
            if (job.GetRenderCounter() > 0) {
                (void)mInactiveVector->Add(job);
            }
        }
        mActiveVector->Clear();

        RenderJobVector* oldVector = mActiveVector;
        mActiveVector = mInactiveVector;
        mInactiveVector = oldVector;
    }
}

// ------------------------------------------------------------------------
bool RenderJobs::HasJob(bool is2D) const
{
    FEATSTD_DEBUG_ASSERT(0 != mActiveVector);
    if (0 != mActiveVector) {
        for (FeatStd::SizeType i = 0; i < mActiveVector->Size(); ++i) {
            RenderJob& job = (*mActiveVector)[i];
            if (job.Is2D() == is2D) {
                return true;
            }
        }
    }
    return false;
}

}
