//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_Gdu_h)
#define Courier_Visualization_Gdu_h

#include <Candera/Candera.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget.h>
#include <CanderaPlatform/Device/Common/Base/ImageSource2D.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <Courier/Util/Util.h>

#if defined(CANDERA_2D_ENABLED)
#include <Courier/Visualization/ClearingData2D.h>
#endif


#if defined(CANDERA_3D_ENABLED)
#include <Candera/Engine3D/Core/NodeListener.h>
#include <Courier/Visualization/ClearingData3D.h>
#endif

namespace Courier {
    namespace UnitTest {
        class GduWrapper;
    }
}
namespace Courier { 
    class Renderer;
    class RenderJob;
    
/// @addtogroup COURIER_VISUALIZATION
/// @{
/** Gdu is a wrapper class which represents a GraphicDeviceUnit.
    A Gdu vector is maintained by the Renderer object and is filled with Gdu objects during the initialization phase.
    Each Gdu object is used for controlling the rendering process. It means it contains several flags and states, which
    are set by the Renderer object.
    The invalidation flag is set to true if a camera, which refers to a certain Candera::RenderTarget, is rendered.
*/
#if defined(CANDERA_3D_ENABLED)
class Gdu : public Candera::NodeListener
#else
class Gdu
#endif
{
    public:
        /**
         * The RenderState defines the current state of the GDU. It is updated and used by the Courier Renderer.
         */
        typedef enum
        {
            /**
             * If the Gdu is in the state RenderTargetValid then nothing has changed. No RenderJob has been rendered since the last swap.
             * Therefore, no swap is required for this Gdu.
             */
            RenderTargetValid,
            /**
             * If the Gdu is in the state RenderTargetIncomplete then some RenderJobs have rendered onto it completely or
             * partially but not all render jobs that have to render to it are yet finished. Therefore, a swap on this Gdu is not allowed.
             */
            RenderTargetIncomplete,
            /**
            * If the Gdu is in the state RenderTargetCompleted then all RenderJobs have rendered onto it completely. Therefore, a swap on this Gdu is required.
            */
            RenderTargetCompleted
        } RenderState;

#if defined(CANDERA_3D_ENABLED)

        /** Possible states of the owner of the GraphicDeviceUnit (only available if 3D is enabled). */
        enum OwnerState {
            OwnerInvalid,
            NoOwner,
            GraphicDeviceUnitOwner,
            ContextResourcePoolOwner
        };
#endif

        /** Default constructor.
            @param layerIndex */
        Gdu(FeatStd::SizeType layerIndex);

        /** Constructor of the wrapper object.
            @param gdu the Candera::GraphicDeviceUnit read out from the asset.
            @param layerIndex */
        Gdu(Candera::GraphicDeviceUnit * gdu, FeatStd::SizeType layerIndex);

        /// Frees Gdu resources.
        void Finalize();

        /** Invalidates the Gdu object, which means it shall be rendered during the render process.
            @param invalidate the invalidation flag which shall be set, usually default value true is used.*/
        void Invalidate(bool invalidate = true) { mInvalidate = invalidate; }

        /** Returns if the Gdu object is invalidated.
            @return <em>true</em> if the Gdu object is invalidated,
                    <em>false</em> otherwise. */
        bool IsInvalidated() const { return mInvalidate; }

        /** Returns if the Gdu object contains a 2D RenderTarget.
            @return <em>true</em> if the RenderTarget is 2D,
                    <em>false</em> otherwise. */
        bool Is2D() const { return m2DRT!=0; }

        /** Returns if the Gdu object contains a 3D RenderTarget.
            @return <em>true</em> if the RenderTarget is 3D,
                    <em>false</em> otherwise. */
        bool Is3D() const { return m3DRT!=0; }

        /** Returns the Candera::GraphicDeviceUnit.
            @return <em>GraphicDeviceUnit</em> */
        Candera::GraphicDeviceUnit * GetGdu() const { return mGdu; }

        /** Returns the Candera::RenderTarget.
            @return <em>GraphicDeviceUnit</em> */
        Candera::RenderTarget * GetRenderTarget() const { return (m3DRT!=0) ? m3DRT : m2DRT; }

        /** Returns the Candera::RenderTarget.
            @return <em>GraphicDeviceUnit</em> */
        Candera::RenderTarget * GetRenderTarget2D() const { return m2DRT; }

        /** Returns the Candera::RenderTarget.
            @return <em>GraphicDeviceUnit</em> */
        Candera::RenderTarget * GetRenderTarget3D() const { return m3DRT; }

        /** Returns the layer id of the Candera::RenderTarget.
            This value is read out from the meta information of the used render target object.
            @return <em>Int</em> the layer id */
        Int GetLayerId() const { return mLayerId; }

        /** Returns the layer index of the Candera::RenderTarget that will be used to locate the Gdu instance.
            @return <em>Int</em> the layer index */
        FeatStd::SizeType GetLayerIndex() const { return mLayerIndex; }

        /** Returns if the Candera::RenderTarget is an OffScreen render target.
            This value is read out from the meta information of the used render target object.
            @return <em>Int</em> the layer number */
        bool IsOffscreen() const { return mIsOffscreen; }

        /** Returns the VideoMemoryHandle of OffScreen render target.
            @return <em>Handle</em> the VideoMemoryHandle */
        Handle GetImageSourceHandle() const { return mImageSourceHandle; }

       /** Returns the SurfaceHandle of an OffScreen render target.
            @return <em>Handle</em> the SurfaceHandle */
        Handle GetSurfaceHandle() const { return mSurfaceHandle; }

        /** Activates or deactivates the rendering of the render target.
            The activation flag might be set to false for debugging purposes and is used by the Renderer if a layer shall be disabled.
            @param active the activation flag which shall be set.*/
        void SetActivate(bool active);

        /** Returns if the render target is active or not.
            @return <em>true</em> if the Gdu object and its render target is active,
                    <em>false</em> otherwise. */
        bool IsActive() const { return mIsActive; }

        /** Uploads the render target.
        @param renderer - pointer to Courier renderer.
        @return <em>true</em> if uploading was successful,
        <em>false</em> otherwise. */
        bool Upload(Renderer* renderer);

        /** Unloads the render target. */
        void Unload(Renderer* renderer);

        /** Activates the render targets. */
        void Activate() const;

        /** Syncs if necessary. */
        void Sync() const;

        /** Returns if Sync shall be called. */
        bool ShallSync() const;

        /** Returns if the render target is loaded.
            @return <em>true</em> if the Gdu object and its render target is loaded,
                    <em>false</em> otherwise. */
        bool IsLoaded() const { return mIsLoaded; }

        /// Tells the object that a camera uses a render strategy.
        void UsesCameraWithRenderStrategy() { mHasCameraWithRenderStrategy = true; }

        /// Returns if the render target is used by a camera which has a render strategy set.
        bool HasCameraWithRenderStrategy() const { return mHasCameraWithRenderStrategy; }

        /// Logs properties of Gdu.
        void Log() const;

        /// Get the number of render target upload requests.
        FeatStd::SizeType GetUploadRequestNo() const { return mUploadRequestNo; }

#if defined(CANDERA_2D_ENABLED)
        /** Creates (on demand) and returns the clear 2D camera.
            @return <em>Camera2D</em> if the camera was created once,
                    <em>0</em> otherwise. */
        Candera::Camera2D * GetClearCamera2D();
#endif

#if defined(CANDERA_3D_ENABLED)
        /** Creates (on demand) and returns the clear 3D camera.
            @return <em>Camera</em> if the camera was created once,
                    <em>0</em> otherwise. */
        Candera::Camera * GetClearCamera3D();

        /** Access the owner of the GraphicDeviceUnit (only available if 3D is enabled).
            NOTE: software layers and the composition camera are only available if 3D is enabled */
        Candera::Internal::GraphicDeviceUnitOwnerAccess& GetOwnerAccess();

        /** Returns the state of the owner of the GraphicDeviceUnit (only available if 3D is enabled).
            NOTE: software layers and the composition camera are only available if 3D is enabled. */
        OwnerState GetOwnerState() const { return mOwnerState; }

        ///
        virtual void OnTransformChange(Candera::Node* /*node*/) { }

        ///
        virtual void OnNodeRemoved(Candera::Node* parent, Candera::Node* node);
#endif

    private:
        friend class UnitTest::GduWrapper;
        
        friend class Renderer;
        friend class RenderJob;
        friend class RenderJobs;

        /// Returns the current render state of the Gdu. Will be used by the Renderer to control the swapping of Gdu's.
        RenderState GetRenderState() const { return mRenderState; }
        /// The Renderer will update the Gdu render state to its current state.
        void SetRenderState(RenderState renderState) { mRenderState = renderState; }

        /// The Renderer will trigger an early swap as soon as the last render job of a Gdu has been completely rendered.
        RenderJob* GetLastRenderJob() const { return mLastRenderJob; }
        /// The Renderer will set the last render job as part of a preprocessing step.
        void SetLastRenderJob(RenderJob* lastRenderJob) { mLastRenderJob = lastRenderJob; }

        /// Helper method
        static bool IsOffScreen(Candera::DevicePackageDescriptor::UnitCategory unitCategory);
        ///
        Candera::Internal::GraphicDeviceUnitOwnerAccess mOwnerAccess;
#if defined(CANDERA_3D_ENABLED)
        ///
        OwnerState mOwnerState;
#endif
        ///
        Candera::GraphicDeviceUnit * mGdu;
        ///
        Candera::RenderTarget * m2DRT;
        ///
        Candera::RenderTarget * m3DRT;

#if defined(CANDERA_2D_ENABLED)
        /// 2D clearing data
        Courier::Internal::ClearingData2D mClearData2D;
#endif
#if defined(CANDERA_3D_ENABLED)
        /// 3D clearing data
        Courier::Internal::ClearingData3D mClearData3D;
#endif
        ///
        Handle mImageSourceHandle;
        ///
        Handle mSurfaceHandle;
        ///
        Int mLayerId;
        ///
        FeatStd::SizeType mLayerIndex;
        ///
        Candera::DevicePackageDescriptor::UnitCategory mUnitCategory;
        ///
        bool mInvalidate;
        ///
        bool mIsActive;
        ///
        bool mIsOffscreen;
        ///
        bool mIsLoaded;
        ///
        bool mHasCameraWithRenderStrategy;

        FeatStd::SizeType mUploadRequestNo;

        FEATSTD_MAKE_CLASS_UNCOPYABLE(Gdu);
        ///
        RenderState mRenderState;
        ///
        RenderJob* mLastRenderJob;
};

/// @}
}

#endif
