//########################################################################
// (C)  Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_FrameworkWidget_h)
#define Courier_Visualization_FrameworkWidget_h

#include <Courier/Base.h>
#include <Courier/Messaging/Message.h>
#include <Courier/DataBinding/DataBindingFwd.h>
#include <Courier/DataBinding/ModelBindingTargetHost.h>
#include <Courier/Visualization/ItemId.h>
#include <Courier/Visualization/CompositePath.h>
#include <Courier/DataBinding/BindableProperty.h>
#include <FeatStd/Util/Optional.h>
#include <Candera/System/Mathematics/Rectangle.h>

#define CdaInjectedWidgetPrivateProperties() \
    Candera::MetaInfo::Internal::AggregationInstanceRef< Courier::Internal::DataBinding::BindablePropertyMetaInfo<CurrentType> > _CdaUniqueName(Item);

// Forward declarations
namespace Courier { 
    class View;
}

namespace Courier { 

/// @addtogroup COURIER_VISUALIZATION
/// @{
/** Base class for widgets, will be injected in Candera::WidgetBase. */
class FrameworkWidget : public Courier::Internal::ModelBindingTargetHost {
public:
    ///
    FrameworkWidget();

    ///
    virtual ~FrameworkWidget();

    /** Is triggered when a message is received.
        @note The return parameter defines that the message shall be routed to the succeeding message
                receiving entities of the dispatching path (<em>false</em>) or if the message is fully
                consumed and any further dispatching of that message shall be stopped (<em>true</em>).
        @param msg Message object to be processed.
        @return <em>true</em> if the message is consumed
                <em>false</em> if the message shall be sent to following message receiving entities. */
    virtual bool OnMessage(const Message & msg);

    /// Returns the ScopeMask of the widget.
    virtual Message::ScopeMask GetScopeMask() const;

    /** Is called whenever the parent view changes its activation state. 
        @param activate if the view was activated or deactivated. */
    virtual void OnParentViewActivate(bool activate);

    /** Is called whenever enable rendering of the parent view is enabled or disabled. 
        @param enable if the rendering is enabled or disabled. */
    virtual void OnParentViewRenderingEnabled(bool enable);

    /** Is called whenever the parent view is loaded or unloaded. 
        @param load if the scene context was loaded or before it gets unloaded. */
    virtual void OnParentViewLoad(bool load);

    /// Invalidates the parent view.
    FEATSTD_LINT_NEXT_EXPRESSION(1735,"Virtual function does not override a base-class function.")
    virtual void Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

    /// Call this method to wake up all render component.
    void WakeUpAllRenderComponents() const;

    /** Set the parent view.
        @param view the parent view. */
    void SetParentView(View * view) {
        mParentView = view;
    }

    /// Returns the parent view.
    View* GetParentView() {
        return mParentView;
    }

    const View* GetParentView() const {
        return mParentView;
    }

    /// Is triggered when the widget got the focus.
    virtual void OnFocus();

    /// Is triggered when the widget looses the focus.
    virtual void OnLostFocus();

    /// Is triggered when the parent view transition was started.
    virtual void OnTransitionStarted(View * firstView, View * secondView);

    /// Is triggered when the parent views transition was finished.
    virtual void OnTransitionFinished(View * firstView, View * secondView, bool wasReset);

    /** Is called to check if the widget is touchable. 
        May be overwritten and may return true if the widget shall be a touchable widget. 
        @return <em>true</em> if the widget shall be checked if it was touched.
                <em>false</em> if the widget shall be not checked if it was touched (default). */
    virtual bool IsTouchable() const;

    /** This class defines a const iterator for the bindings of the widget and overloads all needed operators.*/
    class ConstBindingIterator
    {
        public:
            inline const Courier::Internal::Binding* operator*() const {
                return mCurrent;
            }

            ConstBindingIterator& operator++();

            ConstBindingIterator operator++(Int) {
                ConstBindingIterator it(mCurrent);
                ++(*this);
                return it;
            }

            inline bool operator==(const ConstBindingIterator &it) const {
                return mCurrent == it.mCurrent;
            }

            inline bool operator!=(const ConstBindingIterator &it) const {
                return mCurrent != it.mCurrent;
            }

        private:
            friend class FrameworkWidget;
            ConstBindingIterator(Courier::Internal::Binding* binding) :
               mCurrent(binding)
            {
            }

            mutable Courier::Internal::Binding* mCurrent;
    };

    /** Retrievs the iterator for the bindings of the widget.
        @return ConstBindingIterator object.*/
    ConstBindingIterator GetBindingsIterator() const
    {
            return ConstBindingIterator(mBindingsHead);
    }

protected:
    void PropertyModified(const Char *propertyName) const;

    void DispatchEvent(const Message & msg);

    /// Notifies the parent view of changes in the scope mask.
    virtual void OnScopeMaskChanged();

private:
    View * mParentView;
    ItemId mItemId;
    CompositePath mCompositePath;
    mutable Courier::Internal::Binding *mBindingsHead;

    virtual bool Enlist(Courier::Internal::Binding *binding);
    virtual bool Unlist(Courier::Internal::Binding *binding);

    friend class ViewScene;
    friend class ViewHandler;
};

/// @}

}

/** Candera expects that the injected widget base class is named CanderaInjectedWidgetBase */
#ifndef COURIER_CUSTOM_INJECT_WIDGET_BASE_ENABLED
FEATSTD_LINT_NEXT_EXPRESSION(1960, "this is a correct global typedef to guarantee backwards compatibility")
typedef Courier::FrameworkWidget CanderaInjectedWidgetBase;
#endif //COURIER_CUSTOM_INJECT_WIDGET_BASE_ENABLED

#endif
