//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "FrameworkWidget.h"

#include <Courier/DataBinding/Binding.h>
#include <Courier/DataBinding/PropertyBindingData.h>
#include <Courier/Diagnostics/Log.h>
#include <Courier/Util/ObjectList.h>
#include <Courier/Visualization/View.h>
#include <Courier/Visualization/VisualizationMsgs.h>
#include <Courier/Visualization/Renderer.h>

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

using namespace Internal;

typedef ObjectListFunc<Binding, BindingTargetHostListAccessor> WidgetBindingList;
typedef ObjectListIterator<Binding, BindingTargetHostListAccessor> WidgetBindingListIt;

FrameworkWidget::ConstBindingIterator& FrameworkWidget::ConstBindingIterator::operator++() {
    if (0 != mCurrent) {
        mCurrent = Courier::Internal::BindingTargetHostListAccessor::Next(mCurrent);
    }
    return *this;
}

// ------------------------------------------------------------------------
FrameworkWidget::FrameworkWidget() : mParentView(0), mBindingsHead(0)
{
}

// ------------------------------------------------------------------------
FrameworkWidget::~FrameworkWidget()
{
    WidgetBindingListIt it(mBindingsHead);          // if bindings are registered to the
    while (*it != 0) {                              // widget, unbind the binding objects
        Binding *pb = *it;
        ++it;
        pb->Destroy();
    }
    mBindingsHead = 0;
    mParentView = 0;
}

// ------------------------------------------------------------------------
bool FrameworkWidget::OnMessage(const Message & msg)
{
    FEATSTD_UNUSED(msg);
    return false;
}

// ------------------------------------------------------------------------
Message::ScopeMask FrameworkWidget::GetScopeMask() const
{
    return 0xFFFFFFFF;
}

// ------------------------------------------------------------------------
void FrameworkWidget::OnParentViewActivate(bool activate)
{
    FEATSTD_UNUSED(activate);
}

// ------------------------------------------------------------------------
void FrameworkWidget::OnParentViewRenderingEnabled(bool enable)
{
    FEATSTD_UNUSED(enable);
}

// ------------------------------------------------------------------------
void FrameworkWidget::OnParentViewLoad(bool load)
{
    FEATSTD_UNUSED(load);
}

// ------------------------------------------------------------------------
void FrameworkWidget::Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
    if (0 != mParentView) {
        mParentView->Invalidate(dirtyArea);
    }
}

// ------------------------------------------------------------------------
void FrameworkWidget::WakeUpAllRenderComponents() const
{
    if (0 != mParentView) {
        View::WakeUpAllRenderComponents();
    }
}

// ------------------------------------------------------------------------
bool FrameworkWidget::Enlist(Binding *binding)
{
    return WidgetBindingList::Add(mBindingsHead, binding);
}

// ------------------------------------------------------------------------
bool FrameworkWidget::Unlist(Binding *binding)
{
    return WidgetBindingList::Remove(mBindingsHead, binding);
}

// ------------------------------------------------------------------------
void FrameworkWidget::PropertyModified(const Char *propertyName) const
{
    if (GetParentView() != 0) {
        WidgetBindingListIt it(mBindingsHead);
        while ((*it != 0) && (FeatStd::Internal::String::CompareStrings(propertyName, (*it)->GetTargetName()) != 0)) {
            ++it;
        }

        if (*it != 0) {
            (*it)->SetTargetModifiedTag();
        }
    }
}

// ------------------------------------------------------------------------
void FrameworkWidget::OnFocus()
{
}

// ------------------------------------------------------------------------
void FrameworkWidget::OnLostFocus()
{
}

// ------------------------------------------------------------------------
bool FrameworkWidget::IsTouchable() const
{
    return false;
}

// ------------------------------------------------------------------------
void FrameworkWidget::OnTransitionStarted(View * firstView, View * secondView)
{
    FEATSTD_UNUSED2(firstView, secondView);
}

// ------------------------------------------------------------------------
void FrameworkWidget::OnTransitionFinished(View * firstView, View * secondView, bool wasReset)
{
    FEATSTD_UNUSED3(firstView, secondView, wasReset);
}

// ------------------------------------------------------------------------
void FrameworkWidget::DispatchEvent(const Message & msg)
{
    switch(msg.GetId()) {
        case ParentViewActivateEvent::ID: {
            bool activated = Courier::message_cast<const ParentViewActivateEvent*>(&msg)->GetActivated();
            OnParentViewActivate(activated);
            break;
        }
        case ParentViewRenderingEnabledEvent::ID: {
            bool enabled = Courier::message_cast<const ParentViewRenderingEnabledEvent*>(&msg)->GetEnabled();
            OnParentViewRenderingEnabled(enabled);
            break;
        }
        case ParentViewLoadEvent::ID: {
            bool loaded = Courier::message_cast<const ParentViewLoadEvent*>(&msg)->GetLoaded();
            if (loaded) {
                WidgetBindingListIt it(mBindingsHead);
                while (*it != 0) {
                    if (!(*it)->OnTargetInitialized()) {
                        COURIER_LOG_WARN("OnWidgetInitialized failed on property %s", (*it)->GetTargetName());
                    }
                    ++it;
                }
            }
            OnParentViewLoad(loaded);
            break;
        }
        case FocusEvent::ID: {
            OnFocus();
            break;
        }
        case LostFocusEvent::ID: {
            OnLostFocus();
            break;
        }
        case TransitionStartedEvent::ID: {
            const TransitionStartedEvent * ev = Courier::message_cast<const TransitionStartedEvent*>(&msg);
            View * firstView = ev->GetFirstView();
            View * secondView = ev->GetSecondView();
            OnTransitionStarted(firstView,secondView);
            break;
        }
        case TransitionFinishedEvent::ID: {
            const TransitionFinishedEvent * ev = Courier::message_cast<const TransitionFinishedEvent*>(&msg);
            View * firstView = ev->GetFirstView();
            View * secondView = ev->GetSecondView();
            bool wasReset = ev->GetWasReset();
            OnTransitionFinished(firstView,secondView,wasReset);
            break;
        }
        default: {
            // garantuee to fall through !
        }
    }
    bool rc = OnMessage(msg);
    FEATSTD_UNUSED(rc);
}

void FrameworkWidget::OnScopeMaskChanged()
{
    if (0 != GetParentView()) {
        GetParentView()->OnScopeMaskChanged();
    }
}

}
