//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_CompositionPath_h)
#define Courier_Visualization_CompositionPath_h

#include <Courier/Util/FixedSizeString.h>
#include <Courier/Util/HashId.h>
#include <FeatStd/Platform/String.h>
#include <FeatStd/Util/StringBuffer.h>

// ----------------------------------------------------------------------------
namespace Courier { 
/// @addtogroup COURIER_VISUALIZATION
/// @{
/** An CompositePath represents the path to a composite instance.
 *  CompositePath(), CompositePath("") and CompositePath(reinterpret_cast<const Candera::StringIdentifier*>(0)) will have the same results
 *  StringIdentifiers will be handled with a "/" separator.
 *
 *  On comparison between two CompositePath instances the generated hash is used. The hash generation is different depending on the way how the CompositePath was initialized.
 *  For a StringIdentifier with an input of "/Scene:MyModule#MyScene2D/Group/MyComposite" the hash is calculated from "Group/MyComposite".
 *  This is useful as the CompositePath in this way only considers the relevant part of a CompositePath.
 *  A CompositePath initialized with a FeatStd::Char* has to be initialized with the value "Group/MyComposite" to have the same hash.
 */
class CompositePath 
{
    public:
        ///
        enum IdentifierType
        {
            StringIdentifierType,
            CStringType
        };

        ///
        CompositePath();

        ///
        explicit CompositePath(const FeatStd::Char* compositePathString);

        /// 
        explicit CompositePath(const Candera::StringIdentifier* stringIdentifier);

        ///
        CompositePath(const CompositePath & compositePath);

        /// CompositePath comparison. The generated hash is used for comparison.
        bool operator==(const CompositePath & id) const {
            return id.mHashId == mHashId;
        }
        /// CompositePath comparison. The generated hash is used for comparison.
        bool operator!=(const CompositePath & id) const {
            return id.mHashId != mHashId;
        }

        ///
        const HashId GetHashId() const {
            return mHashId;
        }

        ///
        IdentifierType GetIdentifierType() const {
            return mIdentifierType;
        }

        ///
        const FeatStd::Char* ToCString() const {
            return mId;
        }

        ///
        const Candera::StringIdentifier* ToStringIdentifier() const {
            return mStringIdentifier;
        }

private:
        ///
        IdentifierType mIdentifierType;
        union {
            ///
            const Candera::StringIdentifier* mStringIdentifier;
            ///
            const FeatStd::Char* mId;
        };
        ///
        HashId mHashId;
};

/// @}
}


#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {
    template<> UInt32 StringBufferAppender< Courier::CompositePath >::Append(StringBuffer& stringBuffer, Courier::CompositePath const & object);
}
#endif

#endif
