//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CameraGroupHandler.h"
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

#include <Courier/Diagnostics/Log.h>
#include "View.h"
#include <Courier/Visualization/ViewScene2D.h>
#include <Courier/Visualization/ViewScene3D.h>
#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif


// ------------------------------------------------------------------------
// CameraGroupHandler::Group methods
// ------------------------------------------------------------------------

namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

using namespace Candera;

namespace Internal {
    // ------------------------------------------------------------------------
    template<typename NodeType> static Candera::CanderaObject* CameraGetParent(Candera::CanderaObject* object)
    {
        Candera::CanderaObject* parent = 0;
        NodeType* node = static_cast<NodeType*>(object);
        if (0 != node) {
            parent = node->GetParent();
        }
        return parent;
    }

    typedef Candera::CanderaObject* (*GetParentCallback)(Candera::CanderaObject* object);

    // ------------------------------------------------------------------------
    static SizeType GetQualifiedNameLength(CanderaObject* node, GetParentCallback getParentCallback)
    {
        SizeType length = 0;
        Candera::CanderaObject* currentNode = node;
        while (0 != currentNode) {
            length += StringPlatform::Length(currentNode->GetName()) + 1; // add 1 for delimiter '@' or '\0'
            currentNode = getParentCallback(currentNode);
        }
        return length;
    }

    // ------------------------------------------------------------------------
    static Char* GetQualifiedName(Char* buffer, CanderaObject* node, GetParentCallback getParentCallback)
    {
        CanderaObject* parent = getParentCallback(node);
        if (0 !=parent) {
            buffer = GetQualifiedName(buffer, parent, getParentCallback);
            *buffer = '@';
            ++buffer;
        }
        if (0 != node) {
            const Char* name = node->GetName();
            SizeType length = StringPlatform::Length(name);
            StringPlatform::Copy(buffer, name);
            buffer += length;
        }
        return buffer;
    }
}

using namespace Internal;

// ------------------------------------------------------------------------
CameraGroupHandler::Group::Group() :
    mId(),
    mCameraGroup(0),
    mViewHandler(0),
    mIsRenderingEnabled(false),
    mIsActivated(false),
    mNameVct()
{
}

// ------------------------------------------------------------------------
CameraGroupHandler::Group::Group(const ItemId & id, Candera::CameraGroup* cg, IViewHandler* viewHandler) :
    mId(id),
    mCameraGroup(cg),
    mViewHandler(viewHandler),
    mIsRenderingEnabled(false),
    mIsActivated(false),
    mNameVct()
{
    RegisterAsListener();
}

// ------------------------------------------------------------------------
CameraGroupHandler::Group::Group(const Group & ref) :
#if defined(CANDERA_3D_ENABLED)
    NodeListener(ref),
#endif
#if defined(CANDERA_2D_ENABLED)
    Node2DListener(ref),
#endif
    mId(ref.mId),
    FEATSTD_LINT_NEXT_EXPRESSION(1554, "Direct pointer copy of member, thats what we want")
    mCameraGroup(ref.mCameraGroup),
    mViewHandler(ref.mViewHandler),
    mIsRenderingEnabled(ref.mIsRenderingEnabled),
    mIsActivated(ref.mIsActivated),
    mNameVct(ref.mNameVct)
{
    RegisterAsListener();
}

void CameraGroupHandler::Group::RegisterAsListener()
{
    FEATSTD_DEBUG_ASSERT(0 != mCameraGroup);
    if (0 != mCameraGroup) {
#ifdef CANDERA_2D_ENABLED
        for (Int i = 0; i < static_cast<Int>(mCameraGroup->GetCamera2DCount()); ++i) {
            if (0 != mCameraGroup->GetCamera2D(i)) {
                bool result = mCameraGroup->GetCamera2D(i)->AddNodeListener(this);
                FEATSTD_UNUSED(result); // if logging is disabled, variable/parameter is not used
                COURIER_LOG_DEBUG("AddNode2DListener returned %s for camera %s", ((result) ? "TRUE" :"FALSE"), mCameraGroup->GetCamera2D(i)->GetName());
            }
        }
#endif // CANDERA_2D_ENABLED
#ifdef CANDERA_3D_ENABLED
        for (Int i = 0; i < static_cast<Int>(mCameraGroup->GetCameraCount()); ++i) {
            if (0 != mCameraGroup->GetCamera(i)) {
                bool result = mCameraGroup->GetCamera(i)->AddNodeListener(this);
                FEATSTD_UNUSED(result); // if logging is disabled, variable/parameter is not used
                COURIER_LOG_DEBUG("AddNodeListener returned %s for camera %s", ((result) ? "TRUE" :"FALSE"), mCameraGroup->GetCamera(i)->GetName());
            }
        }
#endif //CANDERA_3D_ENABLED
    }
}

// ------------------------------------------------------------------------
CameraGroupHandler::Group::~Group()
{
    if (0 != mCameraGroup) {
#ifdef CANDERA_2D_ENABLED
        for (Int i = 0; i < static_cast<Int>(mCameraGroup->GetCamera2DCount()); ++i) {
            if (0 != mCameraGroup->GetCamera2D(i)) {
                bool result = mCameraGroup->GetCamera2D(i)->RemoveNodeListener(this);
                FEATSTD_UNUSED(result); // if logging is disabled, variable/parameter is not used
                COURIER_LOG_DEBUG("RemoveNode2DListener returned %s for camera %s", ((result) ? "TRUE" :"FALSE"), mCameraGroup->GetCamera2D(i)->GetName());
            }
        }
#endif //CANDERA_2D_ENABLED
#ifdef CANDERA_3D_ENABLED
        for (Int i = 0; i < static_cast<Int>(mCameraGroup->GetCameraCount()); ++i) {
            if (0 != mCameraGroup->GetCamera(i)) {
                bool result = mCameraGroup->GetCamera(i)->RemoveNodeListener(this);
                FEATSTD_UNUSED(result); // if logging is disabled, variable/parameter is not used
                COURIER_LOG_DEBUG("RemoveNodeListener returned %s for camera %s", ((result) ? "TRUE" :"FALSE"), mCameraGroup->GetCamera(i)->GetName());
            }
        }
#endif //CANDERA_3D_ENABLED
    }
    mViewHandler = 0;
    mCameraGroup = 0;
    mNameVct.Clear();
}
#if defined(CANDERA_3D_ENABLED)
// ------------------------------------------------------------------------
void CameraGroupHandler::Group::OnNodeRemoved(Candera::Node* parent, Candera::Node* node)
{
    if (0 != node) {
        for (Int i = 0; i <  static_cast<Int>(mCameraGroup->GetCameraCount()); ++i) {
            if (node == mCameraGroup->GetCamera(i)) {
                if (0 != parent) {
                    const Char* sceneName = GetName(parent->GetScene());
                    ViewScene3D* viewScene3D = FindViewScene3D(sceneName);
                    if (0 != viewScene3D){
                        StoreNames(viewScene3D, parent, node, CameraGetParent<Candera::Node>);
                    }
                    mCameraGroup->RemoveCamera(i);
                }
            }
        }
    }
}
#endif
#if defined(CANDERA_2D_ENABLED)
// ------------------------------------------------------------------------
void CameraGroupHandler::Group::OnNodeRemoved(Candera::Node2D* parent, Candera::Node2D* node)
{
    if (0 != node) {
        for (Int i = 0; i <  static_cast<Int>(mCameraGroup->GetCamera2DCount()); ++i) {
            if (node == mCameraGroup->GetCamera2D(i)) {
                if (0 != parent) {
                    const Char* sceneName = GetName(parent->GetScene());
                    ViewScene2D* viewScene2D = FindViewScene2D(sceneName);
                    if (0 != viewScene2D) {
                        StoreNames(viewScene2D, parent, node, CameraGetParent<Candera::Node2D>);
                    }
                    mCameraGroup->RemoveCamera2D(i);
                }
            }
        }
    }
}
#endif

// ------------------------------------------------------------------------
void CameraGroupHandler::Group::RetrieveLostCamerasFromScene(ViewScene* viewScene)
{
    if (0 != viewScene) {
        for (FeatStd::OffsetType i = static_cast<FeatStd::OffsetType>(mNameVct.Size() - 1); i >= 0; --i) {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(mNameVct[i].mSceneName.GetCriticalSection());
#endif
            if ((viewScene->GetId() == mNameVct[i].mViewId) ||
                (0 == StringPlatform::CompareStrings(viewScene->GetId().CStr(), mNameVct[i].mSceneName.GetCString()))){
#ifdef CANDERA_2D_ENABLED
                    ViewScene2D* viewScene2D = viewScene->ToViewScene2D();
                    if (0 != viewScene2D) {
                        Camera2D* camera2D = viewScene2D->FindCamera2D(mNameVct[i].mCameraName.GetCString());
                        if (0 != camera2D) {
                            bool result = camera2D->AddNodeListener(this);
                            FEATSTD_UNUSED(result); // if logging is disabled, variable/parameter is not used
                            COURIER_LOG_DEBUG("AddNode2DListener returned %s for camera %s", ((result) ? "TRUE" :"FALSE"), camera2D->GetName());
                            mCameraGroup->AddCamera2D(camera2D);
                            if (mIsActivated) {
                                viewScene->ActivateForCamera(mIsActivated, camera2D);
                            }
                            if (mIsRenderingEnabled) {
                                viewScene->EnableRenderingForCamera(mIsRenderingEnabled, camera2D);
                            }
                        }
                    }
#endif //CANDERA_2D_ENABLED
#ifdef CANDERA_3D_ENABLED
                    ViewScene3D* viewScene3D = viewScene->ToViewScene3D();
                    if (0 != viewScene3D) {
                        Camera* camera = viewScene3D->FindCamera3D(mNameVct[i].mCameraName.GetCString());
                        if (0 != camera) {
                            bool result = camera->AddNodeListener(this);
                            FEATSTD_UNUSED(result); // if logging is disabled, variable/parameter is not used
                            COURIER_LOG_DEBUG("AddNodeListener returned %s for camera %s", ((result) ? "TRUE" :"FALSE"), camera->GetName());
                            mCameraGroup->AddCamera(camera);
                            if (mIsActivated) {
                                viewScene->ActivateForCamera(mIsActivated, camera);
                            }
                            if (mIsRenderingEnabled) {
                                viewScene->EnableRenderingForCamera(mIsRenderingEnabled, camera);
                            }
                        }
                    }
#endif //CANDERA_3D_ENABLED
                    static_cast<void>(mNameVct.Remove(i));
            }
        }
    }
}

// ------------------------------------------------------------------------
void CameraGroupHandler::Group::StoreNames(const ViewScene* viewScene, Candera::CanderaObject* parent, Candera::CanderaObject* camera, GetParentCallback getParentCallback)
{
    if ((0 != viewScene) && (0 != parent) && (0 != camera)) {
        const Char* sceneName = viewScene->GetId().CStr();
        if (0 != sceneName) {
            SizeType parentLength = GetQualifiedNameLength(parent, getParentCallback);
            SizeType cameraLenght = GetQualifiedNameLength(camera, getParentCallback);
            Char* cameraFullName = CANDERA_NEW_ARRAY(Char, parentLength + cameraLenght + 1);
            if (0 != cameraFullName) {
                // parent of the camera is set to 0;
                // thus, for getting the cameraFullName, first we must retrieve the parent name and after this add a delimiter '@' and the actual camera name
                Char* bufferEnd = GetQualifiedName(cameraFullName, parent, getParentCallback); 
                *bufferEnd = '@';
                ++bufferEnd;
                bufferEnd = GetQualifiedName(bufferEnd, camera, getParentCallback);
                *bufferEnd = '\0';
                bool result = mNameVct.Add(NodesPathNames(viewScene->GetId(), sceneName, cameraFullName)); // Makes a copy of cameraFullName so we can delete local char*
                FEATSTD_UNUSED(result); // if logging is disabled, variable/parameter is not used
                COURIER_LOG_DEBUG("mNameVct.Add returned %s for %s %s", ((result) ? "TRUE" :"FALSE"), sceneName, cameraFullName);
                CANDERA_DELETE_ARRAY(cameraFullName);
            }
        }
    }
}

// ------------------------------------------------------------------------
CameraGroupHandler::NodesPathNames* CameraGroupHandler::Group::GetNodesPathNames(FeatStd::SizeType idx)
{
    return (ValidIndex(idx, mNameVct.Size())) ? &mNameVct[idx] : 0;
}

#if defined(CANDERA_3D_ENABLED)

// ------------------------------------------------------------------------
FeatStd::SizeType CameraGroupHandler::Group::GetCamera3DCount() const
{
    FEATSTD_DEBUG_ASSERT(0 != mCameraGroup);
    return (0 != mCameraGroup) ? mCameraGroup->GetCameraCount() : 0;
}

// ------------------------------------------------------------------------
const Char * CameraGroupHandler::Group::GetScene3DName(Int index) const
{
    const Candera::Camera* camera =  GetCamera3D(index);
    return (0 != camera) ? GetName(camera->GetScene()) : 0;
}

// ------------------------------------------------------------------------
const Char* CameraGroupHandler::Group::GetCamera3DName(Int index) const
{
    const Candera::Camera* camera = GetCamera3D(index);
    return (0 != camera) ? GetName(camera) : 0;
}

// ------------------------------------------------------------------------
Candera::Camera* CameraGroupHandler::Group::GetCamera3D(Int index) const
{
    return (ValidIndex(index, GetCamera3DCount())) ? Valid(mCameraGroup->GetCamera(index)) : 0;
}

// ------------------------------------------------------------------------
ViewScene3D* CameraGroupHandler::Group::FindViewScene3D(const Char* sceneName) const
{
    ViewScene3D * viewScene = 0;
    FEATSTD_DEBUG_ASSERT(0 != sceneName);

    if (0 != mViewHandler) {
        View * view = mViewHandler->FindView(ViewId(sceneName));
        if ((0 != view) && (view->Is2D() == false)) {
            viewScene = view->ToViewScene3D();
        }
    }
    if (0 == viewScene) {
        COURIER_LOG_ERROR("View '%s' not created yet", sceneName);
    }
    return viewScene;
}

// ------------------------------------------------------------------------
void CameraGroupHandler::Group::ActivateView(ViewScene3D& viewScene, Int cameraId, bool activate) const
{
    const Camera* cameraFromScene = GetCamera3D(cameraId);
    COURIER_LOG_DEBUG("camera3D = %s, activate = %s", ((0 != cameraFromScene) ? cameraFromScene->GetName() : "NULL"), (activate ? "TRUE" : "FALSE"));
    viewScene.ActivateForCamera(activate, cameraFromScene);
}

// ------------------------------------------------------------------------
void CameraGroupHandler::Group::EnableRendering(ViewScene3D& viewScene, Int cameraId, bool enable, bool onlyCameraGroupCameras) const
{
    const Camera * cameraFromScene = GetCamera3D(cameraId);
    COURIER_LOG_DEBUG("camera3D = %s, enable = %s", ((0 != cameraFromScene) ? cameraFromScene->GetName() : "NULL"), (enable ? "TRUE" : "FALSE"));
    if (onlyCameraGroupCameras) {
        viewScene.EnableRenderingForCamera(enable, cameraFromScene);
    } else {
        viewScene.EnableRenderingImpl(enable);
    }
}

// ------------------------------------------------------------------------
void CameraGroupHandler::Group::Invalidate(ViewScene3D& viewScene, Int cameraId, bool onlyCameraGroupCameras, const FeatStd::Optional<Candera::Rectangle>& dirtyArea) const
{
    Camera * cameraFromScene = GetCamera3D(cameraId);
    if (onlyCameraGroupCameras) {
        viewScene.Invalidate(cameraFromScene, dirtyArea);
    }
    else {
        viewScene.Invalidate(dirtyArea);
    }
}
#endif //defined(CANDERA_3D_ENABLED)

#if defined(CANDERA_2D_ENABLED)
// ------------------------------------------------------------------------
FeatStd::SizeType CameraGroupHandler::Group::GetCamera2DCount() const
{
    FEATSTD_DEBUG_ASSERT(0 != mCameraGroup);
    return (0 != mCameraGroup) ? mCameraGroup->GetCamera2DCount() : 0;
}

// ------------------------------------------------------------------------
const Char* CameraGroupHandler::Group::GetScene2DName(Int index) const
{
    const Candera::Camera2D* camera =  GetCamera2D(index);
    return (0 != camera) ? GetName(camera->GetScene()) : 0;
}

// ------------------------------------------------------------------------
const Char* CameraGroupHandler::Group::GetCamera2DName(Int index) const
{
    const Candera::Camera2D* camera = GetCamera2D(index);
    return (0 != camera) ? GetName(camera) : 0;
}

// ------------------------------------------------------------------------
Candera::Camera2D* CameraGroupHandler::Group::GetCamera2D(Int index) const
{
    return (ValidIndex(index, GetCamera2DCount())) ? Valid(mCameraGroup->GetCamera2D(index)) : 0;
}

// ------------------------------------------------------------------------
ViewScene2D* CameraGroupHandler::Group::FindViewScene2D(const Char* sceneName) const
{
    ViewScene2D * viewScene = 0;
    FEATSTD_DEBUG_ASSERT(0 != sceneName);

    if (0 != mViewHandler) {
        View * view = mViewHandler->FindView(ViewId(sceneName));
        if ((0 != view) && view->Is2D()) {
            viewScene = view->ToViewScene2D();
        }
    }
    if (0 == viewScene) {
        COURIER_LOG_ERROR("View '%s' not created yet", sceneName);
    }
    return viewScene;
}

// ------------------------------------------------------------------------
void CameraGroupHandler::Group::ActivateView(ViewScene2D& viewScene, Int cameraId, bool activate) const
{
    const Camera2D* cameraFromScene = GetCamera2D(cameraId);
    COURIER_LOG_DEBUG("camera2D = %s, activate = %s", ((0 != cameraFromScene) ? cameraFromScene->GetName() : "NULL"), (activate ? "TRUE" : "FALSE"));
    viewScene.ActivateForCamera(activate, cameraFromScene);
}

// ------------------------------------------------------------------------
void CameraGroupHandler::Group::EnableRendering(ViewScene2D& viewScene, Int cameraId, bool enable, bool onlyCameraGroupCameras) const
{
    const Camera2D * cameraFromScene = GetCamera2D(cameraId);
    COURIER_LOG_DEBUG("camera2D  = %s, enable = %s", ((0 != cameraFromScene) ? cameraFromScene->GetName() : "NULL"), (enable ? "TRUE" : "FALSE"));
    if (onlyCameraGroupCameras) {
        viewScene.EnableRenderingForCamera(enable, cameraFromScene);
    } else {
        viewScene.EnableRenderingImpl(enable);
    }
}

// ------------------------------------------------------------------------
void CameraGroupHandler::Group::Invalidate(ViewScene2D& viewScene, Int cameraId, bool onlyCameraGroupCameras, const FeatStd::Optional<Candera::Rectangle>& dirtyArea) const
{
    Camera2D * cameraFromScene = GetCamera2D(cameraId);
    if (onlyCameraGroupCameras) {
        viewScene.Invalidate(cameraFromScene, dirtyArea);
    } else {
        viewScene.Invalidate(dirtyArea);
    }
}
#endif //defined(CANDERA_2D_ENABLED)

// ------------------------------------------------------------------------
template<typename T>
T* CameraGroupHandler::Group::Valid(T* p)
{
    FEATSTD_DEBUG_ASSERT(0 != p);
    return p;
}

// ------------------------------------------------------------------------
const Char* CameraGroupHandler::Group::GetName(const Candera::CanderaObject* object)
{
    FEATSTD_DEBUG_ASSERT(0 != object);
    return (0 != object) ? Valid(object->GetName()) : 0;
}

// ------------------------------------------------------------------------
bool CameraGroupHandler::Group::ValidIndex(FeatStd::OffsetType index, FeatStd::OffsetType count)
{
    FEATSTD_DEBUG_ASSERT(index >= 0);
    if (index >= 0) {
        FEATSTD_DEBUG_ASSERT(index < count);
        if (index < count) {
            return true;
        }
    }
    return false;
}

} // end namespace Courier

// ------------------------------------------------------------------------
// CameraGroupHandler methods
// ------------------------------------------------------------------------

namespace Courier {
// ------------------------------------------------------------------------
CameraGroupHandler::CameraGroupHandler() : mViewHandler(0)
{
}

// ------------------------------------------------------------------------
CameraGroupHandler::~CameraGroupHandler()
{
    mViewHandler = 0;
}

// ------------------------------------------------------------------------
void CameraGroupHandler::Finalize()
{
    mViewHandler = 0;
    mGroupVector.Clear();
}

// ------------------------------------------------------------------------
bool CameraGroupHandler::Init(IViewHandler * viewHandler)
{
    FEATSTD_DEBUG_ASSERT(viewHandler!=0);
    mViewHandler = viewHandler;
    return mViewHandler!=0;
}

// ------------------------------------------------------------------------
CameraGroupHandler::Group * CameraGroupHandler::GetGroup(const ItemId & cameraGroupId)
{
    for (SizeType i = 0; i < mGroupVector.Size(); i++) {
        if(mGroupVector[i].GetId()==cameraGroupId) {
            return &mGroupVector[i];
        }
    }
    return 0;
}
#if defined(CANDERA_3D_ENABLED)
// ------------------------------------------------------------------------
bool CameraGroupHandler::IsCamera3DMultipleUsed(const Camera * lookupCamera, const CameraGroupHandler::Group * excludeGroup) const 
{
    Int count = 0;
    for (SizeType i = 0; i < mGroupVector.Size(); i++) {
        if(&mGroupVector[i]!=excludeGroup) {
            const CameraGroup * cg = mGroupVector[i].GetCameraGroup();
            if((cg!=0) && mGroupVector[i].IsRenderingEnabled()) {
                for(Int idx = 0; idx < static_cast<Int>(cg->GetCameraCount()); ++idx) {
                    const Camera * camera = cg->GetCamera(idx);
                    if(lookupCamera==camera) {
                        ++count;
                    }
                }
            }
        }
    }
    return count>0;
}
#endif
#if defined(CANDERA_2D_ENABLED)
// ------------------------------------------------------------------------
bool CameraGroupHandler::IsCamera2DMultipleUsed(const Camera2D * lookupCamera, const CameraGroupHandler::Group * excludeGroup) const 
{
    Int count = 0;
    for (SizeType i = 0; i < mGroupVector.Size(); i++) {
        if(&mGroupVector[i]!=excludeGroup) {
            const CameraGroup * cg = mGroupVector[i].GetCameraGroup();
            if((cg!=0) && mGroupVector[i].IsRenderingEnabled()) {
                for(Int idx = 0; idx< static_cast<Int>(cg->GetCamera2DCount()); ++idx) {
                    const Camera2D * camera = cg->GetCamera2D(idx);
                    if(lookupCamera==camera) {
                        ++count;
                    }
                }
            }
        }
    }
    return count>0;
}
#endif

// ------------------------------------------------------------------------
bool CameraGroupHandler::Load(const ItemId & cameraGroupId) 
{
    FEATSTD_DEBUG_ASSERT(mViewHandler!=0);
    const Char * camGroupName = cameraGroupId.CStr();
    if(mViewHandler!=0) {
        Candera::AssetProvider * assetProvider = mViewHandler->GetAssetProvider();
        if(assetProvider!=0) {
            Candera::Id id = Candera::Internal::AssetProviderFunctions::GetIdByName(assetProvider, CameraGroupLib, camGroupName);
            Candera::CameraGroup * cg = assetProvider->GetCameraGroupById(id);
            if ((0 != cg) && mGroupVector.Add(Group(cameraGroupId, cg, mViewHandler))) {
                CameraGroupHandler::Group * group = GetGroup(cameraGroupId);
                if (0 != group) {
                    COURIER_LOG_INFO("Loaded CameraGroup '%s'", camGroupName);
                    return true;
                }
            } 
        }
    }
    COURIER_LOG_ERROR("Cannot load CameraGroup '%s'", camGroupName);
    return false;
}

// ------------------------------------------------------------------------
bool CameraGroupHandler::CreateCameraGroupViews(ViewAction::Enum action, const ItemId & cameraGroupId, bool loadOnCreation)
{
    if((action==ViewAction::Destroy) || (action==ViewAction::DestroyAll)) {
        COURIER_LOG_ERROR("Action (%d) not supported",action);
        return false;
    }

    FEATSTD_DEBUG_ASSERT(mViewHandler!=0);
    CameraGroupHandler::Group * group = GetGroup(cameraGroupId);  
    if(group==0) {
        if(! Load(cameraGroupId)) {
            return false;
        }
        group = GetGroup(cameraGroupId);  
    }
    FEATSTD_DEBUG_ASSERT(group!=0);
    if((mViewHandler!=0) && (group!=0)) { 
            // special case CreateAll: create all View but do not load them, instead only load the views of the given camera group
            if(action==ViewAction::CreateAll) {
                bool rc = mViewHandler->ExecuteViewAction(action,ViewId(),false,true);
                if(rc) {
                    COURIER_LOG_INFO("Created all Views without loading them ...");
                }
                action = ViewAction::Create;
            }
        Int idx;
#if defined(CANDERA_3D_ENABLED)
        for(idx=0; idx< static_cast<Int>(group->GetCamera3DCount()); ++idx) {
            const Char * sceneName3D = group->GetScene3DName(idx);
            if(sceneName3D!=0) {
                bool rc = mViewHandler->ExecuteViewAction(action,ViewId(sceneName3D),loadOnCreation, true);
                if(rc) {
                    COURIER_LOG_INFO("Processed ViewScene3D '%s' part of CamGrp",sceneName3D);
                }
            }
        }
#endif
#if defined(CANDERA_2D_ENABLED)
        for(idx=0; idx< static_cast<Int>(group->GetCamera2DCount()); ++idx) {
            const Char * sceneName2D = group->GetScene2DName(idx);
            if(sceneName2D!=0) {
                bool rc = mViewHandler->ExecuteViewAction(action,ViewId(sceneName2D),loadOnCreation, true);
                if(rc) {
                    COURIER_LOG_INFO("Processed ViewScene2D '%s' part of CamGrp",sceneName2D);
                }

            }
        }
#endif
        return true;
    }
    return false;
}

// ------------------------------------------------------------------------
bool CameraGroupHandler::ActivateCameraGroupCameras(const ItemId & cameraGroupId, bool activate, bool enableCameras, bool onlyCameraGroupCameras)
{
    FEATSTD_DEBUG_ASSERT(0 != mViewHandler);

    CameraGroupHandler::Group * group = GetGroup(cameraGroupId);
    if (0 == group) {
        if (!Load(cameraGroupId)) {
            return false;
        }
        else {
            group = GetGroup(cameraGroupId);
        }
    }

    FEATSTD_DEBUG_ASSERT(0 != group);

    if ((0 != mViewHandler) && (0 != group)) {
#if defined(CANDERA_3D_ENABLED)
        for (Int idx = 0; idx < static_cast<Int>(group->GetCamera3DCount()); ++idx) {
            const Char * sceneName = group->GetScene3DName(idx);
            ViewScene3D * viewScene = group->FindViewScene3D(sceneName);
            if (0 != viewScene) {
                if (activate != group->IsActivated()) {
                    group->ActivateView(*viewScene, idx, activate);
                }
                if (enableCameras != group->IsRenderingEnabled()) {
                    group->EnableRendering(*viewScene, idx, enableCameras, onlyCameraGroupCameras);
                }
                group->Invalidate(*viewScene, idx, onlyCameraGroupCameras, FeatStd::Optional<Candera::Rectangle>());
            }
        }
#endif
#if defined(CANDERA_2D_ENABLED)
        for (Int idx = 0; idx < static_cast<Int>(group->GetCamera2DCount()); ++idx) {
            const Char * sceneName = group->GetScene2DName(idx);
            ViewScene2D * viewScene = group->FindViewScene2D(sceneName);
            if (0 != viewScene) {
                if (activate != group->IsActivated()) {
                    group->ActivateView(*viewScene, idx, activate);
                }
                if (enableCameras != group->IsRenderingEnabled()) {
                    group->EnableRendering(*viewScene, idx, enableCameras, onlyCameraGroupCameras);
                }
                group->Invalidate(*viewScene, idx, onlyCameraGroupCameras, FeatStd::Optional<Candera::Rectangle>());
            }
        }
#endif
        if (activate != group->IsActivated()) {
            group->SetActivated(activate);
        }
        if (enableCameras != group->IsRenderingEnabled()) {
            group->SetRenderingEnabled(enableCameras);
        }
       return true;
    }
    return false;
}

// ------------------------------------------------------------------------
void CameraGroupHandler::LoadView(bool is2D, const Char * sceneName, bool load)
{
    FEATSTD_DEBUG_ASSERT(mViewHandler!=0);
    if((mViewHandler!=0) && (sceneName!=0)) {
        bool rc = true;
#ifdef FEATSTD_LOG_ENABLED
        const Char * xDString = (is2D ? "2D" : "3D");
#else
        FEATSTD_UNUSED(is2D); // if logging is disabled, variable/parameter is not used
#endif
        View * view = mViewHandler->FindView(ViewId(sceneName));
        if(view!=0) {
            if((load && (! view->IsContentLoaded())) || ((! load) && view->IsContentLoaded())) {
                rc = view->LoadContent(load, true);
            }
            if(! rc) {
                COURIER_LOG_ERROR("ViewScene%s '%s' cannot be %s",xDString, sceneName, (load ? "LOADED" : "UNLOADED"));
            }
        } else {
            COURIER_LOG_ERROR("ViewScene%s '%s' not created yet",xDString, sceneName);
        }
    }
}

// ------------------------------------------------------------------------
bool CameraGroupHandler::LoadCameraGroupViews(const ItemId & cameraGroupId, bool load)
{
    FEATSTD_DEBUG_ASSERT(mViewHandler!=0);
    CameraGroupHandler::Group * group = GetGroup(cameraGroupId);  
    if(group==0) {
        if(! Load(cameraGroupId)) {
            return false;
        }
        group = GetGroup(cameraGroupId);  
    }
    FEATSTD_DEBUG_ASSERT(group!=0);
    if((mViewHandler!=0) && (group!=0)) {
        Int idx;
#if defined(CANDERA_3D_ENABLED)
        for(idx = 0; idx < static_cast<Int>(group->GetCamera3DCount()); ++idx) {
            bool execute = true;
            const Camera * camera3D = group->GetCamera3D(idx);
            FEATSTD_DEBUG_ASSERT(camera3D!=0);

            if(! load) {
                execute = ! IsCamera3DMultipleUsed(camera3D,group);
            }
            if(! execute) {
                COURIER_LOG_WARN("Camera3D '%s' referenced multiple times, keep view LOADED",camera3D->GetName());
            } else {
                LoadView(false,group->GetScene3DName(idx),load);
            }
        }
#endif
#if defined(CANDERA_2D_ENABLED)
        for(idx = 0; idx < static_cast<Int>(group->GetCamera2DCount()); ++idx) {
            bool execute = true;
            const Camera2D * camera2D = group->GetCamera2D(idx);
            FEATSTD_DEBUG_ASSERT(camera2D!=0);

            if(! load) {
                execute = ! IsCamera2DMultipleUsed(camera2D,group);
            }
            if(! execute) {
                COURIER_LOG_WARN("Camera2D '%s' referenced multiple times, keep view LOADED",camera2D->GetName());
            } else {
                LoadView(true,group->GetScene2DName(idx),load);
            }
        }
#endif
        return true;
    }
    return false;
}

// ------------------------------------------------------------------------
void CameraGroupHandler::OnViewInitContent(bool init, View& view)
{
    if (init) {
        for (SizeType i = 0 ; i < mGroupVector.Size(); ++i) {
            for (SizeType idx = 0; idx < mGroupVector[i].GetNamesVctCount(); ++idx) {
                NodesPathNames* nodePathNames = mGroupVector[i].GetNodesPathNames(idx);
                if (0 != nodePathNames) {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                    FeatStd::Internal::CriticalSectionLocker lock(nodePathNames->mSceneName.GetCriticalSection());
#endif
                    if ((view.GetId() == nodePathNames->mViewId) || 
                        (0 == StringPlatform::CompareStrings(view.GetId().CStr(), nodePathNames->mSceneName.GetCString()))) {
#ifdef CANDERA_2D_ENABLED
                        ViewScene2D* viewScene2D = view.ToViewScene2D();
                        if (0 != viewScene2D){
                            mGroupVector[i].RetrieveLostCamerasFromScene(viewScene2D);
                        }
#endif //CANDERA_2D_ENABLED
#ifdef CANDERA_3D_ENABLED
                        ViewScene3D* viewScene3D = view.ToViewScene3D();
                        if (0 != viewScene3D) {
                            mGroupVector[i].RetrieveLostCamerasFromScene(viewScene3D);
                        }
#endif //CANDERA_3D_ENABLED
                    }
                }
            }
        }
    }
}

}

