//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Visualization_AnimationInvalidator_h)
#define Courier_Visualization_AnimationInvalidator_h

#include <Candera/Candera.h>

#include <Courier/Base.h>
#include <Courier/Util/Vector.h>

#include <Courier/Visualization/ViewScene2D.h>
#include <Courier/Visualization/ViewScene3D.h>

namespace Courier {
    class ViewHandler;
}

namespace Courier { namespace UnitTest {
    class AnimationInvalidatorWrapper;
}}

namespace Courier {
 /** @addtogroup COURIER_VISUALIZATION
  * @{
*/
/** */
class AnimationInvalidator
{
    public:
        /** Setter enumeration values define the type of property setter.*/
        enum Setter {
            FEATSTD_LINT_CURRENT_SCOPE(1960, "NodePoperty initialization is required and is the only one needed")
            /** */
            FirstSetter,
            // 3D
            /** */
            NodePoperty = FirstSetter,
            /** */
            MorphWeightProperty,
            /** */
            MaterialProperty,
            /** */
            LightProperty,
            /** */
            CameraFovProperty,
            /** */
            CameraProperty,
            /** */
            UniformProperty,
            /** */
            GenericLodCriterionValueProperty,
            /** */
            CanvasRenderable,
            // 2D 
            /** */
            BaseNode2DProperty,
            /** */
            BaseTransformable2DProperty,
            /** */
            BaseEffect2DProperty,
            /** */
            /** */
            Camera2DProperty,
            /** When adding a new property setter type to this enumeration, also the internal mSetterControl must be addapted. */
            LastSetter
        };

    private:
        friend class UnitTest::AnimationInvalidatorWrapper;

        struct AnimationViewRelation {
            AnimationViewRelation(Candera::Animation::AnimationPlayer * player) : mPlayer(player), mInvalidationFallbackDetected(false) {}
            FeatStd::SizeType HitCount() const { return mViewPtrVector.Size(); }
            void Add(View * view) { (void)mViewPtrVector.Add(view); }

            Candera::Animation::AnimationPlayer * mPlayer;
            ViewPtrVector mViewPtrVector;
            bool mInvalidationFallbackDetected;
        };

        typedef Vector< AnimationViewRelation > AnimationViewRelationVector;

        class AnimationViewRelationContainer 
        {
            public:
                AnimationViewRelationContainer() {}

                FeatStd::SizeType GetAnimationCount() const { return mRelationVector.Size(); }

                bool Add(const AnimationViewRelation & relation) { return mRelationVector.Add(relation); }

                bool Release(RenderHint * renderHint, const Candera::Animation::AnimationPlayer * player);

                bool IsDetected(const Candera::Animation::AnimationPlayer * player);

                bool InvalidateViews();

                void Clear();

                void RemoveView(const View* view);

            private:
                friend class UnitTest::AnimationInvalidatorWrapper;
                static bool InvalidateRelationVector(AnimationViewRelationVector& relationVector, ViewPtrVector& invalidationVector);
                static void RemoveView(const View* view, ViewPtrVector& viewVector);
                static void RemoveView(const View* view, AnimationViewRelationVector& relationVector);

                AnimationViewRelationVector mRelationVector;
                AnimationViewRelationVector mFinishedVector;
                ViewPtrVector               mInvalidationVector;
        };

    public:
         /** Default Constructor */
        AnimationInvalidator();

        ///
        ~AnimationInvalidator();

        /** Enables or disables the scene lookup for the given setter type. If disabled specific not used setters this may increase lookup speed.
            @param setter the type of setter.
            @param enable true if lookup shall be done (default=true), or false if lookup shall not be done. */
        static void ControlProcessing(AnimationInvalidator::Setter setter, bool enable);

        /** Enables or disables the invalidation in general. If set to true, all cameras will be invalidated if an animation is running.
            Default is now false..
            @param invalidateAlways true if all cameras shall be invalidated, false if specific ones shall be invalidated. */
        static void ControlInvalidation(bool invalidateAlways) { mInvalidateAlways = invalidateAlways; }

        /** Enables or disables the invalidation fallback. If set to true the fallback will be used, which means if an animation is analyzed but
            no property setters used by Views can be found the return value of Invalidate will be false. This then causes invalidation of all
            cameras inside ViewHandler. When disabled (default) always true is returned by Invalidate and only AnimationInvalidator invalidate Views.
            Default is now false..
            @param useFallback true if fallback shall be used, false if it shall not be used. */
        static void ControlInvalidationFallback(bool useFallback) { mInvalidationFallback = useFallback; }

         /** */
        bool Init(ViewHandler * handler);

         /** */
        void ProcessScenes(RenderHint * renderHint, Candera::Animation::AnimationPlayerBase * player, bool force);

         /** */
        void ReleaseScenes(RenderHint * renderHint, Candera::Animation::AnimationPlayerBase * player);

         /** */
        bool Invalidate(RenderHint * renderHint);

        /** */
        void Clear();

        /** */
        void RemoveView(const View* view);

    private:
        ///
        void Iterate(RenderHint * renderHint,
                     Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationGroupPlayer> group, 
                     Candera::MemoryManagement::SharedPointer<Candera::Animation::AnimationPlayerBase> player,
                     bool process, bool force);
        ///
        void ProcessScenes(RenderHint * renderHint, Candera::Animation::AnimationPlayer * player, bool force);

#if defined(CANDERA_3D_ENABLED)
        ///
        bool ProcessNodePropertySetter(Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        bool ProcessMorphWeightPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        bool ProcessMaterialPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        bool ProcessLightPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        bool ProcessCameraPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        bool ProcessUniformPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        bool ProcessGenericLodCriterionValuePropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        bool ProcessCanvasRenderablePropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        static void TraverseAllViewScene3D(const ViewPtrVector & activeViews, Candera::TreeTraverserBase<Candera::Node> & traverser);
        ///
        AnimationViewRelationContainer m3DContainer;
#endif

#if defined(CANDERA_2D_ENABLED)
        ///
        bool ProcessBaseNode2DPropertySetter(Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        bool ProcessBaseTransformable2DPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        bool ProcessBaseEffect2DPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        bool ProcessBaseCamera2DPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;
        ///
        AnimationViewRelationContainer m2DContainer;
#endif

        bool ProcessAbstractNodePropertySetter(Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const;



        ///
        ViewHandler * mViewHandler;
        ///
        static bool mSetterControl[LastSetter];
        ///
        static bool mInvalidateAlways;
        ///
        static bool mInvalidationFallback;
      
};

 /** @} */
}

#endif
