//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationInvalidator.h"
#include "ViewHandler.h"
#include "RenderHint.h"

#include <Courier/Diagnostics/Log.h>
#include <FeatStd/Util/StaticObject.h>


namespace Courier {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

using namespace Candera;
using namespace Candera::Animation;
using namespace Candera::MemoryManagement;


// ------------------------------------------------------------------------

bool AnimationInvalidator::mSetterControl[LastSetter] = { true, true, true, true, true, true, true, true, true, true, true, true, true};
bool AnimationInvalidator::mInvalidateAlways = false;
bool AnimationInvalidator::mInvalidationFallback = false;

// ------------------------------------------------------------------------

static bool AddAnimatedView(ViewPtrVector& vector, View* view)
{
   return (0 != view) && (!vector.Contains(view)) && vector.Add(view);
}

bool AnimationInvalidator::AnimationViewRelationContainer::Release(RenderHint * renderHint, const Candera::Animation::AnimationPlayer * player) 
{
    SizeType i;
    for (i=0; i<mRelationVector.Size(); ++i) {
        if(mRelationVector[i].mPlayer==player) {
            break;
        }
    }
    if (i<mRelationVector.Size()) {
        FEATSTD_UNUSED(renderHint); // if logging is disabled, variable/parameter is not used
        COURIER_LOG_INFO("ReleaseScenes of AnimationPlayer '%s' %s",player->GetName(),renderHint->ShallRender2D() ? "2D" : "3D");
        bool rc = mFinishedVector.Add(mRelationVector[i]);
        rc = rc && mRelationVector.Remove(i);
        return rc;
    }
    return true;
}

bool AnimationInvalidator::AnimationViewRelationContainer::IsDetected(const Candera::Animation::AnimationPlayer * player) 
{
    for (SizeType i=0; i<mRelationVector.Size(); ++i) {
        if(mRelationVector[i].mPlayer==player) {
            return true;
        }
    }
    return false;
}

void AnimationInvalidator::AnimationViewRelationContainer::Clear()
{
    mInvalidationVector.Clear();
    mRelationVector.Clear();
    mFinishedVector.Clear();
}

void AnimationInvalidator::AnimationViewRelationContainer::RemoveView(const View* view)
{
    RemoveView(view, mInvalidationVector);
    RemoveView(view, mRelationVector);
    RemoveView(view, mFinishedVector);
}

bool AnimationInvalidator::AnimationViewRelationContainer::InvalidateRelationVector(AnimationViewRelationVector& relationVector, ViewPtrVector& invalidationVector)
{
    bool result = true;
    for (SizeType i = 0; i<relationVector.Size(); ++i) {
        AnimationViewRelation& animationViewRelation = relationVector[i];
        ViewPtrVector& viewVector = animationViewRelation.mViewPtrVector;
        if (animationViewRelation.mInvalidationFallbackDetected) {
            // return false this means invalidate all, because we found no suitable property change
            result = false;
        }
        for (SizeType s = 0; s < viewVector.Size(); ++s) {
            View * view = viewVector[s];
            if (view->IsRenderingEnabled() && AddAnimatedView(invalidationVector, view)) {
                COURIER_LOG_DEBUG("InvalidateRelationVector: Invalidate view='%s'", view->GetId().CStr());
                // within the animation context no resulting dirty area information is available.
                // therefore, the complete area has to be invalidated.
                view->Invalidate(FeatStd::Optional<Candera::Rectangle>());
            }
        }
    }
    return result;
}

void AnimationInvalidator::AnimationViewRelationContainer::RemoveView(const View* view, ViewPtrVector& viewVector)
{
    SizeType i = 0;
    while (i < viewVector.Size()) {
        if (view == viewVector[i]) {
            static_cast<void>(viewVector.Remove(i));
        }
        else {
            ++i;
        }
    }
}

void AnimationInvalidator::AnimationViewRelationContainer::RemoveView(const View* view, AnimationViewRelationVector& relationVector)
{
    for (SizeType i = 0; i < relationVector.Size(); ++i) {
        RemoveView(view, relationVector[i].mViewPtrVector);
    }
}

bool AnimationInvalidator::AnimationViewRelationContainer::InvalidateViews() 
{
    mInvalidationVector.Clear();
    bool result = InvalidateRelationVector(mRelationVector, mInvalidationVector);
    if (!InvalidateRelationVector(mFinishedVector, mInvalidationVector)) {
        result = false;
    }
    mFinishedVector.Clear();
    return result;
}

// ------------------------------------------------------------------------
#if defined(CANDERA_3D_ENABLED)

template<typename T>
class SpecificNodeTreeTraverser : public Candera::TreeTraverserBase<Candera::Node>
{
    public:
        SpecificNodeTreeTraverser() :
            mLookFor(0),
            mVector(0),
            mViewHandler(0)
        {
        }

        SpecificNodeTreeTraverser(const T * lookFor, ViewPtrVector* vector, Courier::ViewHandler* viewHandler) :
            mLookFor(lookFor),
            mVector(vector),
            mViewHandler(viewHandler)
        {
        }

    protected:
        virtual TraverserAction ProcessNode(Candera::Node& node)
        {
            if (node.IsTypeOf(T::GetTypeId())) {
                FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
                FEATSTD_LINT_NEXT_EXPRESSION(826,"type already checked")
                if (mLookFor == static_cast<T*>(&node)) {
                    Scene * scene = node.GetScene();
                    if ((0 != scene) && (0 != mVector)) {
                        (void)AddAnimatedView(*mVector, mViewHandler->FindView(ViewId(scene->GetName())));
                    }
                    return Candera::TreeTraverserBase<Candera::Node>::StopTraversing;
                }
            }
            return Candera::TreeTraverserBase<Candera::Node>::ProceedTraversing;
        }

        virtual TraverserAction ProcessConstNode(const Candera::Node& node)
        {
            return ProcessNode(const_cast<Node&>(node));
        }

    private:
        const T * mLookFor;
        Courier::ViewPtrVector* mVector;
        Courier::ViewHandler* mViewHandler;
};

class MaterialTreeTraverser : public Candera::TreeTraverserBase<Candera::Node>
{
    public:
        MaterialTreeTraverser() :
            mLookFor(0),
            mVector(0),
            mViewHandler(0)
        {
        }

        MaterialTreeTraverser(const Material * lookFor, ViewPtrVector* vector, Courier::ViewHandler* viewHandler) :
            mLookFor(lookFor),
            mVector(vector),
            mViewHandler(viewHandler)
        {
        }

    protected:
        virtual TraverserAction ProcessNode(Candera::Node& node)
        {
            Appearance* appearance = node.GetAppearance().GetPointerToSharedInstance();
            if (0 != appearance) {
                Material* material = appearance->GetMaterial().GetPointerToSharedInstance();
                if (mLookFor == material) {
                    Scene* scene = node.GetScene();
                    if ((0 != scene) && (0 != mVector)) {
                        (void)AddAnimatedView(*mVector, mViewHandler->FindView(ViewId(scene->GetName())));
                    }
                    return Candera::TreeTraverserBase<Candera::Node>::StopTraversing;
                }
            }
            return Candera::TreeTraverserBase<Candera::Node>::ProceedTraversing;
        }

        virtual TraverserAction ProcessConstNode(const Candera::Node& node)
        {
            return ProcessNode(const_cast<Node&>(node));
        }

    private:
        const Material * mLookFor;
        Courier::ViewPtrVector* mVector;
        Courier::ViewHandler* mViewHandler;
};

class GenericValueLodCriterionTreeTraverser : public Candera::TreeTraverserBase<Candera::Node>
{
    public:
        GenericValueLodCriterionTreeTraverser() :
            mLookFor(0),
            mVector(0),
            mViewHandler(0)
        {
        }

        GenericValueLodCriterionTreeTraverser(const GenericValueLodCriterion * lookFor, ViewPtrVector* vector, Courier::ViewHandler* viewHandler) :
            mLookFor(lookFor),
            mVector(vector),
            mViewHandler(viewHandler)
        {
        }

    protected:
        virtual TraverserAction ProcessNode(Candera::Node& node)
        {
            if (node.IsTypeOf(LodNode::GetTypeId())) {
                FEATSTD_LINT_NEXT_EXPRESSION(826, "pointer conversion")
                LodNode* lodNode = static_cast<LodNode*>(&node);
                const LodRenderStrategy* strategy = lodNode->GetLodRenderStrategy();
                if ((0 != strategy) && (strategy->GetLodCriterion() == mLookFor)) {
                    Scene* scene = node.GetScene();
                    if ((0 != scene) && (0 != mVector)) {
                        (void)AddAnimatedView(*mVector, mViewHandler->FindView(ViewId(scene->GetName())));
                    }
                    return Candera::TreeTraverserBase<Candera::Node>::StopTraversing;
                }
            }
            return Candera::TreeTraverserBase<Candera::Node>::ProceedTraversing;
        }

        virtual TraverserAction ProcessConstNode(const Candera::Node& node)
        {
            return ProcessNode(const_cast<Node&>(node));
        }

    private:
        const GenericValueLodCriterion * mLookFor;
        Courier::ViewPtrVector* mVector;
        Courier::ViewHandler* mViewHandler;
};

class ShaderParamSetterTreeTraverser : public Candera::TreeTraverserBase<Candera::Node>
{
    public:
        ShaderParamSetterTreeTraverser() :
            mLookFor(0),
            mVector(0),
            mViewHandler(0)
        {
        }
        ShaderParamSetterTreeTraverser(const ShaderParamSetter * lookFor, ViewPtrVector* vector, Courier::ViewHandler* viewHandler) :
            mLookFor(lookFor),
            mVector(vector),
            mViewHandler(viewHandler)
        {
        }

    protected:
        virtual TraverserAction ProcessNode(Candera::Node& node)
        {
            Appearance* appearance = node.GetAppearance().GetPointerToSharedInstance();
            if (0 != appearance) {
                AbstractShaderParamSetter* shaderParam = appearance->GetShaderParamSetter().GetPointerToSharedInstance();
                if (mLookFor == shaderParam) {
                    Scene* scene = node.GetScene();
                    if ((0 !=scene) && (0 != mVector)) {
                        (void)AddAnimatedView(*mVector, mViewHandler->FindView(ViewId(scene->GetName())));
                    }
                    return Candera::TreeTraverserBase<Candera::Node>::StopTraversing;
                }
            }
            return Candera::TreeTraverserBase<Candera::Node>::ProceedTraversing;
        }

        virtual TraverserAction ProcessConstNode(const Candera::Node& node)
        {
            return ProcessNode(const_cast<Node&>(node));
        }

    private:
        const ShaderParamSetter * mLookFor;
        Courier::ViewPtrVector * mVector;
        Courier::ViewHandler* mViewHandler;
};

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessNodePropertySetter(Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    if (mViewHandler != 0) {
        Node* node = 0;
        if (setter->IsTypeOf(AlphaNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(NodeAppearancePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(RenderingEnabledNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(RotateNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(RotateXNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(RotateYNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(RotateZNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(ScaleNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(ScaleXNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(ScaleYNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(ScaleZNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(TranslateNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(TranslateXNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(TranslateYNodePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(TranslateZNodePropertySetter::GetTypeId())
            ) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774, "type already checked")
                NodePropertySetter * npSetter = static_cast<NodePropertySetter*>(setter);
            node = npSetter->GetNode();
        }
        else if (setter->IsTypeOf(RotateNodeRelativePropertySetter::GetTypeId()) ||
                 setter->IsTypeOf(RotateXNodeRelativePropertySetter::GetTypeId()) ||
                 setter->IsTypeOf(RotateYNodeRelativePropertySetter::GetTypeId()) ||
                 setter->IsTypeOf(RotateZNodeRelativePropertySetter::GetTypeId()) ||
                 setter->IsTypeOf(ScaleNodeRelativePropertySetter::GetTypeId()) ||
                 setter->IsTypeOf(ScaleXNodeRelativePropertySetter::GetTypeId()) ||
                 setter->IsTypeOf(ScaleYNodeRelativePropertySetter::GetTypeId()) ||
                 setter->IsTypeOf(ScaleZNodeRelativePropertySetter::GetTypeId()) ||
                 setter->IsTypeOf(TranslateNodeRelativePropertySetter::GetTypeId()) ||
                 setter->IsTypeOf(TranslateXNodeRelativePropertySetter::GetTypeId()) ||
                 setter->IsTypeOf(TranslateYNodeRelativePropertySetter::GetTypeId()) ||
                 setter->IsTypeOf(TranslateZNodeRelativePropertySetter::GetTypeId())
                 ) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774, "type already checked")
                NodeRelativePropertySetter * npSetter = static_cast<NodeRelativePropertySetter*>(setter);
            node = npSetter->GetNode();
        }
        else {
            // MISRA
        }
        if (0 != node) {
            Scene* scene = node->GetScene();
            if (0 != scene) {
                (void)AddAnimatedView(vector, mViewHandler->FindView(ViewId(scene->GetName())));
            }
            return true;
        }
        if (ProcessAbstractNodePropertySetter(setter, vector)) {
            return true;
        }
    }

    return false;
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessMorphWeightPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    bool processed = false;
    if(mViewHandler!=0) {
        const MorphingMesh * mesh = 0;
        if(setter->IsTypeOf(MorphWeightPropertySetter::GetTypeId())) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
            MorphWeightPropertySetter * mwpSetter = static_cast<MorphWeightPropertySetter*>(setter);
            mesh = mwpSetter->GetMorphingMesh();
            processed = true;
        } else if(setter->IsTypeOf(AllMorphWeightsPropertySetter::GetTypeId())) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
            AllMorphWeightsPropertySetter * amwpSetter = static_cast<AllMorphWeightsPropertySetter*>(setter);
            mesh = amwpSetter->GetMorphingMesh();
            processed = true;
        } else {
            // MISRA
        }
        if(mesh!=0) {
            SpecificNodeTreeTraverser<Mesh> traverser(mesh, &vector, mViewHandler);
            TraverseAllViewScene3D(activeViews,traverser);
        }
    }
    return processed;
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessMaterialPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    if(mViewHandler!=0) {
        if(setter->IsTypeOf(MaterialAmbientBPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialAmbientGPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialAmbientPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialAmbientRPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialDiffuseBPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialDiffuseGPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialDiffusePropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialDiffuseRPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialDiffuseAPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialEmissiveBPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialEmissiveGPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialEmissivePropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialEmissiveRPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialSpecularBPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialSpecularGPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialSpecularPowerPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialSpecularPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(MaterialSpecularRPropertySetter::GetTypeId())
        ) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
            MaterialPropertySetter * materialSetter = static_cast<MaterialPropertySetter*>(setter);
            const Material * material = materialSetter->GetMaterial().GetPointerToSharedInstance();
            if(material!=0) {
                MaterialTreeTraverser traverser(material, &vector, mViewHandler);
                TraverseAllViewScene3D(activeViews,traverser);
            }
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessLightPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    if(mViewHandler!=0) {
        if(setter->IsTypeOf(AmbientBLightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(AmbientGLightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(AmbientLightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(AmbientRLightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(DiffuseBLightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(DiffuseGLightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(DiffuseLightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(DiffuseRLightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(SpecularBLightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(SpecularGLightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(SpecularLightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(SpecularRLightPropertySetter::GetTypeId())
        ) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
            LightPropertySetter * lightSetter = static_cast<LightPropertySetter*>(setter);
            const Light * light = lightSetter->GetLight();
            if(light!=0) {
                SpecificNodeTreeTraverser<Light> traverser(light, &vector, mViewHandler);
                TraverseAllViewScene3D(activeViews,traverser);
            }
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessCameraPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    bool process = false;
    const Camera * camera = 0;
    if(mViewHandler!=0) {
        if(setter->IsTypeOf(CameraFovPropertySetter::GetTypeId())) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
            CameraFovPropertySetter * cameraSetter = static_cast<CameraFovPropertySetter*>(setter);
            camera = cameraSetter->GetCamera();
            process = true;
        } else if (
            (setter->IsTypeOf(CameraViewportPropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CameraViewportLeftPropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CameraViewportTopPropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CameraViewportWidthPropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CameraViewportHeightPropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CameraScissorRectanglePropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CameraScissorRectangleLeftPropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CameraScissorRectangleTopPropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CameraScissorRectangleWidthPropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CameraScissorRectangleHeightPropertySetter::GetTypeId()))
            ) {
            BaseCameraPropertySetter * cameraSetter = static_cast<BaseCameraPropertySetter*>(setter);
            camera = cameraSetter->GetCamera();
            process = true;
        }
        else {
            // do nothing
        }

        if (camera != 0) {
            SpecificNodeTreeTraverser<Camera> traverser(camera, &vector, mViewHandler);
            TraverseAllViewScene3D(activeViews, traverser);
        }
    }
    return process;
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessGenericLodCriterionValuePropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    if(mViewHandler!=0) {
        if(setter->IsTypeOf(GenericLodCriterionValuePropertySetter::GetTypeId())) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
            GenericLodCriterionValuePropertySetter * valueSetter = static_cast<GenericLodCriterionValuePropertySetter*>(setter);
            const GenericValueLodCriterion * criterion = valueSetter->GetGenericValueLodCriterion();
            if(criterion!=0) {
                GenericValueLodCriterionTreeTraverser traverser(criterion, &vector, mViewHandler);
                TraverseAllViewScene3D(activeViews,traverser);
            }
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessCanvasRenderablePropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    if (mViewHandler != 0) {
        if ((setter->IsTypeOf(CanvasVertexRectanglePropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CanvasVertexRectangleLeftPropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CanvasVertexRectangleTopPropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CanvasVertexRectangleWidthPropertySetter::GetTypeId())) ||
            (setter->IsTypeOf(CanvasVertexRectangleHeightPropertySetter::GetTypeId()))
            ) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774, "type already checked")
                BaseCanvasRenderablePropertySetter * valueSetter = static_cast<BaseCanvasRenderablePropertySetter*>(setter);
            const Candera::CanvasRenderable* canvasRenderable = valueSetter->GetCanvasRenderable();
            if (canvasRenderable != 0) {
                SpecificNodeTreeTraverser<Candera::CanvasRenderable> traverser(canvasRenderable, &vector, mViewHandler);
                TraverseAllViewScene3D(activeViews, traverser);
            }
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessUniformPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    if(mViewHandler!=0) {
        if(setter->IsTypeOf(BoolUniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(BoolVec2UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(BoolVec3UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(BoolVec4UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(FloatMat2UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(FloatMat3UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(FloatMat4UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(FloatUniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(FloatVec2UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(FloatVec3UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(FloatVec4UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(IntUniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(IntVec2UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(IntVec3UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(IntVec4UniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Sampler2DUniformPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(SamplerCubeUniformPropertySetter::GetTypeId())
        ) {
            UniformPropertySetter * uniformSetter = static_cast<UniformPropertySetter*>(setter);
            const ShaderParamSetter * shaderParamSetter = uniformSetter->GetShaderParamSetter().GetPointerToSharedInstance();
            if(shaderParamSetter!=0) {
                ShaderParamSetterTreeTraverser traverser(shaderParamSetter, &vector, mViewHandler);
                TraverseAllViewScene3D(activeViews,traverser);
            }
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
void AnimationInvalidator::TraverseAllViewScene3D(const ViewPtrVector & activeViews, Candera::TreeTraverserBase<Candera::Node> & traverser)
{
    for (SizeType i = 0; i < activeViews.Size(); ++i) {
        ViewScene3D * view3D = activeViews[i]->ToViewScene3D();
        if (0 != view3D) {
            Candera::SceneContext * context = view3D->GetScene3DContext();
            if (0 != context) {
                Candera::Scene * scene3D = context->GetScene();
                if (0 != scene3D) {
                    traverser.Traverse(*scene3D);
                }
            }
        }
    }
}

#endif

// ------------------------------------------------------------------------
#if defined(CANDERA_2D_ENABLED)

class Effect2DTreeTraverser : public Candera::TreeTraverserBase<Candera::Node2D>
{
    public:
        Effect2DTreeTraverser() :
            mLookFor(0),
            mVector(0),
            mViewHandler(0)
        {
        }

        Effect2DTreeTraverser(const Effect2D* lookFor, Courier::ViewPtrVector* vector, Courier::ViewHandler* viewHandler) :
            mLookFor(lookFor),
            mVector(vector),
            mViewHandler(viewHandler)
        {
        }

    protected:
        virtual TraverserAction ProcessNode(Candera::Node2D & node)
        {
            if (node.IsTypeOf(RenderNode::GetTypeId())) {
                FEATSTD_LINT_NEXT_EXPRESSION(826, "ignore pointer warning")
                RenderNode * renderNode = static_cast<RenderNode*>(&node);
                UInt8 idx = 0;
                Effect2D* effect2D = renderNode->GetEffect(idx);
                while (0 != effect2D) {
                    if (effect2D == mLookFor) {
                        Scene2D* scene = node.GetScene();
                        if ((0 != scene) && (0 != mVector)) {
                            (void)AddAnimatedView(*mVector, mViewHandler->FindView(ViewId(scene->GetName())));
                        }
                        return Candera::TreeTraverserBase<Candera::Node2D>::StopTraversing;
                    }
                    idx++;
                    effect2D = renderNode->GetEffect(idx);
                }
            }
            return Candera::TreeTraverserBase<Candera::Node2D>::ProceedTraversing;
        }

        virtual TraverserAction ProcessConstNode(const Candera::Node2D & node)
        {
            return ProcessNode(const_cast<Node2D &>(node));
        }

    private:
        const Effect2D * mLookFor;
        Courier::ViewPtrVector* mVector;
        Courier::ViewHandler* mViewHandler;
};

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessBaseNode2DPropertySetter(Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    if(mViewHandler!=0) {
        if(setter->IsTypeOf(Candera::Animation::AlphaNode2DPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::RenderingEnabledNode2DPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::SnapToDevicePixelEnabled2DPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Node2DBoundingRectanglePropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Node2DBoundingRectangleLeftPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Node2DBoundingRectangleTopPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Node2DBoundingRectangleWidthPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Node2DBoundingRectangleHeightPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Node2DClippingRectanglePropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Node2DClippingRectangleLeftPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Node2DClippingRectangleTopPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Node2DClippingRectangleWidthPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Node2DClippingRectangleHeightPropertySetter::GetTypeId())
        ) {
            Candera::Animation::BaseNode2DPropertySetter * npSetter = static_cast<Candera::Animation::BaseNode2DPropertySetter*>(setter);
            Node2D * node = npSetter->GetNode();
            if (0 != node) {
                Scene2D * scene = node->GetScene();
                if (0 != scene) {
                    (void)AddAnimatedView(vector, mViewHandler->FindView(ViewId(scene->GetName())));
                }
            }
            return true;
        }
        if (ProcessAbstractNodePropertySetter(setter, vector)) {
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessBaseTransformable2DPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    bool processed = false;
    if (0 != mViewHandler) {
        const Transformable2D * transformable2D = 0;
        if (setter->IsTypeOf(Candera::Animation::Transformable2DRotatePropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DScalePropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DScaleXPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DScaleYPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DTranslatePropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DTranslateXPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DTranslateYPropertySetter::GetTypeId())
        ) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
            Candera::Animation::BaseTransformable2DPropertySetter * trans2dSetter = static_cast<Candera::Animation::BaseTransformable2DPropertySetter*>(setter);
            transformable2D = trans2dSetter->GetTransformable();
            processed = true;
        }
        else if (setter->IsTypeOf(Candera::Animation::Transformable2DRelativeRotatePropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DRelativeScalePropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DRelativeScaleXPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DRelativeScaleYPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DRelativeTranslatePropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DRelativeTranslateXPropertySetter::GetTypeId()) ||
           setter->IsTypeOf(Candera::Animation::Transformable2DRelativeTranslateYPropertySetter::GetTypeId())
        ) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
            Candera::Animation::BaseTransformable2DRelativePropertySetter * trans2dSetter = static_cast<Candera::Animation::BaseTransformable2DRelativePropertySetter*>(setter);
            transformable2D = trans2dSetter->GetTransformable();
            processed = true;
        }
        else {
            // MISRA
        }

        if (0 != transformable2D) {
            const Node2D* node2d = Dynamic_Cast<const Node2D*>(transformable2D);
            if (0 != node2d) {
                Scene2D* scene2D = node2d->GetScene();
                if (0 != scene2D) {
                    for (SizeType i = 0; i< activeViews.Size(); ++i) {
                        ViewScene2D * view2D = activeViews[i]->ToViewScene2D();
                        Candera::Scene2DContext * context = view2D->GetScene2DContext();
                        if ((0 != context) && (context->GetScene() == scene2D)) {
                            (void)AddAnimatedView(vector, view2D);
                            return processed;
                        }
                    }
                }
            }
        }
    }
    return processed;
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessBaseEffect2DPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    if (0 != mViewHandler) {
        Candera::Internal::ObjectPropertySetter<Effect2D>* effectPropertySetter = Dynamic_Cast<Candera::Internal::ObjectPropertySetter<Effect2D>* >(setter);
        if (0 != effectPropertySetter) {
            Effect2D* effect2D = effectPropertySetter->GetAnimatedObject();
            Effect2DTreeTraverser traverser(effect2D, &vector, mViewHandler);
            for (SizeType i = 0; i < activeViews.Size(); ++i) {
                ViewScene2D * view2D = activeViews[i]->ToViewScene2D();
                Candera::Scene2DContext* context = view2D->GetScene2DContext();
                if (0 != context) {
                    Candera::Scene2D* scene2D = context->GetScene();
                    if (0 != scene2D) {
                        traverser.Traverse(*scene2D);
                    }
                }
            }
            return true;
        }
    }
    return false;
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessBaseCamera2DPropertySetter(const ViewPtrVector & activeViews, Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
    if (0 != mViewHandler) {
        if (setter->IsTypeOf(Candera::Animation::Camera2DScissorRectanglePropertySetter::GetTypeId()) ||
            setter->IsTypeOf(Candera::Animation::Camera2DScissorRectangleLeftPropertySetter::GetTypeId()) ||
            setter->IsTypeOf(Candera::Animation::Camera2DScissorRectangleTopPropertySetter::GetTypeId()) ||
            setter->IsTypeOf(Candera::Animation::Camera2DScissorRectangleWidthPropertySetter::GetTypeId()) ||
            setter->IsTypeOf(Candera::Animation::Camera2DScissorRectangleHeightPropertySetter::GetTypeId()) ||
            setter->IsTypeOf(Candera::Animation::Camera2DViewportPropertySetter::GetTypeId()) ||
            setter->IsTypeOf(Candera::Animation::Camera2DViewportLeftPropertySetter::GetTypeId()) ||
            setter->IsTypeOf(Candera::Animation::Camera2DViewportTopPropertySetter::GetTypeId()) ||
            setter->IsTypeOf(Candera::Animation::Camera2DViewportWidthPropertySetter::GetTypeId()) ||
            setter->IsTypeOf(Candera::Animation::Camera2DViewportHeightPropertySetter::GetTypeId())
            ) {
            BaseCamera2DPropertySetter* cameraPropertySetter = Dynamic_Cast<BaseCamera2DPropertySetter* >(setter);
            if (0 != cameraPropertySetter) {
                Camera2D* camera2D = cameraPropertySetter->GetCamera();
                if (0 != camera2D) {
                    const Node2D* node2d = Dynamic_Cast<const Node2D*>(camera2D);
                    if (0 != node2d) {
                        Scene2D* scene2D = node2d->GetScene();
                        if (0 != scene2D) {
                            for (SizeType i = 0; i < activeViews.Size(); ++i) {
                                ViewScene2D * view2D = activeViews[i]->ToViewScene2D();
                                Candera::Scene2DContext * context = view2D->GetScene2DContext();
                                if ((0 != context) && (context->GetScene() == scene2D)) {
                                    (void)AddAnimatedView(vector, view2D);
                                }
                            }
                        }
                    }
                }
            }
            return true;
        }
    }
    return false;
}

#endif


// ------------------------------------------------------------------------
bool AnimationInvalidator::ProcessAbstractNodePropertySetter(Candera::Animation::AnimationPropertySetter * setter, ViewPtrVector & vector) const
{
#if defined(CANDERA_LAYOUT_ENABLED)
    if (
        setter->IsTypeOf(Candera::Animation::LayoutMarginsPropertySetter::GetTypeId()) ||
        setter->IsTypeOf(Candera::Animation::LayoutMarginsLeftPropertySetter::GetTypeId()) ||
        setter->IsTypeOf(Candera::Animation::LayoutMarginsTopPropertySetter::GetTypeId()) ||
        setter->IsTypeOf(Candera::Animation::LayoutMarginsRightPropertySetter::GetTypeId()) ||
        setter->IsTypeOf(Candera::Animation::LayoutMarginsBottomPropertySetter::GetTypeId())
        ) {
        Candera::Animation::AbstractNodePropertySetter * npSetter = static_cast<Candera::Animation::AbstractNodePropertySetter*>(setter);
        AbstractNodePointer& node = npSetter->GetNode();
        if (node.IsValid()) {
#if defined(CANDERA_2D_ENABLED)
            if (node.GetType() == AbstractNodePointer::Candera2D) {
                Scene2D * scene = node.ToNode2D()->GetScene();
                if (0 != scene) {
                    (void)AddAnimatedView(vector, mViewHandler->FindView(ViewId(scene->GetName())));
                }

            }
#endif
#if defined(CANDERA_3D_ENABLED)
#if defined(CANDERA_2D_ENABLED)
            else
#endif
            if (node.GetType() == AbstractNodePointer::Candera3D) {
                Scene * scene = node.ToNode()->GetScene();
                if (0 != scene) {
                    (void)AddAnimatedView(vector, mViewHandler->FindView(ViewId(scene->GetName())));
                }
            }
#endif
#if (defined(CANDERA_2D_ENABLED)) || (defined(CANDERA_3D_ENABLED))
            else
#endif
            {
                return false;
            }

        }
        return true;
    }
#else
    FEATSTD_UNUSED2(setter, vector);
#endif
    return false;

}

// ------------------------------------------------------------------------
AnimationInvalidator::AnimationInvalidator() : mViewHandler(0)
{
}

// ------------------------------------------------------------------------
AnimationInvalidator::~AnimationInvalidator()
{
    mViewHandler = 0;
}

// ------------------------------------------------------------------------
void AnimationInvalidator::ControlProcessing(AnimationInvalidator::Setter setter, bool enable)
{
    if((setter>=FirstSetter) && (setter<LastSetter)) {
        mSetterControl[setter] = enable;
    }
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::Init(ViewHandler * handler)
{
    FEATSTD_DEBUG_ASSERT(handler!=0);

    if(handler==0) {
        COURIER_LOG_ERROR("ViewHandler not set.");
        return false;
    }

    mViewHandler = handler;
    return true;
}

// ------------------------------------------------------------------------
void AnimationInvalidator::Iterate(RenderHint * renderHint, SharedPointer<Candera::Animation::AnimationGroupPlayer> group, SharedPointer<Candera::Animation::AnimationPlayerBase> player, bool process, bool force)
{
    if (player != 0) {
        Iterate(renderHint, group, group->GetFirstSuccessor(player, Candera::Animation::AnimationGroupPlayer::WithPrevious),process, force);
        Iterate(renderHint, group, group->GetFirstSuccessor(player, Candera::Animation::AnimationGroupPlayer::AfterPrevious),process, force);
        Iterate(renderHint, group, group->GetNext(player),process, force);
        if(player->IsTypeOf(Candera::Animation::AnimationPlayer::GetTypeId())) {
            FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
            Candera::Animation::AnimationPlayer * ap = static_cast<Candera::Animation::AnimationPlayer*>(player.GetPointerToSharedInstance());
            if(process) {
                ProcessScenes(renderHint, ap, force);
            } else {
                ReleaseScenes(renderHint, ap);
            }
        }
    }
}

// ------------------------------------------------------------------------
void AnimationInvalidator::ProcessScenes(RenderHint * renderHint, Candera::Animation::AnimationPlayerBase * player, bool force)
{
    if(mInvalidateAlways) {
        // do nothing
        return;
    }
    if(player->IsTypeOf(Candera::Animation::AnimationPlayer::GetTypeId())) {
        FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
        ProcessScenes(renderHint, static_cast<Candera::Animation::AnimationPlayer*>(player), force);
    } else if(player->IsTypeOf(Candera::Animation::AnimationGroupPlayer::GetTypeId())) {
        FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
        Candera::Animation::AnimationGroupPlayer * groupPlayer = static_cast<Candera::Animation::AnimationGroupPlayer*>(player);
        SharedPointer<Candera::Animation::AnimationGroupPlayer> group(groupPlayer);
        Iterate(renderHint, group, group->GetFirstSuccessor(SharedPointer<Candera::Animation::AnimationPlayer>(0), Candera::Animation::AnimationGroupPlayer::WithPrevious),true, force);
    } else {
        // MISRA
    }
}

// ------------------------------------------------------------------------
void AnimationInvalidator::ReleaseScenes(RenderHint * renderHint, Candera::Animation::AnimationPlayerBase * player)
{
    if(mInvalidateAlways) {
        return;
    }
    if(player->IsTypeOf(Candera::Animation::AnimationPlayer::GetTypeId())) {
        FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
        Candera::Animation::AnimationPlayer * ap = static_cast<Candera::Animation::AnimationPlayer*>(player);
#if defined(CANDERA_2D_ENABLED)
        if(renderHint->ShallRender2D()) {
            (void)m2DContainer.Release(renderHint, ap);
        }
#endif
#if defined(CANDERA_3D_ENABLED)
        if(renderHint->ShallRender3D()) {
            (void)m3DContainer.Release(renderHint, ap);
        }
#endif
    } else if(player->IsTypeOf(Candera::Animation::AnimationGroupPlayer::GetTypeId())) {
        FEATSTD_LINT_NEXT_EXPRESSION(1774,"type already checked")
        Candera::Animation::AnimationGroupPlayer * groupPlayer = static_cast<Candera::Animation::AnimationGroupPlayer*>(player);
        SharedPointer<Candera::Animation::AnimationGroupPlayer> group(groupPlayer);
        Iterate(renderHint, group, group->GetFirstSuccessor(SharedPointer<Candera::Animation::AnimationPlayer>(0), Candera::Animation::AnimationGroupPlayer::WithPrevious),false, false);
    } else {
        // MISRA
    }
}

// ------------------------------------------------------------------------
void AnimationInvalidator::ProcessScenes(RenderHint * renderHint, Candera::Animation::AnimationPlayer * player, bool force)
{
    bool isEnabled = player->IsEnabled() || player->IsPaused();
#if defined(CANDERA_2D_ENABLED)
    bool isDetected2D = m2DContainer.IsDetected(player);
#endif
#if defined(CANDERA_3D_ENABLED)
    bool isDetected3D = m3DContainer.IsDetected(player);
#endif
    if (!isEnabled) {
#if defined(CANDERA_2D_ENABLED)
        if (isDetected2D) {
            FEATSTD_UNSYNCED_STATIC_OBJECT(RenderHint2D, hint);
            ReleaseScenes(&hint, player);
        }
#endif
#if defined(CANDERA_3D_ENABLED)
        if (isDetected3D) {
            FEATSTD_UNSYNCED_STATIC_OBJECT(RenderHint3D, hint);    
            ReleaseScenes(&hint, player);
        }
#endif
        return;
    }
#if defined(CANDERA_2D_ENABLED)
    if (isDetected2D && (!force)) {
        return;
    }
#endif
#if defined(CANDERA_3D_ENABLED)
    if(isDetected3D && (!force)) {
        return;
    }
#endif

    // no we have not in list then proceed
    Candera::Animation::AnimationController::SharedPointer controller = player->GetController();
    if((mViewHandler!=0) && (controller!=0)) {
#if defined(CANDERA_2D_ENABLED)
        FEATSTD_UNSYNCED_STATIC_OBJECT(ViewPtrVector, active2DViews);
        mViewHandler->GetViewAccessor().GetEnabledViewSceneVector(true,active2DViews);
        AnimationViewRelation relation2D(player);
#endif
#if defined(CANDERA_3D_ENABLED)
        FEATSTD_UNSYNCED_STATIC_OBJECT(ViewPtrVector, active3DViews);
        mViewHandler->GetViewAccessor().GetEnabledViewSceneVector(false,active3DViews);
        AnimationViewRelation relation3D(player);
#endif
        FeatStd::SizeType no = controller->GetNumberOfProperties();
        for (FeatStd::SizeType i=0; i < no; ++i) {
            bool processed = false;
            Candera::Animation::AnimationBlendedProperty::SharedPointer prop = controller->GetProperty(i);
            Candera::Animation::AnimationPropertySetter::SharedPointer setterPtr = prop->GetAnimationPropertySetter();
            Candera::Animation::AnimationPropertySetter * setter = setterPtr.GetPointerToSharedInstance();
            Candera::Internal::ObjectPropertySetter<WidgetBase>* widgetPropertySetter = Dynamic_Cast<Candera::Internal::ObjectPropertySetter<WidgetBase>* >(setter);
            if (0 != widgetPropertySetter) {
                WidgetBase* widget = widgetPropertySetter->GetAnimatedObject();
                if (0 != widget) {
                    View* view = widget->GetParentView();
#if defined(CANDERA_2D_ENABLED)
                    if (view->Is2D()) {
                        (void)AddAnimatedView(relation2D.mViewPtrVector, view);
                    }
#endif
#if defined(CANDERA_3D_ENABLED)
                    if (view->Is3D()) {
                        (void)AddAnimatedView(relation3D.mViewPtrVector, view);
                    }
#endif
                }
                processed = true;
            }
            FEATSTD_LINT_NEXT_EXPRESSION(774, "always evaluates to true, ignore it because of #defines")
#if defined(CANDERA_2D_ENABLED)
            bool processed2D = false;
            if (renderHint->ShallRender2D()) {
                if (mSetterControl[AnimationInvalidator::BaseNode2DProperty]) {
                    processed2D = ProcessBaseNode2DPropertySetter(setter, relation2D.mViewPtrVector);
                }
                if ((!processed2D) && mSetterControl[AnimationInvalidator::BaseTransformable2DProperty]) {
                    processed2D = ProcessBaseTransformable2DPropertySetter(active2DViews, setter, relation2D.mViewPtrVector);
                }
                if ((!processed2D) && mSetterControl[AnimationInvalidator::BaseEffect2DProperty]) {
                    processed2D = ProcessBaseEffect2DPropertySetter(active2DViews, setter, relation2D.mViewPtrVector);
                }

                if ((!processed2D) && mSetterControl[AnimationInvalidator::Camera2DProperty]) {
                    processed2D = ProcessBaseCamera2DPropertySetter(active2DViews, setter, relation2D.mViewPtrVector);
                }
                
                if (processed2D) {
                    processed = true;
                }
            }
#endif
#if defined(CANDERA_3D_ENABLED)
            if (renderHint->ShallRender3D()) {
                if ((!processed) && mSetterControl[AnimationInvalidator::NodePoperty]) {
                    processed = ProcessNodePropertySetter(setter, relation3D.mViewPtrVector);
                }
                if ((!processed) && mSetterControl[AnimationInvalidator::MorphWeightProperty]) {
                    processed = ProcessMorphWeightPropertySetter(active3DViews, setter, relation3D.mViewPtrVector);
                }
                if ((!processed) && mSetterControl[AnimationInvalidator::MaterialProperty]) {
                    processed = ProcessMaterialPropertySetter(active3DViews, setter, relation3D.mViewPtrVector);
                }
                if ((!processed) && mSetterControl[AnimationInvalidator::LightProperty]) {
                    processed = ProcessLightPropertySetter(active3DViews, setter,  relation3D.mViewPtrVector);
                }
                if ((!processed) && mSetterControl[AnimationInvalidator::CameraFovProperty]) {
                    processed = ProcessCameraPropertySetter(active3DViews, setter, relation3D.mViewPtrVector);
                }
                if ((!processed) && mSetterControl[AnimationInvalidator::CameraProperty]) {
                    processed = ProcessCameraPropertySetter(active3DViews, setter, relation3D.mViewPtrVector);
                }
                if ((!processed) && mSetterControl[AnimationInvalidator::UniformProperty]) {
                    processed = ProcessUniformPropertySetter(active3DViews, setter, relation3D.mViewPtrVector);
                }
                if ((!processed) && mSetterControl[AnimationInvalidator::GenericLodCriterionValueProperty]) {
                    processed = ProcessGenericLodCriterionValuePropertySetter(active3DViews, setter, relation3D.mViewPtrVector);
                }
                if ((!processed) && mSetterControl[AnimationInvalidator::CanvasRenderable]) {
                    processed = ProcessCanvasRenderablePropertySetter(active3DViews, setter, relation3D.mViewPtrVector);
                }
                if (!processed) {
                    relation3D.mInvalidationFallbackDetected = true;
                }
            }
#endif
#if defined(CANDERA_2D_ENABLED)
            if (renderHint->ShallRender2D() && (!processed)) {
                relation2D.mInvalidationFallbackDetected = true;
        }
#endif
        }
#if defined(CANDERA_2D_ENABLED)
        if (renderHint->ShallRender2D()) {
            (void)m2DContainer.Add(relation2D);
            COURIER_LOG_INFO("ProcessScenes of AnimationPlayer '%s' 2D Hits(%d) #Players(%d)",player->GetName(),relation2D.HitCount(),m2DContainer.GetAnimationCount());
        }
#endif
#if defined(CANDERA_3D_ENABLED)
        if (renderHint->ShallRender3D()) {
            (void)m3DContainer.Add(relation3D);
            COURIER_LOG_INFO("ProcessScenes of AnimationPlayer '%s' 3D Hits(%d) #Players(%d)",player->GetName(),relation3D.HitCount(),m3DContainer.GetAnimationCount());
        }
#endif
    }
}

// ------------------------------------------------------------------------
bool AnimationInvalidator::Invalidate(RenderHint * renderHint)
{
    if(mInvalidateAlways) {
        // tell that all scenes shall be invalidated
        return false;
    }
    bool result = true;
#if defined(CANDERA_2D_ENABLED)
    if(renderHint->ShallRender2D()) {
        if (!m2DContainer.InvalidateViews()) {
            result = false;
        }
    }
#endif
#if defined(CANDERA_3D_ENABLED)
    if(renderHint->ShallRender3D()) {
        if (!m3DContainer.InvalidateViews()) {
            result = false;
        }
    }
#endif
    return (!mInvalidationFallback) || result;
}

void AnimationInvalidator::Clear()
{
#if defined(CANDERA_2D_ENABLED)
    m2DContainer.Clear();
#endif
#if defined(CANDERA_3D_ENABLED)
    m3DContainer.Clear();
#endif
}

void AnimationInvalidator::RemoveView(const View* view)
{
    if (0 != view) {
#if defined(CANDERA_2D_ENABLED)
        m2DContainer.RemoveView(view);
#endif
#if defined(CANDERA_3D_ENABLED)
        m3DContainer.RemoveView(view);
#endif
    }
}

}
