//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Util_TemplateOperations_h)
#define Courier_Util_TemplateOperations_h

#include <Courier/Util/Traits.h>
#include <FeatStd/MemoryManagement/TypeTraits.h>
#include <FeatStd/Platform/Memory.h>

/// @addtogroup COURIER_UTIL
/// @{
namespace Courier { namespace Internal {
    namespace UtilPrivate {
        template<typename T, bool isPodType> struct Operations {
            static inline void Copy(T *dst, const T *src, FeatStd::SizeType nItems) {
                FEATSTD_DEBUG_ASSERT((src != 0) && (dst != 0) && (((dst + nItems) <= src) || (dst >= (src + nItems))));
                if ((src != 0) && (dst != 0) && (((dst + nItems) <= src) || (dst >= (src + nItems)))) {
                    FeatStd::Internal::Memory::Copy(dst, src, (sizeof(T[2]) / 2) * nItems);
                }
            }

            static inline void Move(T *dst, const T *src, FeatStd::SizeType nItems) {
                FEATSTD_DEBUG_ASSERT((src != 0) && (dst != 0));
                if ((src != 0) && (dst != 0)) {
                    FeatStd::Internal::Memory::Move(dst, src, (sizeof(T[2]) / 2) * nItems);
                }
            }
        };

        template<typename T> struct Operations<T, false> {
            static void Copy(T *dst, const T *src, FeatStd::SizeType nItems) {
                FEATSTD_DEBUG_ASSERT((src != 0) && (dst != 0) && (((dst + nItems) <= src) || (dst >= (src + nItems))));
                if ((src != 0) && (dst != 0)) {
                    T *last = dst + nItems;
                    while (dst != last) {
                        *dst = *src;
                        ++src;
                        ++dst;
                    }
                }
            }

            static void Move(T *dst, const T *src, FeatStd::SizeType nItems) {
                FEATSTD_DEBUG_ASSERT((src != 0) && (dst != 0));
                if ((src != 0) && (dst != 0)) {
                    // check if regular copy
                    if ((dst <= src) || (dst >= (src + nItems))) {
                        // copy first to last
                        T *last = dst + nItems;
                        while (dst != last) {
                            *dst = *src;
                            ++src;
                            ++dst;
                        }
                    }
                    else {
                        // copy last to first
                        T *last = dst;
                        src += nItems;
                        dst += nItems;
                        while (dst != last) {
                            --dst;
                            --src;
                            *dst = *src;
                        }
                    }
                }
            }
        };
    }   // namespace UtilPrivate

    /**
        <summary>
            Copies given number of items from src to dst If T is a POD (plain old data) type, copy operations
            will be done with copy. Otherwise element by element will be copied.

            \note The memory areas to copy must not overlapp.
        </summary>
        <param name="dst">destination of the copy operation.</param>
        <param name="src">Source of the copy operation.</param>
        <param name="nItems">The number of items to copy.</param>
     */
    template<typename T> inline void Copy(T *dst, const T *src, FeatStd::SizeType nItems) {
        typedef typename QualifierRemover<T>::Type Type;
        UtilPrivate::Operations<Type, FeatStd::TypeTraits::Internal::TypeTrait<Type>::IsPodType>::Copy(dst, src, nItems);
    }

    /**
        <summary>
            Moves the given number of items. If T is a POD (plain old data) type, Move will be used for the
            move operation. Otherwise element by element will be moved.
        </summary>
        <param name="dst">Destination of the move operation.</param>
        <param name="src">Source of the move operation.</param>
        <param name="nItems">The number of items to move.</param>
     */
    template<typename T> inline void Move(T *dst, const T *src, FeatStd::SizeType nItems) {
        typedef typename QualifierRemover<T>::Type Type;
        UtilPrivate::Operations<Type, FeatStd::TypeTraits::Internal::TypeTrait<Type>::IsPodType>::Move(dst, src, nItems);
    }

    /**
        <summary>Inserts in a buffer with the given number of items an item at the given index.</summary>
        <param name="index">insertion index.</param>
        <param name="value">the value which the item at the insertion index shall receive.</param>
        <param name="buffer">Pointer to the buffer.</param>
        <param name="bufferItemCount">Number of buffer items.</param>
        <returns>true if it succeeds, false if it fails.</returns>
     */
    template<typename T> bool Insert(FeatStd::SizeType index, const T &value, T *buffer, FeatStd::SizeType bufferItemCount)
    {
        bool ok = (buffer != 0) && (index < bufferItemCount);
        if (ok) {
            Internal::Move(buffer + index + 1, buffer + index, bufferItemCount - index - 1);
            buffer[index] = value;
        }
        return ok;
    }

    /**
        <summary>Removes the item with the given index from buffer</summary>
        <param name="index">index of the item to be removed</param>
        <param name="buffer">The buffer.</param>
        <param name="bufferItemCount">Number of buffer items.</param>
        <returns>true if it succeeds, false if it fails.</returns>
     */
    template<typename T> bool Remove(FeatStd::SizeType index, T *buffer, FeatStd::SizeType bufferItemCount)
   {
        bool ok = (buffer != 0) && (index < bufferItemCount);
        if (ok) {
            Internal::Move(buffer + index, buffer + index + 1, bufferItemCount - index - 1);
        }
        return ok;
    }

    /**
        <summary>
            Insert at the given index the item at bufferItemCount and increments bufferItemCount by one.
        </summary>
        <param name="index">Insertion index.</param>
        <param name="buffer">[in,out] The buffer.</param>
        <param name="bufferItemCount">[in,out] Number of buffer items.</param>
        <param name="bufferSize">Size of the buffer.</param>
        <returns>true if it succeeds, false if it fails.</returns>
     */
    template<typename T, typename N> bool ShiftUp(FeatStd::SizeType index, T *buffer, N &bufferItemCount, FeatStd::SizeType bufferSize)
    {
        bool ok = (buffer != 0) && (index <= bufferItemCount) && (bufferItemCount < bufferSize);
        if (ok) {
            T item(buffer[bufferItemCount]);            // preserve item at buffer end & move items up
            Internal::Move(buffer + index + 1, buffer + index, bufferItemCount - index);
            buffer[index] = item;                       // item at insertion index receives preserved item
            ++bufferItemCount;                          // increment the number of items by one
        }
        return ok;
    }

    /**
        <summary>
            Moves the item with the given index to the end of the buffer and decrements bufferItemCount by one.
        </summary>
        <param name="index">Index of the item to be removed.</param>
        <param name="buffer">[in,out] The buffer.</param>
        <param name="bufferItemCount">[in,out] Number of buffer items.</param>
        <returns>true if it succeeds, false if it fails.</returns>
     */
    template<typename T, typename N> bool ShiftDown(FeatStd::SizeType index, T *buffer, N &bufferItemCount)
    {
        bool ok = (buffer != 0) && (index < bufferItemCount);
        if (ok) {
            --bufferItemCount;
            T item(buffer[index]);
            Internal::Move(buffer + index, buffer + index + 1, bufferItemCount - index);
            buffer[bufferItemCount] = item;
        }
        return ok;
    }

    /**
        <summary>Moves the item with the given index from to index to.</summary>
        <param name="to">the index the item shall be moved to.</param>
        <param name="from">the index of the item to move.</param>
        <param name="buffer">[in,out] The buffer.</param>
        <param name="bufferItemCount">Number of buffer items.</param>
        <returns>true if it succeeds, false if it fails.</returns>
     */
    template<typename T> bool Move(FeatStd::SizeType to, FeatStd::SizeType from, T *buffer, FeatStd::SizeType bufferItemCount)
    {
        bool ok = (buffer != 0) && (to < bufferItemCount) && (from < bufferItemCount);
        if (ok && (from != to)) {
            if (from < to) {
                ++to;
                ok = Internal::ShiftDown(from, buffer, to);
            }
            else {
                ok = Internal::ShiftUp(to, buffer, from, bufferItemCount);
            }
        }
        return ok;
    }
}}   // namespace
/// @}
#endif // Courier_Util_TemplateOperations_h
