//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Util_StaticObjectBuffer_h)
#define Courier_Util_StaticObjectBuffer_h

#include <Courier/Platform/Base.h>
#include <FeatStd/Util/PointerUtil.h>
#include <Courier/Diagnostics/Log.h>

#define SIZE_OF_VOID_PTR sizeof(void*)

namespace Courier {
/// @addtogroup COURIER_UTIL
/// @{
/**
 * StaticObjectBuffer internals.
 * @remark Simulates nested class, which didn't work.
 */
namespace Internal { namespace StaticObjectBuffer_Private {

/**
 * Holds size information and pointer to buffer for a certain type.
 */
struct BufferData
{
    BufferData(UInt32 size, void* dataPtr) : mSize(size), mDataPtr(dataPtr) {
        FEATSTD_DEBUG_ASSERT(mSize > 0);
        FEATSTD_DEBUG_ASSERT(mDataPtr != 0);
    }

    const UInt32 mSize;
    void*const mDataPtr;

private:
    FEATSTD_MAKE_CLASS_UNCOPYABLE(BufferData);
};

/**
 * Base for RawObject definitions.
 *   Holds administrative information (mMark) about "used state"
 *   of ObjectBuffer elements.
 */
struct RawObjectBase
{
    typedef FeatStd::SizeType MarkType;

    static const MarkType cMarkFree = 0x00000000; ///< Unused-Sign
    static const MarkType cMarkUsed = 0xBEEFCAFE; ///< Used-Sign

    static bool IsUsed(void* pointer) {
        FEATSTD_DEBUG_ASSERT(pointer != 0);

        const MarkType& lMarkSign(*(reinterpret_cast<MarkType*>(pointer) - 1));

        if (lMarkSign == cMarkUsed) {
            return true;
        }
        FEATSTD_DEBUG_ASSERT(lMarkSign == cMarkFree);
        return false;
    }

    /// @remark Unchecked means without check of boundaries.
    static void FreeUnchecked(void* pointer) {
        FEATSTD_DEBUG_ASSERT(pointer != 0);

        MarkType& lMarkSign(*(reinterpret_cast<MarkType*>(pointer) - 1));

        FEATSTD_DEBUG_ASSERT(lMarkSign == cMarkUsed);
        lMarkSign = cMarkFree;
    }

// TODO private (at least protected):
        MarkType mMark; ///> "used state" indicator
};

/**
 * Struct to hold one instance of class Type.
 * @tparam Type Class name of objects.
 */
template<typename Type>
struct RawObject : RawObjectBase
{
    FeatStd::SizeType mObject[(sizeof(Type) + SIZE_OF_VOID_PTR - 1) / SIZE_OF_VOID_PTR]; ///< Force SIZE_OF_VOID_PTR byte boundary
};

class StaticObjectBufferBase
{
protected:
    static void* AllocBase(const Internal::StaticObjectBuffer_Private::BufferData* mBufferData, void* (*AccessRawObject)(void* data, UInt32 index), RawObjectBase::MarkType& (*GetMark)(void* data), void* (*GetObject)(void* data));
    static void FreeBase(void* object, const Internal::StaticObjectBuffer_Private::BufferData* mBufferData, void* (*AccessRawObject)(void* data, UInt32 index), RawObjectBase::MarkType& (*GetMark)(void* data));
    static UInt32 CountBase(const Internal::StaticObjectBuffer_Private::BufferData* mBufferData, void* (*AccessRawObject)(void* data, UInt32 index), RawObjectBase::MarkType& (*GetMark)(void* data));
    static UInt32 CapacityBase(const Internal::StaticObjectBuffer_Private::BufferData* mBufferData);
    static void* GetRawObjectBase(UInt32 index, const Internal::StaticObjectBuffer_Private::BufferData* mBufferData, void* (*AccessRawObject)(void* data, UInt32 index));
    static void InitBufferDataBase(const Internal::StaticObjectBuffer_Private::BufferData*& mBufferData, const Internal::StaticObjectBuffer_Private::BufferData* bufferData);
};

}} //Internal

/**
 * Container to hold a requested number (Size) of instances of class Type.
 *   Memory is preallocated for all instances
 *   at declaration time (static).<br>
 *   Requesting more than Size instances isn't handled as error!
 *   Null will be returned.
 * @attention CTOR/DTOR are not called!
 *            See StaticMessageFactory for usage.
 * @see StaticMessageFactory
 * @tparam Type Class name which buffer works for.
 * @tparam Size Number of instances.
 */
template <typename Type>
class StaticObjectBuffer : public Internal::StaticObjectBuffer_Private::StaticObjectBufferBase
{
    typedef Internal::StaticObjectBuffer_Private::RawObject<Type> RawObject;
    typedef Internal::StaticObjectBuffer_Private::RawObjectBase RawObjectBase;

    COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Messaging);

public:
    /// Returns pointer to free (but preallocated) object buffer.
    static Type* Alloc() {
        return reinterpret_cast<Type*>(AllocBase(mBufferData, AccessRawObject, GetMark, GetObject));
    }

    /// Frees object buffer (memory remains allocated).
    static void Free(Type* object) {
        FreeBase(object, mBufferData, AccessRawObject, GetMark);
    }

    /// Returns number of used instances.
    static UInt32 Count() {
        return CountBase(mBufferData, AccessRawObject, GetMark);
    }

    static UInt32 Capacity() {
        return CapacityBase(mBufferData);
    }
    static RawObject& GetRawObject(UInt32 index) {
        return *static_cast<RawObject* >(GetRawObjectBase(index, mBufferData, AccessRawObject));
    }

    static const Internal::StaticObjectBuffer_Private::BufferData* mBufferData;
    static void InitBufferData(const Internal::StaticObjectBuffer_Private::BufferData* bufferData) {
        InitBufferDataBase(mBufferData, bufferData);
    }

private:
    static void* GetObject(void* data)
    {
        return reinterpret_cast<RawObject*>(data)->mObject;
    }

    static RawObjectBase::MarkType& GetMark(void* data)
    {
        return reinterpret_cast<RawObject* >(data)->mMark;
    }

    static void* AccessRawObject(void* data, UInt32 index) {
        return reinterpret_cast<void*>(&static_cast<RawObject*>(data)[index]);
    }

    FEATSTD_MAKE_CLASS_STATIC(StaticObjectBuffer);
};

template <typename Type>
const Internal::StaticObjectBuffer_Private::BufferData* StaticObjectBuffer<Type>::mBufferData;

namespace Internal { namespace StaticObjectBuffer_Private {
template <typename Type>
struct BufferDataTyped : public BufferData
{
    BufferDataTyped(UInt32 size, void* dataPtr) : BufferData(size, dataPtr) {
        StaticObjectBuffer<Type>::InitBufferData(this);
    }
};
}} //Internal
/// @}
}

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//----------------------- CONFIGURATION ---------------------------------------
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#define COURIER_STATIC_OBJECT_BUFFER_CONFIGURATION(Type, Count) \
    static FeatStd::SizeType FEATSTD_CONCAT2(sBuffer, __LINE__)[((sizeof(Type) + SIZE_OF_VOID_PTR + SIZE_OF_VOID_PTR - 1) / SIZE_OF_VOID_PTR) * Count]; \
    static const Courier::Internal::StaticObjectBuffer_Private::BufferDataTyped< Type > FEATSTD_CONCAT2(scBufferData, __LINE__)(Count, FEATSTD_CONCAT2(sBuffer, __LINE__))
#endif
