//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Util_Identifier_h)
#define Courier_Util_Identifier_h

#include <FeatStd/Util/Hash.h>
#include <Courier/Util/Limits.h>
#include <FeatStd/Util/StringBuffer.h>

#if defined(COURIER_NO_MESSAGE_STRING_IDENTIFIERS)
#   define COURIER_IDENTIFIER_STORAGE_INITIALIZER(hashId, stringId) { hashId, FEATSTD_STRINGIZE(hashId) }
#else
#   define COURIER_IDENTIFIER_STORAGE_INITIALIZER(hashId, stringId) { hashId, FEATSTD_STRINGIZE(stringId) }
#endif

namespace Courier {
    /// @addtogroup COURIER_UTIL
    /// @{
    /** class IdentifierStorage stores depending on build mode a hash and a string identifier */
    struct IdentifierStorage {
        UInt32 mHash;
        const Char *mStringId;

        UInt32 HashId() const {
            return mHash;
        }

        const Char* StringId() const {
            return mStringId;
        }
    };

    class Identifier {
        public:
            static const Char cHashIndicator = FeatStd::Hash::cHashIndicator;
            static const Char cIndexIndicator = FeatStd::Hash::cIndexIndicator;
            static const UInt32 cInvalidHashValue = FeatStd::Hash::cInvalidHashValue;
            FEATSTD_LINT_SYMBOL(1938, Courier::Identifier::Identifier, "access to global constant data only")

            Identifier() : mHash(Identifier::cInvalidHashValue) { }

            explicit Identifier(const Char *name) : mHash(Identifier::cInvalidHashValue) {
                if (0 != name) {
                    mHash = CalcHash(name);
                }
            }

            Identifier(const Char *name, UInt32 nameLen) : mHash(Identifier::cInvalidHashValue) {
                if (0 != name) {
                    mHash = CalcHash(name, nameLen);
                }
            }

            explicit Identifier(UInt32 hash) : mHash(hash) { }

            Identifier(const IdentifierStorage &identifierStorage) : mHash(identifierStorage.mHash) { }

            bool IsValid() const {
                return mHash != cInvalidHashValue;
            }

            UInt32 HashValue() const {
                return mHash;
            }

            void Reset() {
                mHash = this->cInvalidHashValue;
            }

            bool operator==(const Identifier &id) const {
                return mHash == id.mHash;
            }

            bool operator==(UInt32 hash) const {
                return mHash == hash;
            }

            bool operator!=(const Identifier &id) const {
                return mHash != id.mHash;
            }

            bool operator!=(UInt32 hash) const {
                return mHash != hash;
            }

            bool operator<(const Identifier &id) const {
                return mHash < id.mHash;
            }

            bool operator<(UInt32 hash) const {
                return mHash < hash;
            }

            bool operator>(const Identifier &id) const {
                return mHash > id.mHash;
            }

            bool operator>(UInt32 hash) const {
                return mHash > hash;
            }

            static UInt32 CalcHash(register const Char *name, UInt32 nameLen = Internal::Limits<UInt32>::cMaxValue) {
                return FeatStd::Hash::CalcHash(name, nameLen);
            }

        private:
            UInt32 mHash;
    };

    // ------------------------------------------------------------------------
    inline bool operator==(UInt32 left, const Identifier &right) {
        return left == right.HashValue();
    }

    // ------------------------------------------------------------------------
    inline bool operator!=(UInt32 left, const Identifier &right) {
        return left != right.HashValue();
    }

    /// @}
}

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {
    template<> UInt32 StringBufferAppender<Courier::Identifier>::Append(StringBuffer& stringBuffer, Courier::Identifier const & object);
}
#endif
#endif
