//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Util_DynamicSizeList_h)
#define Courier_Util_DynamicSizeList_h

#include <Courier/Base.h>
#include <Courier/Util/Limits.h>
#include <Courier/Util/TemplateOperations.h>
#include <Courier/Diagnostics/ErrorHandling.h>

#include <FeatStd/MemoryManagement/PlacementNew.h>
#include <FeatStd/Container/Vector.h>

namespace Courier { namespace Internal {
    /// @addtogroup COURIER_UTIL
    /// @{
    /** FixedSizeList is a container with fixed number of items
        FixedSizeList does not use dynamic memory allocation, all memory is
        reserved statically at compile time. Insert and remove operations are
        fast, random item access time independent from number of elements in
        the container.
        @param T the item type
        @param maxSize capacity of the container */
    template<typename T> class DynamicSizeList {
        public:
            //@{
            /** DynamicSizeList default and copy ctors */
            DynamicSizeList()
            {
            }

            DynamicSizeList(const DynamicSizeList &other) :
                mItem(other.mItem)
            {
            }
            //@}

            /** DynamicSizeList dtor - destructs all objects in the container */
            ~DynamicSizeList()
            {
                Clear();
            }

            /** return the number of items in the container
                @return number of items in the container */
            inline  FeatStd::SizeType Count() const
            {
                return mItem.Size();
            }

            /** returns the maximum number of items that can be stored in the container
                @return the maximum number of items  */
            inline  FeatStd::SizeType Capacity() const
            {
                return mItem.GetCapacity();
            }

            inline void Reserve(FeatStd::SizeType capacity)
            {
                return mItem.Reserve(capacity);
            }

            //@{
            /** returns the item at position index in the container
                @param index where 0 <= index < Count()
                @return the item at position index */
            inline T& Item(FeatStd::SizeType index)
            {
                FEATSTD_DEBUG_ASSERT(index < Count());
                return mItem[index];
            }

            inline const T& Item(FeatStd::SizeType index) const
            {
                FEATSTD_DEBUG_ASSERT(index < Count());
                return mItem[index];
            }
            //@}

            //@{
            /** operator[] returns the list item with the given index
                @param index where 0 <= index < Count()
                @return the item at position index */
            inline T& operator[](FeatStd::SizeType index)
            {
                return Item(index);
            }

            inline const T& operator[](FeatStd::SizeType index) const
            {
                return Item(index);
            }
            //@}

            /** appends the given item at the end of the container
                @param item the item to insert
                @return true if the item could be appended, false otherwise */
            bool PushBack(const T &item)
            {
                return mItem.Add(item);
            }

            /** inserts the given item before the given position
                @param index the index the item shall be inserted before - 0 <= index <= Count()
                @param item the item to insert
                @return true if the item could be inserted, false otherwise */
            bool InsertAt(FeatStd::SizeType index, const T &item)
            {
                return mItem.Insert(index, item);
            }

            /** removes the item at the given position from the container
                @param index the position of the item that shall be removed
                @return true if the item could be removed, false otherwise */
            bool RemoveAt(FeatStd::SizeType index)
            {
                return mItem.Remove(index);
            }

            /**
                <summary>Moves item at index src to index dst</summary>
                <param name="dst">Destination index for the item.</param>
                <param name="src">Source index for the item.</param>
                <returns>true item could be moved, false otherwise.</returns>
             */
            inline bool Move(FeatStd::SizeType dst, FeatStd::SizeType src)
            {
                if (dst > src) {
                    T item = Item(src);
                    return mItem.Remove(src) && mItem.Insert(dst - 1, item);
                }
                T item = Item(src);
                return mItem.Remove(src) && mItem.Insert(dst, item);
            }

            /** removes all items from the container */
            void Clear()
            {
                mItem.Clear();
            }

            //@{
            /** assignment operator
                @param fsList the FixedSizeList to copy items from
                @return the result of the assignment */
            DynamicSizeList& operator=(const DynamicSizeList& other)
            {
                mItem = other.mItem;
                return *this;
            }
            //@}

        private:
            FeatStd::Internal::Vector<T> mItem;
    };
    /// @}
} } // namespace
#endif // Courier_Util_DynamicSizeList_h
