//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Courier_Util_DynamicSizeBitVector_h)
#define Courier_Util_DynamicSizeBitVector_h

#include <Courier/Util/BitTools.h>
#include <FeatStd/Platform/Memory.h>
#include <FeatStd/Container/Vector.h>

namespace Courier { namespace Internal {
    /// @addtogroup COURIER_UTIL
    /// @{
    template<typename T> class DynamicSizeBitVectorIterator;

    /**
       @brief dynamic size bit vector.
       Container class capable to store a given number of bits.
       @tparam BitCapacity the number of bits that can be stored in the container
     */
    class DynamicSizeBitVector {
        public:
            /**
               @brief Default constructor.
             */
            inline DynamicSizeBitVector() {
                Reset();
            }

            /**
               <summary> Gets the capacity of the bit vector.</summary>
               <returns> number of bits that can be stored in the vector </returns>
             */
            inline FeatStd::SizeType Capacity() const {
                return mBits.Size() * 32;
            }

            inline void Resize(FeatStd::SizeType size) {
                FeatStd::SizeType qSize = (size + 31) / 32;
                mBits.Reserve(qSize);
                while (mBits.Size() < qSize) {
                    mBits.Add(0);
                }
            }

            /**
               @brief Tests whether the bit with the given index is set or not
               @param   idx The index.
               @return true if bit is set, false not.
             */
            inline bool Test(FeatStd::SizeType idx) const {
                return (Quad(idx) & Bit(idx)) != 0;
            }

            /**
               @brief Sets the bit with the given index
               @param   idx The index to set.
             */
            inline void Set(FeatStd::SizeType idx) {
                Quad(idx) |= Bit(idx);
            }

            /**
               @brief Clears the bit with the given index
               @param   idx The index.
             */
            inline void Reset(FeatStd::SizeType idx) {
                Quad(idx) &= Mask(idx);
            }

            /**
               <summary>Sets all bits in the container.</summary>
             */
            inline void Set() {
                FeatStd::Internal::Memory::Set(&mBits[0], 0xff, mBits.Size() * sizeof(UInt32));
            }

            /**
               @brief Resets this object.
             */
            inline void Reset() {
                FeatStd::Internal::Memory::Set(&mBits[0], 0, mBits.Size() * sizeof(UInt32));
            }

            /**
               <summary>Gets the index of the first bit set in the vecor.</summary>
               <returns>The index of the first set bit, if no bits are set, EndIndex() will be returned.</returns>
               @code
               DynamicSizeBitVector<33> bv;
               ...
               UInt32 idx = bv.GetFirstSetBit();
               while (idx != EndIndex()) {
                  ...
                  idx = bv.GetNextSetBit(idx);
               }
               @endcode
             */
            FeatStd::SizeType GetFirstSetBit() const {
                return AdvanceToNextSetBit(0);
            };

            /**
               <summary>Gets the index of th enext set bit.</summary>
               <param name="previousIdx">index returned by GetFirstSetBit or GetNextSetBit.</param>
               <returns>The index of the next set bit.</returns>
             */
            FeatStd::SizeType GetNextSetBit(FeatStd::SizeType previousIdx) const {
                return AdvanceToNextSetBit(previousIdx + 1);
            }

            /**
               <summary>returns the last + 1 index</summary>
               <returns>returns the last + 1 index</returns>
             */
            FeatStd::SizeType EndIndex() const {
                return mBits.Size() * 32;
            }

        private:
            FeatStd::Internal::Vector<UInt32> mBits; ///< The bits

            /**
               <summary>Advance to next set bit.</summary>
               <param name="idx">0 for the initial call, otherwise value retured by the previous call.</param>
               <returns>index to the next bit set, if no set bits left, cBitCapacity will be returned</returns>
             */
            FeatStd::SizeType AdvanceToNextSetBit(FeatStd::SizeType idx) const {
                if (idx >= Capacity()) {
                    return EndIndex();
                }

                FeatStd::SizeType qIdx = QuadIdx(idx);
                UInt32 bitIdx = BitIdx(idx);

                for (;;) {
                    while ((qIdx < mBits.Size()) && (mBits[qIdx] == 0)) {     // find non zero quad
                        ++qIdx;
                    }

                    if (qIdx == mBits.Size()) {
                        idx = Capacity();                             // break if at end of vector
                        break;
                    }

                    bitIdx += CountLeadingZeros(mBits[qIdx] >> bitIdx);
                    if (bitIdx < 32) {                                  // set bit found ->
                        idx = (qIdx << 5U) + bitIdx;                    // recalc idx and break
                        break;
                    }

                    bitIdx = 0;
                    ++qIdx;
                }
                return idx;
            }

            static inline UInt32 BitIdx(FeatStd::SizeType idx) {
                return idx & 0x1F;
            }

            /**
               @brief returns the mBits index of the given bit index
               @param   idx The bit index.
               @return the mBits index
             */
            FeatStd::SizeType QuadIdx(FeatStd::SizeType idx) const {
                FEATSTD_DEBUG_ASSERT(idx < Capacity());
                return (idx < Capacity()) ? (idx >> 5U) : 0;
            }

            /**
               @brief returns the mBits array entry of the given bit index
               @param   idx The bit index.
               @return the mBits array entry
             */
            inline UInt32 Quad(FeatStd::SizeType idx) const {
                return mBits[QuadIdx(idx)];
            }

            /**
               @brief returns the mBits array entry of the given bit index
               @param   idx The bit index.
               @return the mBits array entry
             */
            inline UInt32& Quad(FeatStd::SizeType idx) {
                return mBits[QuadIdx(idx)];
            }

            /**
               @brief returns the bit for the given bit index
               @param   idx The bit index.
               @return .
             */
            static inline UInt32 Bit(FeatStd::SizeType idx) {
                return 1U << UInt32(idx & 0x1f);
            }

            /**
               @brief returns the mask for the given bit index
               @param   idx The bit index.
               @return .
             */
            static inline UInt32 Mask(FeatStd::SizeType idx) {
                return ~Bit(idx);
            }
    };
    /// @}
}}

#endif
