//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CanderaTransitionHandler.h"
#include "TransitionMsgs.h"

#include <CanderaTransitions/TransitionStrategy.h>
#include <Courier/Diagnostics/Log.h>
#include <Courier/Visualization/ViewHandler.h>
#include <Candera/EngineBase/Common/AbstractNodePointerTraverser.h>

using namespace Candera;
using namespace Candera::Transitions;

namespace Courier { namespace Internal {

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Visualization);

static ViewScene::MessagingTransitionDeactivationHandler& GetMessagingTransitionDeactivationHandler()
{
    static ViewScene::MessagingTransitionDeactivationHandler s_messagingTransitionDeactivationHandler = 0;
    return s_messagingTransitionDeactivationHandler;
}

void CanderaTransitionHandler::SetMessagingTransitionDeactivationHandler(ViewScene::MessagingTransitionDeactivationHandler messagingTransitionDeactivationHandler)
{
    GetMessagingTransitionDeactivationHandler() = messagingTransitionDeactivationHandler;
}

class FindNodeTraverser : public Candera::Internal::AbstractNodePointerTraverser
{
public:
    FindNodeTraverser(Candera::Id id) :
        m_id(id)
    {
    }

    const AbstractNodePointer& GetMatchingNode() const
    {
        return m_matchingNode;
    }

protected:
    virtual TraverserAction ProcessNode(const AbstractNodePointer& node) override
    {
        if ((0 != node.ToCanderaObject()) && (node.ToCanderaObject()->GetId() == m_id)) {
            m_matchingNode = node;
            return StopTraversing;
        }
        return ProceedTraversing;
    }

private:
    FEATSTD_MAKE_CLASS_STATIC(FindNodeTraverser);
    FEATSTD_MAKE_CLASS_UNCOPYABLE(FindNodeTraverser);
    const Candera::Id m_id;
    AbstractNodePointer m_matchingNode;
};

class TransitionActivationHandler
{
public:
    static Courier::ViewScene* GetView(Courier::IViewHandler& viewHandler, const Candera::Transitions::Identifier& identifier, CanderaObject*& camera)
    {
        AbstractNodePointer scene;
        Candera::Id sceneId = identifier.GetSceneId();
        switch (identifier.GetType()) {
#ifdef CANDERA_2D_ENABLED
            case Transitions::Identifier::Node2DBuiltInIdentifier:
            case Transitions::Identifier::Scene2DBuiltInIdentifier: {
                    Candera::Scene2DContext* sceneContext = DefaultAssetProvider::GetInstance().GetScene2DById(sceneId);
                    if (0 != sceneContext) {
                        scene = AbstractNodePointer(sceneContext->GetScene());
                    }
                }
                break;
#endif
#ifdef CANDERA_3D_ENABLED
            case Transitions::Identifier::Node3DBuiltInIdentifier:
            case Transitions::Identifier::Scene3DBuiltInIdentifier: {
                    Candera::SceneContext* sceneContext = DefaultAssetProvider::GetInstance().GetSceneById(sceneId);
                    if (0 != sceneContext) {
                        scene = AbstractNodePointer(sceneContext->GetScene());
                    }
                }
                break;
            default:
                break;
#endif
        }
        Courier::ViewScene* viewScene = 0;
        camera = 0;
        if (scene.IsValid()) {
            Courier::ViewId viewId(scene.ToCanderaObject()->GetName());
            Courier::View* view = viewHandler.FindView(viewId);
            if (0 == view) {
                if (!viewHandler.ExecuteViewAction(Courier::ViewAction::CreateAll, viewId, true, true)) {
                    FEATSTD_LOG_WARN("View creation failed!");
                }
                view = viewHandler.FindView(viewId);
            }
            if (0 != view) {
#ifdef CANDERA_2D_ENABLED
                Courier::ViewScene2D* viewScene2D = view->ToViewScene2D();
                if (0 != viewScene2D) {
                    viewScene = viewScene2D;
                }
#endif
#ifdef CANDERA_3D_ENABLED
                Courier::ViewScene3D* viewScene3D = view->ToViewScene3D();
                if (0 != viewScene3D) {
                    viewScene = viewScene3D;
                }
#endif
            }
            if (0 != identifier.GetNodeId()) {
                FindNodeTraverser traverser(identifier.GetNodeId());
                traverser.Traverse(scene);
                camera = traverser.GetMatchingNode().ToCanderaObject();
                bool isCamera = false;
#ifdef CANDERA_2D_ENABLED
                if (0 != Dynamic_Cast<Candera::Camera2D*>(camera)) {
                    isCamera = true;
                }
#endif
#ifdef CANDERA_2D_ENABLED
                if (0 != Dynamic_Cast<Candera::Camera2D*>(camera)) {
                    isCamera = true;
                }
#endif
                if (!isCamera) {
                    camera = 0;
                }
            }
        }
        return viewScene;
    }

    static void OnEvent(Courier::IViewHandler& viewHandler, const Candera::Event& event)
    {
        const Transitions::OnStartTransitionFragment* onStartTransitionFragment = Dynamic_Cast<const Transitions::OnStartTransitionFragment*>(&event);
        const Transitions::OnFinishTransitionFragment* onFinishTransitionFragment = Dynamic_Cast<const Transitions::OnFinishTransitionFragment*>(&event);
        Transitions::TransitionFragment::SharedPointer transitionFragment;
        if (0 != onStartTransitionFragment) {
            transitionFragment = onStartTransitionFragment->GetTransitionFragment();
        }
        else if (0 != onFinishTransitionFragment) {
            transitionFragment = onFinishTransitionFragment->GetTransitionFragment();
        }
        else {
            // intentionally left blank (lint rule 1960)
            // it can only be either a start, finish or a foreign event
        }
        if (!transitionFragment.PointsToNull()) {
            Transitions::Trigger::SharedPointer trigger = transitionFragment->GetTrigger();
            if (!trigger.PointsToNull()) {
                Transitions::RequestFragment::SharedPointer request = trigger->GetRequest();
                if (!request.PointsToNull()) {
                    CanderaObject* camera = 0;
                    Courier::ViewScene* viewScene = GetView(viewHandler, request->GetIdentifier(), camera);
                    if (0 != viewScene) {
                        switch (static_cast<Transitions::RequestFragment::Type>(request->GetType())) {
                            case Transitions::RequestFragment::Activate:
                                if (0 != onStartTransitionFragment) {
                                    if (!viewScene->IsContentLoaded()) {
                                        if (!viewScene->LoadContent(true, true)) {
                                            FEATSTD_LOG_WARN("Loading content failed!");
                                        }
                                    }
                                    if (0 != camera) {
                                        viewScene->EnableRenderingForCamera(true, camera);
                                        viewScene->ActivateForCamera(true, camera);
                                    }
                                    else {
                                        viewScene->EnableRendering(true);
                                        viewScene->Activate(true);
                                    }
                                    viewScene->Invalidate();
                                }
                                break;
                            case Transitions::RequestFragment::Deactivate:
                                if (0 != onFinishTransitionFragment) {
                                    if (0 != camera) {
                                        viewScene->EnableRenderingForCamera(false, camera);
                                        if ((0 == GetMessagingTransitionDeactivationHandler()) ||
                                            GetMessagingTransitionDeactivationHandler()(*viewScene, camera)) {
                                            viewScene->ActivateForCamera(false, camera);
                                        }
                                    }
                                    else {
                                        viewScene->EnableRendering(false);
                                        if ((0 == GetMessagingTransitionDeactivationHandler()) ||
                                            GetMessagingTransitionDeactivationHandler()(*viewScene, camera)) {
                                            viewScene->Activate(false);
                                        }
                                    }
                                }
                                break;
                            default:
                                break;
                        }
                    }
                }
            }
        }
    }
};

// ------------------------------------------------------------------------
CanderaTransitionHandler::CanderaTransitionHandler():
    mAssetProvider(0),
    mViewHandler(0),
    mInvalidateAll(false),
    mTransitionOngoing(false)
{
    mTrEventListener.Init(this);
    static_cast<void>(TransitionStrategy::GetInstance().AddListener(mTrEventListener));
}

// ------------------------------------------------------------------------
CanderaTransitionHandler::~CanderaTransitionHandler()
{
    static_cast<void>(TransitionStrategy::GetInstance().RemoveListener(mTrEventListener));
    mAssetProvider = 0;
    mViewHandler = 0;
}

// ------------------------------------------------------------------------
void CanderaTransitionHandler::Init(IViewHandler * viewHandler, Candera::DefaultAssetProvider* assetProvider)
{
    FEATSTD_DEBUG_ASSERT(0 != viewHandler);
    FEATSTD_DEBUG_ASSERT(0 != assetProvider);
    mAssetProvider = assetProvider;
    mViewHandler = viewHandler;
}

// ------------------------------------------------------------------------
void CanderaTransitionHandler::SetAnimationTimeDispatcher(Candera::Animation::AnimationTimeDispatcher::SharedPointer timeDispatcher) const
{
    TransitionStrategy::GetInstance().SetAnimationTimeDispatcher(timeDispatcher);
}

// ------------------------------------------------------------------------
Candera::Animation::AnimationTimeDispatcher::SharedPointer CanderaTransitionHandler::GetAnimationTimeDispatcher() const
{
    return TransitionStrategy::GetInstance().GetAnimationTimeDispatcher();
}

// ------------------------------------------------------------------------
bool CanderaTransitionHandler::SetTransitionRuleSet(const Candera::Transitions::Rule::Set::SharedPointer& ruleSet)
{
    bool rc = false;
    if (!ruleSet.PointsToNull()) {
        mRuleSet = ruleSet;
        rc = true;
    }
    return rc;
}

// ------------------------------------------------------------------------
void CanderaTransitionHandler::CreateTransitionRequestFragment(CanderaTransitionRequestAction::Enum action, const Candera::Transitions::Identifier& identifier, const Hint& hint)
{
    switch (action)
    {
    case CanderaTransitionRequestAction::Activate:
        {
            AddView(identifier);
            Request::Activate(identifier, hint, mReqHandle);
            break;
        }
    case CanderaTransitionRequestAction::Deactivate:
        {
            AddView(identifier);
            Request::Deactivate(identifier, hint, mReqHandle);
            break;
        }
    case CanderaTransitionRequestAction::Finish:
        {
            AddView(identifier);
            Request::Finish(identifier, hint, mReqHandle);
            break;
        }
    default:
        {
            // MISRA needs this
            break;
        }
    }
}

// ------------------------------------------------------------------------
void CanderaTransitionHandler::CreateControlFlowRequest(CanderaTransitionControlFlowAction::Enum action)
{
    if (CanderaTransitionControlFlowAction::Begin == action) {
        if (!mReqHandle.IsRequestStarted()) {
            mTransitionViewsVct.Clear();
            mInvalidateAll = false;
            Request::Begin(mReqHandle);
        } else {
            COURIER_LOG_ERROR("Could not start to compose transition request as a different transition is already in progress.");
        }
    } else if (CanderaTransitionControlFlowAction::End == action) {
            Request::SharedPointer reqPtr = Request::End(mReqHandle);
            if ((!reqPtr.PointsToNull()) && (!mRuleSet.PointsToNull())) {
                static_cast<void>(TransitionStrategy::GetInstance().Start(reqPtr.GetSharedInstance(), mRuleSet.GetSharedInstance(), mTrSet));
            } else {
                COURIER_LOG_ERROR("Transition could not be started.");
            }
    } else {
        // MISRA needs this
    }
}

// ------------------------------------------------------------------------
void CanderaTransitionHandler::Update(Candera::TimeType deltaTime) const
{
    TransitionStrategy::GetInstance().Update(deltaTime);
}

// ------------------------------------------------------------------------
void CanderaTransitionHandler::InvalidateTransitionViews()
{
    for (FeatStd::SizeType i = 0; i < mTransitionViewsVct.Size(); ++i) {
        mTransitionViewsVct[i]->Invalidate();
    }
}

// ------------------------------------------------------------------------
template <typename SceneType, typename SceneContextType>
static const FeatStd::Char* CanderaTransitionHandlerGetViewNameFromContext(SceneContextType* sceneContext)
{
    if (0 != sceneContext) {
        SceneType* scene = sceneContext->GetScene();
        if (0 != scene) {
            return scene->GetName();
        }
    }
    return 0;
}

static const FeatStd::Char* CanderaTransitionHandlerGetViewName(Candera::DefaultAssetProvider* assetProvider, const Candera::Transitions::Identifier& identifier)
{
    const FeatStd::Char* name = 0;
    if (0 != assetProvider) {
        Candera::Id sceneId = identifier.GetSceneId();
        switch (identifier.GetType()) {
#ifdef CANDERA_2D_ENABLED
        case Transitions::Identifier::Scene2DBuiltInIdentifier:
        case Transitions::Identifier::Node2DBuiltInIdentifier:
            name = CanderaTransitionHandlerGetViewNameFromContext<Scene2D, Scene2DContext>(assetProvider->GetScene2DById(sceneId));
            break;
#endif
#ifdef CANDERA_3D_ENABLED
        case Transitions::Identifier::Scene3DBuiltInIdentifier:
        case Transitions::Identifier::Node3DBuiltInIdentifier:
            name = CanderaTransitionHandlerGetViewNameFromContext<Scene, SceneContext>(assetProvider->GetSceneById(sceneId));
            break;
#endif
        default:
            break;
        }
    }
    return name;
}

void CanderaTransitionHandler::AddView(const Candera::Transitions::Identifier& identifier)
{
    const FeatStd::Char* name = CanderaTransitionHandlerGetViewName(mAssetProvider, identifier);
    View* view = (0 != name) ? mViewHandler->FindView(ViewId(name)) : 0;
    if (0 != view) {
        if (!mTransitionViewsVct.Contains(view)) {
            static_cast<void>(mTransitionViewsVct.Add(view));
        }
    }
    else {
        mInvalidateAll = true;
    }
 }

// ------------------------------------------------------------------------
static void CanderaTransitionHandlerPost(Message* message)
{
    if (0 != message) {
        if (!message->Post()) {
            FEATSTD_LOG_WARN("Transition handler message could not be distributed to all receivers!");
        }
    }
}

template < class EventT >
static bool CanderaTransitionHandlerPostTrEventIndMsg(const Candera::Event& event, CanderaTransitionEventType::Enum type)
{
    const EventT* concreteEvent = FeatStd::Dynamic_Cast<const EventT*>(&event);
    if (0 == concreteEvent) {
        return false;
    }
    CanderaTransitionHandlerPost(COURIER_MESSAGE_NEW(CanderaTransitionIndMsg)(type, concreteEvent->GetTransition()));
    return true;
}

template < class EventT >
static bool CanderaTransitionHandlerPostTrFragmentEventIndMsg(const Candera::Event& event, CanderaTransitionEventType::Enum type)
{
    const EventT* concreteEvent = FeatStd::Dynamic_Cast<const EventT*>(&event);
    if (0 == concreteEvent) {
        return false;
    }
    CanderaTransitionHandlerPost(COURIER_MESSAGE_NEW(CanderaTransitionFragmentIndMsg)(type, concreteEvent->GetTransitionFragment()));
    return true;
}

Candera::EventListener::EventStatus CanderaTransitionHandler::OnEvent(const Candera::Event& event)
{
    if (0 != mViewHandler) {
        Internal::TransitionActivationHandler::OnEvent(*mViewHandler, event);
    }
    bool isOnStartEvent = CanderaTransitionHandlerPostTrEventIndMsg<OnStartTransition>(event, CanderaTransitionEventType::OnStartTransitionEvent);
    bool isOnFinishEvent = CanderaTransitionHandlerPostTrEventIndMsg<OnFinishTransition>(event, CanderaTransitionEventType::OnFinishTransitionEvent);
    bool isOnStartFragmentEvent = CanderaTransitionHandlerPostTrFragmentEventIndMsg<OnStartTransitionFragment>(event, CanderaTransitionEventType::OnStartTransitionFragmentEvent);
    bool isOnFinishFragmentEvent = CanderaTransitionHandlerPostTrFragmentEventIndMsg<OnFinishTransitionFragment>(event, CanderaTransitionEventType::OnFinishTransitionFragmentEvent);
    if (isOnStartEvent) {
        mTransitionOngoing = true;
    }
    if (isOnFinishEvent) {
        mTransitionOngoing = false;
    }
    return (isOnStartEvent || isOnFinishEvent || isOnStartFragmentEvent || isOnFinishFragmentEvent) ? Candera::EventListener::EventHandled : Candera::EventListener::EventNotHandled;
}

// ------------------------------------------------------------------------
Candera::EventListener::EventStatus CanderaTransitionHandler::CanderaTransitionEventListener::OnEvent(const Candera::Event& event)
{
    return (0 != mCanderaTransitionHandler) ? mCanderaTransitionHandler->OnEvent(event) : Candera::EventListener::EventNotHandled;
}
}}
