//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Courier/Platform/RendererHelper.h>
#include <Courier/Diagnostics/Log.h>

namespace Courier { namespace Internal {

COURIER_LOG_SET_REALM(Diagnostics::LogRealm::Platform);

class Layer {
public:
    Layer() : mLayerId(INT_MAX), mRefCount(0) {}
    Layer(Int layerId) : mLayerId(layerId), mRefCount(0) {}
    Int mLayerId;
    Int mRefCount;
};

static Courier::Vector<Layer> layerVct;

static FeatStd::SizeType FindLayerIndex(Int layerId)
{
    FeatStd::SizeType layerIndex = FeatStd::Internal::NativeTypeLimit<FeatStd::SizeType>::Max();
    for (FeatStd::SizeType i = 0; i < static_cast<FeatStd::SizeType>(layerVct.Size()); ++i) {
        if (layerId == layerVct[i].mLayerId) {
            layerIndex = i;
        }
    }
    return layerIndex;
}

bool RendererHelper::PlatformUpload(Gdu* lookupGdu, Courier::Renderer* renderer, bool forceEnable)
{
    COURIER_UNUSED(forceEnable);
    bool result = false;

    Int layerId = lookupGdu->GetLayerId();
    FeatStd::SizeType layerIndex = FindLayerIndex(layerId);

    // layerId not in vector
    if (layerIndex == FeatStd::Internal::NativeTypeLimit<FeatStd::SizeType>::Max()) {
        layerIndex = layerVct.Size();
        if (!layerVct.Add(Layer(layerId))) {
            COURIER_LOG_ERROR("Layer not added in vector!");
        }
    }

    if (lookupGdu->IsLoaded()) {
        layerVct[layerIndex].mRefCount++;
        COURIER_LOG_DEBUG("Gdu ref count for layerId (%d) increased to %d", layerId, layerVct[layerIndex].mRefCount);
        lookupGdu->Activate();
    } else {
        // We upload the new render target and start with refcount 1
        if (lookupGdu->Upload(renderer)) {
            lookupGdu->Activate();
            layerVct[layerIndex].mRefCount = 1;
            Int width = (lookupGdu->GetRenderTarget()!=0) ? lookupGdu->GetRenderTarget()->GetWidth() : -1;
            Int height = (lookupGdu->GetRenderTarget()!=0) ? lookupGdu->GetRenderTarget()->GetHeight() : -1;
            COURIER_LOG_DEBUG("Uploaded Gdu on Layer(%d). Width(%d), Height(%d)", layerId, width, height);
            FEATSTD_UNUSED2(width, height); // if logging is disabled, variable/parameter is not used
            result = true;
        } else {
            // Uploading went wrong
            COURIER_LOG_ERROR("Error uploading Gdu on Layer(%d)", layerId);
        }
    }
    return result;
}

bool RendererHelper::PlatformUnload(Gdu* lookupGdu, Renderer* renderer)
{
    bool result = false;
    Int layerId = lookupGdu->GetLayerId();
    if (lookupGdu->IsLoaded()) {
        FeatStd::SizeType layerIndex = FindLayerIndex(layerId);
        if (layerIndex != FeatStd::Internal::NativeTypeLimit<FeatStd::SizeType>::Max()) {
            if (layerVct[layerIndex].mRefCount > 0) {
                layerVct[layerIndex].mRefCount--;
                COURIER_LOG_DEBUG("Gdu ref count for layerId (%d) decreased to %d", layerIndex, layerVct[layerIndex].mRefCount);
            }
            if (layerVct[layerIndex].mRefCount == 0) {
                lookupGdu->Unload(renderer);
                COURIER_LOG_DEBUG("Gdu on Layer(%d) was unloaded.", lookupGdu->GetLayerId());
                if (!layerVct.Remove(layerIndex)) {
                    COURIER_LOG_ERROR("Layer not removed from vector!");
                }
                result = true;
            }
        } else {
            // it should never get here, as the lookupGdu is loaded, however this adds extra safety
            lookupGdu->Unload(renderer);
            COURIER_LOG_DEBUG("Gdu on Layer(%d) was unloaded.", lookupGdu->GetLayerId());
        }
    }
    return result;
}

void RendererHelper::PlatformClean()
{
    layerVct.Clear();
}

}}
