//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Courier/Platform/SerialPort.h>
#include <Courier/Diagnostics/Log.h>

#include <errno.h>
#include <unistd.h>
#include <stdio.h>

#include <fcntl.h>

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Platform);

#define LOG_SYSCALL_ERROR(syscall)   \
    COURIER_LOG_ERROR("SerialPort: ::%s() failed (Error %d)", syscall, errno);

namespace Courier { namespace Platform { namespace Os { namespace Posix {
    // ------------------------------------------------------------------------
    PosixSerialPort::PosixSerialPort() : mSerialPortFileDesc(-1), mSerialSettings()
    {
        for (Int i = StdFileDescriptor::StdIn; i < StdFileDescriptor::Max; i++) {
            mStdFileDescSave[i] = StdFileDescriptor::Invalid;
        }
    }
    
    // ------------------------------------------------------------------------
    bool PosixSerialPort::Open(const Char * serialPortPath)
    {
        bool lRc = true;

        // First flush console
        ::fflush(stdin);
        ::fflush(stdout);

        // Open serial
        mSerialPortFileDesc = open(serialPortPath, O_RDWR | O_NOCTTY | O_NDELAY);
        lRc = (mSerialPortFileDesc >= 0);

        if (lRc) {
            struct termios lSerialSettings;
            ::tcgetattr(mSerialPortFileDesc, &lSerialSettings);
            mSerialSettings = lSerialSettings;
            // Convert carriage return to newline on output
            lSerialSettings.c_oflag |= ONLCR;
            // Disable <enter> confirmation and echo on input
            lSerialSettings.c_lflag &= ~(ICANON | ECHO);
            ::tcsetattr(mSerialPortFileDesc, TCSANOW, &lSerialSettings);
        }

        return lRc;
    }

    // ------------------------------------------------------------------------
    bool PosixSerialPort::Close()
    {
        ::tcsetattr(mSerialPortFileDesc, TCSANOW, &mSerialSettings);
        return (0 == close(mSerialPortFileDesc));
    }

    // ------------------------------------------------------------------------
    bool PosixSerialPort::Pipe(StdFileDescriptor::Enum fileDesc, bool activate)
    {
        COURIER_DEBUG_ASSERT(mSerialPortFileDesc >= 0);
        if (mSerialPortFileDesc < 0) {
            COURIER_LOG_ERROR("Serial port is not open!");
            return false;
        }

        bool lRc = true;
        int lPosixFileDesc;

        // Map posix standard file descriptor for later restoring
        switch (fileDesc) {
        case StdFileDescriptor::StdIn:
            lPosixFileDesc = STDIN_FILENO;
            break;
        case StdFileDescriptor::StdOut:
            lPosixFileDesc = STDOUT_FILENO;
            break;
        case StdFileDescriptor::StdErr:
            lPosixFileDesc = STDERR_FILENO;
            break;
        default:
            COURIER_LOG_ERROR("Invalid standard file descriptor!");
            lRc = false;
        }

        if (lRc) {
            // Enable pipe to of standard file descriptor to serial port
            if (activate) {
                // Duplicate standard file descriptor for later restoring
                mStdFileDescSave[fileDesc] = ::dup(lPosixFileDesc);

                lRc = (-1 != mStdFileDescSave[fileDesc]);
                if (!lRc) {
                    mStdFileDescSave[fileDesc] = StdFileDescriptor::Invalid;
                    LOG_SYSCALL_ERROR("dup");
                }

                // Close given file descriptor and pipe it to serial
                if (lRc && (-1 == ::dup2(mSerialPortFileDesc, lPosixFileDesc))) {
                    mStdFileDescSave[fileDesc] = StdFileDescriptor::Invalid;
                    LOG_SYSCALL_ERROR("dup2");
                    lRc = false;
                }
            }
            // Restore standard file descriptor
            else {
                lRc = (-1 != ::dup2(mStdFileDescSave[fileDesc], lPosixFileDesc));
                if (!lRc) {
                    LOG_SYSCALL_ERROR("dup2");
                }
            }
        }

        return lRc;
    }

    // ------------------------------------------------------------------------
    Int32 PosixSerialPort::GetDescriptor() const
    {
        return mSerialPortFileDesc;
    }
}}}}
