//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Courier/Platform/MessageQueue.h>
#include <Courier/Platform/CriticalSectionLocker.h>
#include <Courier/Platform/Thread.h>
#include <Courier/Messaging/Message.h>
#include <Courier/Diagnostics/ErrorHandling.h>
#include <Courier/Diagnostics/Log.h>

namespace Courier { namespace Platform { namespace Os { namespace Generic {

    // ------------------------------------------------------------------------
    GenericMessageQueue::GenericMessageQueue()
    {
        bool lOk = mMsgWaitSem.Create();
        FEATSTD_PANIC_IF(!lOk, "Semaphore initialization failed!");
    }

    // ------------------------------------------------------------------------
    GenericMessageQueue::~GenericMessageQueue()
    {
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType GenericMessageQueue::GetSize() const 
    { 
        // avoid changing const interface, therefore cast const away to access CS
        GenericMessageQueue * non_const_this = const_cast<GenericMessageQueue*>(this);  
        Platform::CriticalSectionLocker csLock(&(non_const_this->mMsgListCs));

        return mMsgList.GetSize() + mPriorityMsgList.GetSize(); 
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType GenericMessageQueue::GetPriorityMsgsSize() const
    {
        // avoid changing const interface, therefore cast const away to access CS
        GenericMessageQueue * non_const_this = const_cast<GenericMessageQueue*>(this);  
        Platform::CriticalSectionLocker csLock(&(non_const_this->mMsgListCs));

        return mPriorityMsgList.GetSize(); 
    }

    // ------------------------------------------------------------------------
    FeatStd::SizeType GenericMessageQueue::GetNormalMsgsSize() const
    {
        // avoid changing const interface, therefore cast const away to access CS
        GenericMessageQueue * non_const_this = const_cast<GenericMessageQueue*>(this);  
        Platform::CriticalSectionLocker csLock(&(non_const_this->mMsgListCs));

        return mMsgList.GetSize(); 
    }

    // ------------------------------------------------------------------------
    bool GenericMessageQueue::Push(const MessageReferrer & msgRef)
    {
        Platform::CriticalSectionLocker csLock(&mMsgListCs);

        bool lSuccess = mMsgWaitSem.Release();

        Message * msg = msgRef.GetMessage();
        if(msg!=0) {
            Priority::Enum priority = msg->GetPriority();
            if(priority==Priority::Normal) {
                lSuccess = lSuccess && mMsgList.Append(msgRef);
            } else {
                // currently we do not sort messages by priority, just append the new prioritized message
                lSuccess = lSuccess && mPriorityMsgList.Append(msgRef);
            }
        }
        return lSuccess;
    }

    // ------------------------------------------------------------------------
    bool GenericMessageQueue::Pop(MessageReferrer & msgRef)
    {
        return Wait(msgRef, Platform::TicksType::Zero);
    }

    // ------------------------------------------------------------------------
    bool GenericMessageQueue::Wait(MessageReferrer & msgRef, const Platform::Ticks & timeout, bool priorityOnly)
    {
        bool lMsgFound = false;

        // If message found in message queue
        bool lSuccess;
        if (timeout.IsInfinite()) {
            lSuccess = mMsgWaitSem.Obtain();
        } else {
            lSuccess = mMsgWaitSem.Obtain(timeout);
        }

        if (lSuccess) {
            Platform::CriticalSectionLocker csLock(&mMsgListCs);
            // Pop first priority message, if priority message list is not empty
            lMsgFound = ! mPriorityMsgList.IsEmpty();
            if (lMsgFound) {
                msgRef = *mPriorityMsgList.Begin();
                bool lRemoved = mPriorityMsgList.Remove(msgRef);
                FEATSTD_DEBUG_ASSERT(lRemoved);
                FEATSTD_UNUSED(lRemoved);
                // yes we had a priority message in queue so return
                return true;
            }

            if (!priorityOnly) {
                // Pop first message, if message list is not empty
                lMsgFound = !mMsgList.IsEmpty();
                if (lMsgFound) {
                    msgRef = *mMsgList.Begin();
                    bool lRemoved = mMsgList.Remove(msgRef);
                    FEATSTD_DEBUG_ASSERT(lRemoved);
                    FEATSTD_UNUSED(lRemoved);
                }
            }
            else {
                // Non-priority (normal) message was found in queue, but it is intentionally ignored. 
                // So, the semaphore must be released, since the queue is left unchanged.
                static_cast<void>(mMsgWaitSem.Release());
            }
        }
        return lMsgFound;
    }

    // ------------------------------------------------------------------------
    void GenericMessageQueue::Observe(void * data, MessageObserveFct fct)
    {
        FEATSTD_DEBUG_ASSERT(fct!=0);
        if(fct!=0) {
            Platform::CriticalSectionLocker csLock(&mMsgListCs);
            MessageList::Iterator it = mPriorityMsgList.Begin();
            while(it!=mPriorityMsgList.End()) {
                fct(data,(*it).GetMessage());
                ++it;
            }

            it = mMsgList.Begin();
            while(it!=mMsgList.End()) {
                fct(data,(*it).GetMessage());
                ++it;
            }
        }
    }

}}}}
