//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "MessageRouter.h"

#include "Message.h"
#include "MessageReferrer.h"
#include "MessageReceiver.h"
#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
    #include "MessagingMonitor.h"
#endif

#include <Courier/Platform/CriticalSectionLocker.h>
#include <Courier/Platform/MessageFactory.h>

#if defined(COURIER_IPC_ENABLED)
    #include <Courier/Ipc/IpcComponent.h>
    #include <Courier/Serialization/MessageVerification.h>
#endif
#include <Courier/Diagnostics/Trace.h>

namespace Courier {  

    COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Messaging);

    using namespace Diagnostics;

    namespace Internal {

// ------------------------------------------------------------------------
//     /// Global component receiver map.
//     static MessageReceiver * gComponentReceiverMap[COURIER_COMPONENT_MAX_COUNT];
//
//     /// Global receiver list.
//     static MessageReceiver * gReceiverList[cCOURIER_RECEIVER_MAX_COUNT];
//     /// Number of receivers registered in global receiver list.
//     static UInt32 gReceiverListCount = 0;

    static MessageHookSignature gMessageHook = 0;   ///< pointer to the active message hook

    // ------------------------------------------------------------------------
    FEATSTD_LINT_CTORDTOR_EFFECT(Courier::Internal::ComponentReceiverMapLock)
    /// Critical section for component receiver map manipulation.
    class ComponentReceiverMapLock : private Platform::CriticalSectionLocker {
    public:
        FEATSTD_LINT_CURRENT_SCOPE(1509,
            "The virtual destructor in ComponentReceiverMapLock is not needed as there are no non-static class members in the derived class.")
            ComponentReceiverMapLock() : Platform::CriticalSectionLocker(GetLock()) {
        }

    private:
        static Platform::CriticalSection * GetLock() {
            static Platform::CriticalSection cs;
            return &cs;
        }
    };

    // ------------------------------------------------------------------------
    FEATSTD_LINT_CTORDTOR_EFFECT(Courier::Internal::ReceiverListLock)
    /// Critical section for receiver list manipulation.
    class ReceiverListLock : private Platform::CriticalSectionLocker {
    public:
        FEATSTD_LINT_CURRENT_SCOPE(1509,
            "The virtual destructor in ReceiverListLock is not needed as there are no non-static class members in the derived class.")
            ReceiverListLock() : Platform::CriticalSectionLocker(GetLock()) {
        }

    private:
        static Platform::CriticalSection * GetLock() {
            static Platform::CriticalSection cs;
            return &cs;
        }
    };

    // ------------------------------------------------------------------------
    UInt32 MessageRouter::mReceiverListCount = 0;
    MessageReceiver ** MessageRouter::mComponentReceiverMap = 0;
    MessageReceiver ** MessageRouter::mReceiverList = 0;

    // ------------------------------------------------------------------------
    void MessageRouter::Init()
    {
        // Lazy initialization of component receiver map.
        static MessageReceiver * lComponentReceiverMap[COURIER_COMPONENT_MAX_COUNT];

        // Lazy initialization of receiver list.
        static MessageReceiver * lReceiverList[cCOURIER_RECEIVER_MAX_COUNT];

        mComponentReceiverMap = lComponentReceiverMap;
        mReceiverList = lReceiverList;

        COURIER_LOG_INFO("Message router was initialized.");
    }

    // ------------------------------------------------------------------------
    bool MessageRouter::Post(Message * msg)
    {
        bool lSuccess = (0 != msg);
        FEATSTD_DEBUG_ASSERT(lSuccess);

        // Invalid message object
        if (!lSuccess) {
            COURIER_LOG_ERROR("Invalid message!");
            return false;
        }
#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
        // Apply startup time 
        msg->GetMonitoringData().Init();
        UInt32 serializedMessageSize;
#endif

#if defined(COURIER_IPC_ENABLED)
        // compute the unique check sum
    #if defined(COURIER_MESSAGING_MONITOR_ENABLED)
        serializedMessageSize = MessageVerification::ComputeChecksum(*msg);
    #else
        (void)MessageVerification::ComputeChecksum(*msg);
    #endif
#endif

        // Apply reference counting via message referrer (manages message object lifetime from now on)
        MessageReferrer msgRef(msg);

        // if set and this is the initial post of the message, invoke gMessageHook.
        // Initial post means that the index of the recipient list is 0
        // (messages get reposted in sequential distribution).
        if ((gMessageHook != 0)  && (msg->CurrentRecipient() == 0) &&  (! gMessageHook(msg))) {
            // if the message hook returned false, the distribution will be abandoned
            return true;
        }

        COURIER_TRACE_MESSAGE(TraceId::PostMessage, msg);

        // Message distribution
        // --------------------
        // Check message categories
        COURIER_LOG_DEBUG("Distribute %s", msg->GetName());

#if defined(COURIER_MESSAGING_MONITOR_ENABLED)
        // Signal the post event to the monitor
    #if ! defined(COURIER_IPC_ENABLED)
        serializedMessageSize = 0;
    #endif
        MessagingMonitor::EventMessageRouterPost(msg,serializedMessageSize);
#endif
        // Abort posting message, if message router is not initialized
        if ((0 == mComponentReceiverMap) || (0 == mReceiverList)) {
            // Do not panic, as this error will happen in with use in Scene Composer
            COURIER_LOG_FATAL("Message router is not initialized!");
            return false;
        }

        // Sequential distribution
        if (msg->HasSequentialDistribution()) {
            lSuccess = DoSequentialDistribution(msgRef);
        }
        // Parallel distribution
        else {
            lSuccess = DoParallelDistribution(msgRef);
        }

        return lSuccess;
    }

    // ------------------------------------------------------------------------
    void MessageRouter::RegisterReceiver(MessageReceiver * msgReceiver)
    {
        FEATSTD_DEBUG_ASSERT(0 != msgReceiver);

        FEATSTD_PANIC_IF((0 == mComponentReceiverMap) || (0 == mReceiverList), "Message router is not initialized!");

        if (0 != msgReceiver) {
            ReceiverListLock lLock;

            FEATSTD_PANIC_IF(mReceiverListCount >= cCOURIER_RECEIVER_MAX_COUNT, "Number of receivers (%u) exceeds maximum allowed value %d",
                mReceiverListCount, COURIER_COMPONENT_MAX_COUNT);

                mReceiverList[mReceiverListCount] = msgReceiver;
                ++mReceiverListCount;
        } else {
            COURIER_LOG_ERROR("Receiver registration failed (receiver 0x%p)", msgReceiver);
        }
    }

    // ------------------------------------------------------------------------
    void MessageRouter::DeregisterReceiver(MessageReceiver * msgReceiver)
    {
        FEATSTD_DEBUG_ASSERT(0 != msgReceiver);

        FEATSTD_PANIC_IF((0 == mComponentReceiverMap) || (0 == mReceiverList), "Message router is not initialized!");

        if (0 != msgReceiver) {
            ReceiverListLock lLock;

            UInt32 lListIndex = 0;
            while ((lListIndex < mReceiverListCount) && (mReceiverList[lListIndex] != msgReceiver)) {
                ++lListIndex;
            }

            // If receiver found in list, shift list entries for one place to the left
            if (lListIndex < mReceiverListCount) {
                --mReceiverListCount;
                for (; lListIndex < mReceiverListCount; ++lListIndex) {
                    mReceiverList[lListIndex] = mReceiverList[lListIndex + 1];
                }
                mReceiverList[lListIndex] = 0;
            } else {
                COURIER_LOG_DEBUG("Receiver 0x%p is already de-registered", msgReceiver);
            }
        } else {
            COURIER_LOG_ERROR("Receiver 0x%p de-registration failed", msgReceiver);
        }
    }

    // ------------------------------------------------------------------------
    void MessageRouter::RegisterComponentAtReceiver(MessageReceiver * msgReceiver, ComponentId componentId)
    {
        FEATSTD_DEBUG_ASSERT(0 != msgReceiver);
        FEATSTD_DEBUG_ASSERT(ComponentId(ComponentType::Invalid) != componentId);

        FEATSTD_PANIC_IF((0 == mComponentReceiverMap) || (0 == mReceiverList), "Message router is not initialized!");
        FEATSTD_PANIC_IF(componentId >= COURIER_COMPONENT_MAX_COUNT, "Component ID (%u) exceeds maximum allowed value %d", componentId, COURIER_COMPONENT_MAX_COUNT);

        if ((0 != msgReceiver)
            && (ComponentId(ComponentType::Invalid) != componentId)
            && (componentId < COURIER_COMPONENT_MAX_COUNT)) {
            ComponentReceiverMapLock lLock;
            if (0 == mComponentReceiverMap[componentId]) {
                mComponentReceiverMap[componentId] = msgReceiver;
            } else {
                COURIER_LOG_ERROR("Map registration failed: Component %u already registered to 0x%p", componentId, mComponentReceiverMap[componentId]);
            }
        } else {
            COURIER_LOG_ERROR("Map registration failed (component %u, receiver 0x%p)", componentId, msgReceiver);
        }
    }

    // ------------------------------------------------------------------------
    void MessageRouter::DeregisterComponentFromReceiver(MessageReceiver * msgReceiver, ComponentId componentId)
    {
        FEATSTD_DEBUG_ASSERT(0 != msgReceiver);
        FEATSTD_DEBUG_ASSERT(ComponentId(ComponentType::Invalid) != componentId);

        FEATSTD_PANIC_IF((0 == mComponentReceiverMap) || (0 == mReceiverList), "Message router is not initialized!");
        FEATSTD_PANIC_IF(componentId >= COURIER_COMPONENT_MAX_COUNT, "Component ID (%u) exceeds maximum allowed value %d", componentId, COURIER_COMPONENT_MAX_COUNT);

        if ((0 != msgReceiver)
            && (ComponentId(ComponentType::Invalid) != componentId)
            && (componentId < COURIER_COMPONENT_MAX_COUNT)) {
            ComponentReceiverMapLock lLock;
            if (mComponentReceiverMap[componentId] == msgReceiver) {
                mComponentReceiverMap[componentId] = 0;
            } else {
                COURIER_LOG_DEBUG("Map de-registration failed: Map entry for component %u doesn't exist", componentId);
            }
        } else {
            COURIER_LOG_ERROR("Map de-registration failed (component %u, receiver 0x%p)", componentId, msgReceiver);
        }
    }

    // ------------------------------------------------------------------------
    bool MessageRouter::DoSequentialDistribution(const MessageReferrer & msgRef)
    {
        Message * lMsg = msgRef.GetMessage();
        bool lSuccess = (0 != lMsg);
        FEATSTD_DEBUG_ASSERT(lSuccess);
        if (lSuccess){
            // Perform sequential broadcast distribution
            if (lMsg->IsBroadcast()) {
                // Lock component receiver list access
                ReceiverListLock lLock;
                // Deliver message to all messaging receivers
                UInt32 lListIndex = lMsg->CurrentRecipient();
                if (lListIndex < mReceiverListCount) {
                    lSuccess = mReceiverList[lListIndex]->Receive(msgRef);
                }
                else {
                    COURIER_LOG_DEBUG("Sequential broadcast delivery of %s has finished", lMsg->GetName());
                }
            }
            // Perform sequential subscription distribution
            else {
                // Send message to the current subscriber in the delivery sequence
                MessageSubscriberList subscriberList = lMsg->GetSubscriberList();
                ComponentId lCurrentSubscriber = subscriberList.Current();

                if ((ComponentId(ComponentType::Invalid) != lCurrentSubscriber) &&
                    (0 != mComponentReceiverMap)) {
                    // Lock component receiver map access
                    ComponentReceiverMapLock lLock;

#if defined(COURIER_IPC_ENABLED)
                    IpcComponentId ipcComponentId(lCurrentSubscriber);

                    // eliminate process id for further local processing
                    lCurrentSubscriber = ipcComponentId.GetComponentId();

                    // if the process id differs the local id, route message to IPC (PID 0 is always treated as local)
                    if ((ipcComponentId.GetProcessId() != 0) &&
                        (ipcComponentId.GetProcessId() != IpcComponent::GetLocalPid()) &&
                        (0 != mComponentReceiverMap[ComponentType::Ipc])) {
                        lSuccess = mComponentReceiverMap[ComponentType::Ipc]->Receive(msgRef);
                    }
                    else {
#endif
                        // Check if there is a valid receiver mapped to the component
                        while ((!subscriberList.End()) && (0 == mComponentReceiverMap[lCurrentSubscriber])) {
                            lCurrentSubscriber = subscriberList.Next();
                        }
                        if ((lCurrentSubscriber < COURIER_COMPONENT_MAX_COUNT) && (0 != mComponentReceiverMap[lCurrentSubscriber])) {
                            lSuccess = mComponentReceiverMap[lCurrentSubscriber]->Receive(msgRef);
                        }
                        else {
                            // MISRA compliance;
                        }
#if defined(COURIER_IPC_ENABLED)
                    }
#endif
                }
                else {
                    COURIER_LOG_DEBUG("Sequential subscription delivery of %s has finished", lMsg->GetName());
                }
            }
        }
        return lSuccess;
    }

    // ------------------------------------------------------------------------
    bool MessageRouter::DoParallelDistribution(const MessageReferrer & msgRef)
    {
        Message * lMsg = msgRef.GetMessage();
        bool lSuccess = (0 != lMsg);
        FEATSTD_DEBUG_ASSERT(lSuccess);

        // Lock component receiver list access
        ReceiverListLock lLock;
        // Deliver message to all messaging receivers
        for (UInt32 lListIndex = 0; lListIndex < mReceiverListCount; ++lListIndex) {
            FEATSTD_DEBUG_ASSERT(0 != mReceiverList[lListIndex]);
            // Note: Do not use short circuiting with lSuccess, as every receiver shall be triggered
            lSuccess = mReceiverList[lListIndex]->Receive(msgRef) && lSuccess;
        }

        return lSuccess;
    }

    // ------------------------------------------------------------------------
    MessageHookSignature MessageRouter::GetMessageHook()
    {
        return gMessageHook;
    }

    // ------------------------------------------------------------------------
    MessageHookSignature MessageRouter::SetMessageHook(MessageHookSignature aMessageHook)
    {
        MessageHookSignature current = gMessageHook;
        gMessageHook = aMessageHook;
        return current;
    }
}}
